/**
 * @file programmNeu.cpp
 * @brief Implementierung mit Nebenlufigkeit im C++11-Stil.
 * @ingroup programmNeu
 */
#include <iostream>
#include <string>
#include <unordered_map>
#include <stdexcept>
#define _GLIBCXX_USE_NANOSLEEP
#include <chrono>
#include <sstream>
#include <thread>
#include <mutex>
#include <queue>
#include <condition_variable>
#include <stdio.h>
#include <sys/time.h> 
#include <sys/types.h> 
#include <unistd.h> 

using namespace std;

namespace programmNeu
{
class Kommando;
class Server;

/**
 * @brief Reprsentiert einen Benutzerauftrag fr die Serverwarteschlange.
 * Ein Auftrag setzt sich zusammen aus dem Kommando und dem Argument fr dessen
 * Abarbeitung.
 * @ingroup programmNeu
 */
class Auftrag
{
public:

	/**
	 * @brief Konstruktor.
	 * @param kommando Kommando Referenz auf das auszufhrende Kommando.
	 * @param auftrag RValue-Referenz auf den Parameter fr die Ausfhrung.
	 */
	Auftrag(Kommando *kommando,string &&parameter)
: kommando(kommando),
  parameter(std::move(parameter))
{}

	Kommando *kommando; /**< Referenz auf das auszufhrende Kommando. */
	string parameter; /**< Argument fr die Ausfhrung. */
};

/**
 * @brief Implementierung des Systemprfungs-Prozesses.
 * @ingroup programmNeu
 */
class SysPruefer
{
public:

	/**
	 * @brief Konstruktor.
	 * @param server Server, welcher geprft werden soll.
	 */
	SysPruefer(Server *server)
: server(server),
  deltaPrfg(5),
  sollLaufen(true)
{}

	/**
	 * @brief Arbeitsmethode des Systemprfungsprozesses.
	 */
	void ausfuehren();

	/**
	 * @brief Startet den Systemprfungsproze als nebenlufigen Proze.
	 */
	void start()
	{
		if(prfProzess.joinable()==false)
		{
			prfProzess=thread(&SysPruefer::ausfuehren,this);
		}
	}

	/**
	 * Stoppt die Ausfhrung des nebenlufigen Systemprfungsprozesses.
	 */
	void stop()
	{
		sollLaufen=false;
		prfProzessCV.notify_all();
		if(prfProzess.joinable()) prfProzess.join();

	}

protected:
	Server *server; /**< Referenz auf den zu prfenden Server. */
	mutex prfProzessMutex; /**< Zugriffssicherung fr diesen Proze. */
	condition_variable prfProzessCV; /**< Zustandsvariable dieses Prozesses. */
	const chrono::seconds deltaPrfg; /**< Zeitabstand zwischen zwei Prfungen. */
	bool sollLaufen; /**< Ausfhrungserlaubnis fr diesen Proze. */
	thread prfProzess; /**< Instanz des ausfhrenden nebenlufigen Prozesses. */
};

/**
 * @brief Diese Klasse reprsentiert einen typischen Serverdienst.
 * @ingroup programmNeu
 */
class Server
{
	friend class SysPruefer;

public:

	/**
	 * @brief Konstruktor.
	 */
	Server();

	/**
	 * @brief Destruktor.
	 */
	~Server();

	/**
	 * @brief Ausfhrung der Serverttigkeit.
	 */
	void ausfuehren();

	/**
	 * @brief Ausgabe einer Meldung an den Benutzer.
	 * @param meldung An den Benutzer weiterzugebende Textmeldung.
	 */
	void ausgeben(const string &meldung)
	{
		lock_guard<mutex> waechter(ausgabeMutex);
		cerr << meldung << endl;
	}

	/**
	 * @brief Fordert den Serverdienst zur Einstellung seiner Ttigkeit auf.
	 */
	void stop()
	{
		sollLaufen=false;
	}

	bool sollWeiterlaufen() const
	{
		return sollLaufen;
	}

protected: // geschuetzte Methoden

	/**
	 * @brief Arbeitsmethode des Log-Nebenprozesses.
	 */
	void loggen()
	{
		ausgeben("Log-Eintrag "+aktuelleZeit());
	}

	/**
	 * @brief Ausfhrungsschleife des Log-Nebenprozesses.
	 */
	void logProzessAusf()
	{
		std::unique_lock<std::mutex> waechter(logProzessMutex);
		while(sollLaufen)
		{
			logProzessCV.wait_for(waechter,deltaLog);
			if(sollLaufen==false) break;
			loggen();
		};
		ausgeben("Ende des Log-Prozesses!");
	}

	/**
	 * @brief Arbeitsmethode des Systemprfungsprozesses.
	 */
	void sysPruefen()
	{
		ausgeben("System-Test "+aktuelleZeit());
	}

	/**
	 * @brief Gibt die aktuelle Zeit in Textform zurck.
	 * @return Zeichenkette mit der aktuellen Zeit.
	 */
	string aktuelleZeit()
	{
		time_t zeit=time(NULL);
		return string(ctime(&zeit));
	}

	/**
	 * @brief Registriert ein neues Kommando.
	 * @param kommando Kommando, welches beim Server registriert werden soll.
	 */
	void kommandoHinzufuegen(Kommando *kommando);

	/**
	 * @brief Sucht ein Komando bei dessen Namen.
	 * @param name Name des Kommandos.
	 * @return Referenz auf die gefundene Kommando-Instanz.
	 * @exception runtime_error Wird geworfen, wenn das Kommando nicht gefunden wurde.
	 */
	Kommando *sucheKommando(const string &name);

	/**
	 * @brief Zerlegt die Benutzereingabe im Format 'kommandoname=argument' in die Bestandteile
	 * Kommandoname und Argument.
	 * @param [in] aufruf Benutzereingabe
	 * @param [out] kommando Name des Kommandos.
	 * @param [out] argument Argument des Kommandos.
	 * @exception runtime_error Geworfen, wenn die Benutzereingabe syntaktisch falsch ist.
	 */
	bool aufrufZerlegen(const string &aufruf,string &kommando,string &argument);

	/**
	 * @brief Ausfhrungsschleife des Server-Arbeitsprozesses.
	 */
	void arbeitsProzessAusf();

	/**
	 * @brief Einreihen eines Auftrags in die Ausfhrungs-Warteschlange.
	 * @param kommando Referenz auf die auszufhrende Kommando-Instanz.
	 * @param argument Argument fr die Kommando-Ausfhrung.
	 */
	void auftragAnnehmen(Kommando *kommando,string &&argument)
	{
		unique_lock<mutex>(warteSchlangenMutex);
		warteSchlange.push(std::move(Auftrag(kommando,std::move(argument))));
		warteSchlangenCV.notify_all();
	}

	/**
	 * @brief Behandlung von Ausnahmen whrend der Kommandoausfhrung.
	 * @param ptr Referenz auf eine whrend der Kommandoausfhrung aufgetretende Ausnahme.
	 */
	void ausnahmeBehandlungKommandoAusfuehrung(exception_ptr ptr);

protected: // geschuetzte Attribute
	bool sollLaufen; /**< Ausfhrungserlaubnis fr den Server-Prozess. */
	fd_set warteMenge; /**< Menge von Dateideskriptoren, bei denen auf eine Eingabe gewartet werden soll. */
	const chrono::seconds deltaLog; /**< Zeitabstand zwischen zwei Log-Eintrgen in Sekunden. */
	unordered_map<string,Kommando *> kommandos; /**< Assoziation zwischen Kommandonamen und den Kommandos. */
	mutex ausgabeMutex; /**< Mutex zur Sicherung des Ausgabekanals. */
	thread logProzess; /**< Instanz des nebenlufigen Prozesses fr die Log-Ausgaben. */
	mutex logProzessMutex; /**< Mutex zur Absicherung des Log-Prozesses. */
	condition_variable logProzessCV; /**< Zustandsvariable zum Warten/Aufwecken des Log-Prozesses. */
	SysPruefer pruefer; /**< Systemprfungs-Proze. */
	thread arbeitsProzess; /**< Instanz des nebenlufigen Prozesses zur Abarbeitung der Server-Kommandos. */
	mutex arbeitsProzessMutex; /**< Mutex zur Absicherung der nebenlufigen Abarbeitung der Server-Kommandos. */
	queue<Auftrag> warteSchlange; /**< Auftragswarteschlange fr den Server-Proze. */
	mutex warteSchlangenMutex; /**< Mutex zur Absicherung der Auftrags-Warteschlange. */
	condition_variable warteSchlangenCV; /**< Zustandsvariable zur Signalisierung von Statusnderungen bei der Auftragswarteschlange. */
};

/**
 * @brief Basisklasse fr alle Kommandos.
 * @ingroup programmNeu
 */
class Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param name Name des Kommandos.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 * @param direktAusfuehren Wenn wahr, wird das Kommando nicht in die Warteschlange eingereiht sondern
	 * gleich ausgefhrt.
	 */
	Kommando(const string &name,Server *server,bool direktAusfuehren=false)
: name(name),server(server),direktAusfuehren(direktAusfuehren) {}

	/**
	 * @brief Destruktor.
	 */
	virtual ~Kommando() {}

	/**
	 * @brief Ausfhrungsmethode des Kommandos.
	 * @param parameter Parameter fr die Kommandoausfhrung.
	 * @return Ergebnis der Kommandoausfhrung in Textform.
	 */
	virtual string ausfuehren(const string &parameter) = 0;

	/**
	 * @brief Prft, ob das Kommando weiterlaufen darf.
	 * @exception runtime_error Wird geworfen, wenn die Ausfhrungserlaubnis entzogen worden ist.
	 */
	void pruefeAufAbbruch()
	{
		if(server->sollWeiterlaufen()==false)
		{
			throw runtime_error(string("Kommando ")+name+" wurde abgebrochen!");
		}
	}

	const string name; /**< Name des Kommandos. */
	Server *server; /**< Referenz auf den Server, fr den dieses Kommando arbeitet. */
	const bool direktAusfuehren; /**< Wenn wahr, soll das Kommando nicht in die Warteschlange aufgenommen sondern direkt ausgefhrt werden. */
};

/**
 * @brief Ableitung der Kommandobasisklasse fr ein die Fakultt berechnendes Kommando.
 * @ingroup programmNeu
 */
class KommandoFakultaet : public Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 */
	KommandoFakultaet(Server *server)
: Kommando("fakultaet",server) {}

	/**
	 * @implements Kommando::ausfuehren(const string &parameter)
	 */
	string ausfuehren(const string &parameter) override
			{
		istringstream parser(parameter);
		parser.exceptions(ios::badbit | ios::failbit);
		unsigned int value;
		unsigned long fakultaet=1;
		chrono::milliseconds verz(500);
		try{
			parser >> value;
		}catch(const ios::failure &e)
		{
			throw runtime_error("Erwarte Nummern-Wert!");
		}
		for(unsigned int k=1;k<=value;k++)
		{
			pruefeAufAbbruch();
			fakultaet*=k;
			this_thread::sleep_for(verz);
		}
		ostringstream result;
		result << fakultaet;
		return result.str();
			}
};

/**
 * @brief Ableitung der Kommandobasisklasse fr ein den Serverdienst beendendes Kommando.
 * @ingroup programmNeu
 */
class KommandoBeenden : public Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 */
	KommandoBeenden(Server *server)
: Kommando("beenden",server,true) {}

	/**
	 * @implements Kommando::ausfuehren(const string &parameter)
	 */
	string ausfuehren(const string &parameter) override
			{
		server->stop();
		return "Ok";
			}
};


Server::Server()
: sollLaufen(true),
  deltaLog(10),
  pruefer(this)
{
	FD_ZERO(&warteMenge);
	FD_SET(STDIN_FILENO,&warteMenge);
	kommandoHinzufuegen(new KommandoFakultaet(this));
	kommandoHinzufuegen(new KommandoBeenden(this));
	logProzess=thread(&Server::logProzessAusf,this);
	pruefer.start();
	arbeitsProzess=thread(&Server::arbeitsProzessAusf,this);
}

Server::~Server() 
{
	logProzessCV.notify_all();
	if(logProzess.joinable()) logProzess.join();
	pruefer.stop();
	warteSchlangenCV.notify_all();
	if(arbeitsProzess.joinable()) arbeitsProzess.join();
	for(auto eintrag : kommandos)
	{
		delete eintrag.second;
	}
}

void Server::ausfuehren()
{
	string kommandoName,parameter,cmdStr;
	try{
		while(sollLaufen)
		{
			fd_set warteMengeKopie=warteMenge;
			int resultat=select(STDIN_FILENO+1,
					&warteMengeKopie,NULL,NULL,
					NULL);
			if(resultat < 0)
			{
				throw runtime_error("Fehler bei SELECT()!");
			}
			if(FD_ISSET(STDIN_FILENO, &warteMengeKopie))
			{
				try{
					cin >> cmdStr;
					if(aufrufZerlegen(cmdStr,kommandoName,parameter)==false)
					{
						throw runtime_error("Syntax-Fehler!");
					}
					Kommando *kommando=sucheKommando(kommandoName);
					if(kommando->direktAusfuehren)
					{
						ausgeben(kommando->name+"="+kommando->ausfuehren(parameter));
					}
					else
					{
						auftragAnnehmen(kommando,std::move(parameter));
					}
				}catch(...)
				{
					ausnahmeBehandlungKommandoAusfuehrung(current_exception());
				}
			}
		}
	}catch(const exception &error)
	{
		cerr << L"Ausf\x00fchausfuehreng abgebrochen: "
				<< error.what() << endl;
	}
}


inline void Server::kommandoHinzufuegen(Kommando *kommando)
{
	kommandos[kommando->name]=kommando;
}

Kommando *Server::sucheKommando(const string &name)
{
	unordered_map<string,Kommando *>::iterator it=
			kommandos.find(name);
	if(it==kommandos.end())
	{
		throw runtime_error("Unbekanntes Kommando: \""+
				name+"\"!");
	}
	return it->second;
}

bool Server::aufrufZerlegen(const string &aufruf,string &kommando,string &argument)
{
	size_t pos=aufruf.find('=');
	if(pos != string::npos)
	{
		kommando=aufruf.substr(0,pos);
		argument=aufruf.substr(pos+1);
		return true;
	}
	else
	{
		return false;
	}
}

void SysPruefer::ausfuehren()
{
	unique_lock<std::mutex> waechter(prfProzessMutex);
	while(sollLaufen)
	{
		prfProzessCV.wait_for(waechter,deltaPrfg);
		if(sollLaufen==false) break;
		server->sysPruefen();
	}
	server->ausgeben("Pruefprozess beendet!");
}

void Server::arbeitsProzessAusf()
{
	unique_lock<std::mutex> waechter(arbeitsProzessMutex);
	while(sollLaufen)
	{
		warteSchlangenCV.wait(waechter);
		unique_lock<std::mutex> warteSchlangenSicherung(warteSchlangenMutex);
		while(sollLaufen && (warteSchlange.empty()==false))
		{
			const Auftrag &auftrag=warteSchlange.front();
			Kommando *kommando=auftrag.kommando;
			try{
				ausgeben(kommando->name+"="+kommando->ausfuehren(auftrag.parameter));
			}catch(...)
			{
				ausnahmeBehandlungKommandoAusfuehrung(current_exception());
			}
			warteSchlange.pop();
		}
		if(sollLaufen==false) break;
	}
	ausgeben("Arbeitsprozess beendet!");
}

void Server::ausnahmeBehandlungKommandoAusfuehrung(exception_ptr ptr)
{
	if(ptr != exception_ptr())
	{
		try{
			rethrow_exception(ptr);
		}catch(const exception &error)
		{
			ausgeben("Fehler: "+string(error.what()));
		}
	}
}

/**
 * @defgroup programmNeu
 * @brief Implementierung mit der Nebenlufigkeitsuntersttzung von C++11.
 */
}

/**
 * @brief Haupteinsprungsfunktion des Programms.
 * @return 0 im Erfolgsfall.
 * @ingroup programmNeu
 */
int main()
{
	programmNeu::Server server;
	server.ausfuehren();
	cout << "Auf Wiedersehen!" << endl;
	return 0;
}
