/**
 * @file programmBisher.cpp
 * @brief Implementierung im klassischen C++03-Stil ohne Nebenlufigkeit.
 * @ingroup ProgrammBisher
 */
#include <iostream>
#include <string>
#include <unordered_map>
#include <stdexcept>
#define _GLIBCXX_USE_NANOSLEEP
#include <chrono>
#include <sstream>
#include <thread>
#include <stdio.h>
#include <sys/time.h> 
#include <sys/types.h> 
#include <unistd.h> 

using namespace std;

namespace programmBisher{
class Kommando;

/**
 * @brief Diese Klasse reprsentiert einen typischen Serverdienst.
 * @ingroup ProgrammBisher
 */
class Server
{
public:

	/**
	 * @brief Konstruktor.
	 */
	Server();

	/**
	 * @brief Destruktor.
	 */
	~Server();

	/**
	 * @brief Ausfhrung der Serverttigkeit.
	 */
	void ausfuehren();

	/**
	 * @brief Ausgabe einer Meldung an den Benutzer.
	 * @param meldung An den Benutzer weiterzugebende Textmeldung.
	 */
	void ausgeben(const string &meldung)
	{
		cerr << meldung << endl;
	}

	/**
	 * @brief Fordert den Serverdienst zur Einstellung seiner Ttigkeit auf.
	 */
	void stop() { sollLaufen=false; }

protected: // geschuetzte Methoden

	/**
	 * @brief Arbeitsmethode des Log-Nebenprozesses.
	 */
	void loggen()
	{
		ausgeben("Log-Eintrag "+aktuelleZeit());
	}

	/**
	 * @brief Arbeitsmethode des Systemprfungs-Nebenprozesses.
	 */
	void sysPruefen()
	{
		ausgeben("System-Test "+aktuelleZeit());
	}

	/**
	 * @brief Gibt die aktuelle Zeit in Textform zurck.
	 * @return Zeichenkette mit der aktuellen Zeit.
	 */
	string aktuelleZeit()
	{
		time_t zeit=time(NULL);
		return string(ctime(&zeit));
	}

	/**
	 * @brief Registriert ein neues Kommando.
	 * @param kommando Kommando, welches beim Server registriert werden soll.
	 */
	void kommandoHinzufuegen(Kommando *kommando);

	/**
	 * @brief Sucht ein Komando bei dessen Namen.
	 * @param name Name des Kommandos.
	 * @return Referenz auf die gefundene Kommando-Instanz.
	 * @exception runtime_error Wird geworfen, wenn das Kommando nicht gefunden wurde.
	 */
	Kommando *sucheKommando(const string &name);

	/**
	 * @brief Zerlegt die Benutzereingabe im Format 'kommandoname=argument' in die Bestandteile
	 * Kommandoname und Argument.
	 * @param [in] aufruf Benutzereingabe
	 * @param [out] kommando Name des Kommandos.
	 * @param [out] argument Argument des Kommandos.
	 * @exception runtime_error Geworfen, wenn die Benutzereingabe syntaktisch falsch ist.
	 */
	bool aufrufZerlegen(const string &aufruf,string &kommando,string &argument);

protected: // geschuetzte Attribute
	bool sollLaufen; /**< Ausfhrungserlaubnis fr den Server-Prozess. */
	fd_set warteMenge; /**< Menge von Dateideskriptoren, bei denen auf eine Eingabe gewartet werden soll. */
	struct timeval warteZeit; /**< Zeitbegrenzung fr das Warten auf Eingabe. */
	time_t naechsterLogZeitpunkt; /**< Zeitpunkt der nchsten Logproze-Ausfhrung. */
	time_t naechsterPruefZeitpunkt; /**< Zeitpunkt der nchsten Systemprfproze-Ausfhrung. */
	static constexpr unsigned int deltaLog=10; /**< Zeitabstand zwischen zwei Logproze-Ausfhrungen in Sekunden. */
	static constexpr unsigned int deltaPrfg=5; /**< Zeitabstand zwischen zwei Systemprfproze-Ausfhrungen in Sekunden. */
	unordered_map<string,Kommando *> kommandos; /**< Assoziation zwischen Kommandonamen und den Kommandos. */
};

/**
 * @brief Basisklasse fr alle Kommandos.
 * @ingroup ProgrammBisher
 */
class Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param name Name des Kommandos.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 */
	Kommando(const string &name,Server *server)
: name(name),server(server) {}

	/**
	 * @brief Destruktor.
	 */
	virtual ~Kommando() {}

	/**
	 * @brief Ausfhrungsmethode des Kommandos.
	 * @param parameter Parameter fr die Kommandoausfhrung.
	 * @return Ergebnis der Kommandoausfhrung in Textform.
	 */
	virtual string ausfuehren(const string &parameter) = 0;

	string name; /**< Name des Kommandos. */
	Server *server; /**< Referenz auf den Server, fr den dieses Kommando arbeitet. */
};

/**
 * @brief Ableitung der Kommandobasisklasse fr ein die Fakultt berechnendes Kommando.
 * @ingroup ProgrammBisher
 */
class KommandoFakultaet : public Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 */
	KommandoFakultaet(Server *server)
: Kommando("fakultaet",server) {}

	/**
	 * @implements Kommando::ausfuehren(const string &parameter)
	 */
	string ausfuehren(const string &parameter) override
			{
		istringstream parser(parameter);
		parser.exceptions(ios::badbit | ios::failbit);
		unsigned int value;
		unsigned long fakultaet=1;
		chrono::milliseconds verz(500);
		try{
			parser >> value;
		}catch(const ios::failure &e)
		{
			throw runtime_error("Erwarte Nummern-Wert!");
		}
		for(unsigned int k=1;k<=value;k++)
		{
			fakultaet*=k;
			this_thread::sleep_for(verz);
		}
		ostringstream result;
		result << fakultaet;
		return result.str();
			}
};


/**
 * @brief Ableitung der Kommandobasisklasse fr ein den Serverdienst beendendes Kommando.
 * @ingroup ProgrammBisher
 */
class KommandoBeenden : public Kommando
{
public:

	/**
	 * @brief Konstruktor.
	 * @param server Referenz auf die Server-Instanz, fr die das Kommando arbeiten soll.
	 */
	KommandoBeenden(Server *server)
: Kommando("beenden",server) {}

	/**
	 * @implements Kommando::ausfuehren(const string &parameter)
	 */
	string ausfuehren(const string &parameter) override
			{
		server->stop();
		return "Ok";
			}
};


Server::Server()
: sollLaufen(true)
{
	FD_ZERO(&warteMenge);
	FD_SET(STDIN_FILENO,&warteMenge);
	warteZeit.tv_sec=0;
	warteZeit.tv_usec=1000000;
	time_t jetzt=time(NULL);
	naechsterLogZeitpunkt=jetzt+deltaLog;
	naechsterPruefZeitpunkt=jetzt+deltaPrfg;
	kommandoHinzufuegen(new KommandoFakultaet(this));
	kommandoHinzufuegen(new KommandoBeenden(this));
}

Server::~Server() 
{
	for(auto eintrag : kommandos)
	{
		delete eintrag.second;
	}
}

void Server::ausfuehren()
{
	string kommandoName,parameter,cmdStr;
	try{
		while(sollLaufen)
		{
			fd_set warteMengeKopie=warteMenge;
			int resultat=select(STDIN_FILENO+1,
					&warteMengeKopie,NULL,NULL,
					&warteZeit);
			if(resultat < 0)
			{
				throw runtime_error("Fehler bei SELECT()!");
			}
			if(FD_ISSET(STDIN_FILENO, &warteMengeKopie))
			{
				try{
					cin >> cmdStr;
					if(aufrufZerlegen(cmdStr,kommandoName,parameter)==false)
					{
						throw runtime_error("Syntax-Fehler!");
					}
					Kommando *cmd=sucheKommando(kommandoName);
					ausgeben(cmd->name+"="+cmd->ausfuehren(parameter));
				}catch(const exception &error)
				{
					ausgeben("Fehler: "+
							string(error.what()));
				}
			}
			time_t jetzt=time(NULL);
			if(jetzt >= naechsterLogZeitpunkt)
			{
				naechsterLogZeitpunkt=jetzt+deltaLog;
				loggen();
			}
			if(jetzt >= naechsterPruefZeitpunkt)
			{
				naechsterPruefZeitpunkt=jetzt+deltaPrfg;
				sysPruefen();
			}
		}
	}catch(const exception &error)
	{
		cerr << L"Ausf\x00fchrung abgebrochen: "
				<< error.what() << endl;
	}
}

inline void Server::kommandoHinzufuegen(Kommando *kommando)
{
	kommandos[kommando->name]=kommando;
}

Kommando *Server::sucheKommando(const string &name)
{
	unordered_map<string,Kommando *>::iterator it=
			kommandos.find(name);
	if(it==kommandos.end())
	{
		throw runtime_error("Unbekanntes Kommando: \""+
				name+"\"!");
	}
	return it->second;
}

bool Server::aufrufZerlegen(const string &aufruf,string &kommando,string &argument)
{
	size_t pos=aufruf.find('=');
	if(pos != string::npos)
	{
		kommando=aufruf.substr(0,pos);
		argument=aufruf.substr(pos+1);
		return true;
	}
	else
	{
		return false;
	}
}

/**
 * @defgroup ProgrammBisher
 * @brief Klassische Implementierung in C++03 ohne Nebenlufigkeit.
 */
}

/**
 * @brief Haupteinsprungsfunktion des Programms.
 * @return 0 im Erfolgsfall.
 * @ingroup ProgrammBisher
 */
int main()
{
	programmBisher::Server server;
	server.ausfuehren();
	cout << "Auf Wiedersehen!" << endl;
	return 0;
}
