/**
 * @file LogoRotate.cpp
 */

/*
 * Copyright (C) Gemfony scientific UG (haftungsbeschraenkt)
 *
 * Contact: contact [at] gemfony (dot) com
 *
 * This file is part of the LogoRotate program.
 *
 * LogoRotate is free software: you can redistribute and/or modify it under
 * the terms of version 3 of the GNU Affero General Public License
 * as published by the Free Software Foundation.
 *
 * LogoRotate is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with the Geneva library. If not, see <http://www.gnu.org/licenses/>.
 *
 * For further information on Gemfony scientific, visit http://www.gemfony.com
 */

#include "LogoRotate.hpp"

LogoRotate::LogoRotate(const std::string& filename)
{
  try {
    //setup devices and context
    std::vector<cl::Platform> platforms;
    cl::Platform::get(&platforms);
    if(platforms.size() == 0) {
      std::cerr << "Error: No platforms found" << std::endl;
      exit(1);
    } else {
      std::cout << "Found " << platforms.size() << " platforms" << std::endl;
    }
    
    // We use the first available platform for now, with default values
    cl_context_properties properties[] = { 
      CL_CONTEXT_PLATFORM
      , (cl_context_properties)(platforms[0])()
      , 0
    };
    
    // Create the context and retrieve the device info
    context = cl::Context(CL_DEVICE_TYPE_GPU, properties);
    devices = context.getInfo<CL_CONTEXT_DEVICES>();

    // Create the command queue we will use to execute OpenCL commands
    // For now we just use the first available device.
    queue = cl::CommandQueue(context, devices[0], CL_QUEUE_PROFILING_ENABLE);    

    // Create the program from source
    std::ifstream sourceFileStream(filename.c_str());
    std::string sourceFile (
			    std::istreambuf_iterator<char>(sourceFileStream)
			    , (std::istreambuf_iterator<char>())
			    );
    cl::Program::Sources src(1, std::make_pair(sourceFile.c_str(), sourceFile.length()+1));
    program = cl::Program(context, src);

    // Build the actual program
    std::string buildOptions = "-DWIDTH=" + boost::lexical_cast<std::string>(dimX) + " -DHEIGHT=" + boost::lexical_cast<std::string>(dimY);
      program.build(devices, buildOptions.c_str());
    std::cout << "Build Status: " << ((program.getBuildInfo<CL_PROGRAM_BUILD_STATUS>(devices[0]))==0?"SUCCESS":"FAILURE") << std::endl;
    std::cout << "Build Options:\t" << program.getBuildInfo<CL_PROGRAM_BUILD_OPTIONS>(devices[0]) << std::endl;

    // Allocate memory for the source and target images
    source = new cl_uchar4[dimX*dimY];
    target = new cl_uchar4[dimX*dimY];
    black_target = new cl_uchar4[dimX*dimY];

    // Fill the source image with data and set the background color of the target image to white
    for(int i=0; i<dimX*dimY; i++) {
      source[i].s0 = (cl_uchar)ixlogo[i*3 + 0]; // red
      source[i].s1 = (cl_uchar)ixlogo[i*3 + 1]; // green
      source[i].s2 = (cl_uchar)ixlogo[i*3 + 2]; // blue
      source[i].s3 = (cl_uchar)0; // unused / alpha channel

      black_target[i].s0 = (cl_uchar)255;
      black_target[i].s1 = (cl_uchar)255;
      black_target[i].s2 = (cl_uchar)255;
      black_target[i].s3 = (cl_uchar)0;
    }

    // Create the buffers
    source_buffer = cl::Buffer(
			       context 
			       , CL_MEM_READ_ONLY|CL_MEM_COPY_HOST_PTR
			       , dimX*dimY*sizeof(cl_uchar4)
			       , (void *)source
			       );

    target_buffer = cl::Buffer(
			       context 
			       , CL_MEM_WRITE_ONLY
			       , dimX*dimY*sizeof(cl_uchar4)
			       );
  } catch(cl::Error err) {
    std::cerr << "Error! " << err.what() << std::endl
	      << err.err() << std::endl
              << "Build Log:\t " << std::endl
	      << program.getBuildInfo<CL_PROGRAM_BUILD_LOG>(devices[0]) << std::endl;
    exit(1);
  }
}

LogoRotate::~LogoRotate()
{
  delete [] source;
  delete [] target;
  delete [] black_target;
}

std::string LogoRotate::rotate(const float& angle)
{
  try{
    // Will hold the final image
    std::ostringstream result;

    // Create a suitable kernel
    rotate_kernel = cl::Kernel(program, "logo_rotate");
    
    // Re-initialize the black target image on the GPU. The call will
    // only return after all data has been copied, due to the second argument.
    queue.enqueueWriteBuffer(target_buffer, CL_TRUE, 0, sizeof(cl_uchar4)*dimX*dimY, (void *)black_target);

    //set the arguments of our kernel
    rotate_kernel.setArg(0, source_buffer);
    rotate_kernel.setArg(1, target_buffer);
    rotate_kernel.setArg(2, (cl_float)angle);
      
    // Execute the kernel and wait for its termination
    queue.enqueueNDRangeKernel(rotate_kernel, cl::NullRange, cl::NDRange(dimX, dimY), cl::NullRange, NULL, &event); 
    event.wait();
    
    // Retrieve the results buffer
    queue.enqueueReadBuffer(target_buffer, CL_TRUE, 0, sizeof(cl_uchar4)*dimX*dimY, target);

    // Assemble the image
    result 
      << "P3" << std::endl
      << dimX << " " << dimY << std::endl
      << 255 << std::endl;
    for(int i=0; i<dimX*dimY; i++) {
      result << (int)target[i].s0 << " " << (int)target[i].s1 << " " << (int)target[i].s2 << std::endl;
    }

    return result.str();
  } catch (cl::Error err) {
    std::cout << "Error when running the kernel: " << std::endl
	      << err.what() << std::endl 
	      << err.err()  << std::endl;
    exit(1);
  }

  // Make the compiler happy
  return std::string();
}
