/*
Copyright (c) 2011, David Linner <david.linner@tu-berlin.de>

Permission to use, copy, modify, and/or distribute this software for any
purpose with or without fee is hereby granted, provided that the above
copyright notice and this permission notice appear in all copies.

THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
*/


function VerySimpleEditor(container){
	this.__container = container;	
	this._init();
}

VerySimpleEditor.prototype._init = function(){	
	
	while(this.__container.hasChildNodes()){
		this.__container.removeChild(this.__container.firstChild);       
    } 	
	var p = this.__container.ownerDocument.createElement("p");
	p.appendChild(this.__container.ownerDocument.createElement("br"));
	this.__container.appendChild(p);
	
	// work around chrome delete quirk
	if (navigator.userAgent.indexOf("Chrome") >= 0){
		this.__container.addEventListener("keydown", (function(editor){ return function(e){		
			if ((e.keyCode == 8 || e.keyCode == 46) && editable.childNodes.length == 1 && editable.textContent.length == 0) {
				
				e.preventDefault();	
			}
		};})(this),true);		
	}
		
	this.__container.addEventListener("keypress", (function(editor){ return function(e){		
		console.log("raised keypress");
		if (editor.onInsert){
			var code = null;
			if (e.charCode){
				code = e.charCode;
			} else if (e.keyCode > 64){
				code = e.keyCode;
			} 
			if (code && code != 13 && code != 32){
				var offset = editor.getCaret();
				editor.onInsert(offset, String.fromCharCode(code)); //fromCharCode may cause unexpected results for keycodes				
				editor.setCaret(offset); // workaround - collapse selection, override default behavoir replace 
			}			
		}
	};})(this),true);
	
	this.__container.addEventListener("keydown", (function(editor){ return function(e){			
		switch (e.keyCode){		
			case 13: //return
				if (editor.onInsert){
					editor.onInsert(editor.getCaret(), "\n");
				}				
				break;
			case 32: //space
				if (editor.onInsert){
					editor.onInsert(editor.getCaret(), " ");
				}				
				break;
			case 8: //backspace				
				if (editor.onRemove){
					var caret = editor.getCaret();	 
					if (caret -1 >= 0){
						editor.onRemove(editor.getCaret()-1);
					}
				}
				break;			
			case 46: //delete
				if (editor.onRemove){
					var caret = editor.getCaret();	 
					if (caret -1 >= 0){
						editor.onRemove(editor.getCaret());
					}
				}
				break;
		}
	};})(this), true);
	
};

VerySimpleEditor.prototype.insert = function (position, chr){	
	var caret = this.getCaret();
	this.setCaret(position);
	this.__container.focus();
	if (chr == "\n"){		
		if (navigator.userAgent.indexOf("Firefox") >= 0){
			// Firefox implements insertParagraph differently, so a special treatment is neeeded 
			if(this.isCaretAtEOL()){
				this.__container.ownerDocument.execCommand("insertHTML", null, "<p></p>");	
			}else {
				this.__container.ownerDocument.execCommand("insertHTML", null, "<br>");	
				this.__container.ownerDocument.execCommand("insertParagraph", false, null);
			}
		} else { 
			this.__container.ownerDocument.execCommand("insertParagraph", false, null);
		}
	} else if (chr === " ") {
		this.__container.ownerDocument.execCommand("insertHTML", null, "&nbsp;");
	} else {
		this.__container.ownerDocument.execCommand("insertHTML", null, chr);
	}
	this.setCaret(position <= caret ? caret + 1 : caret);
};

VerySimpleEditor.prototype.remove = function (position){
	var caret = this.getCaret();
	// in Opera 'delete' removes the character behind the caret
	this.setCaret(position + (navigator.userAgent.indexOf("Opera") < 0 ? 1 : 0));
	this.__container.ownerDocument.execCommand("delete", null, null);
	this.setCaret(position < caret ? caret - 1 : caret);
};

VerySimpleEditor.prototype.getCaret = function (){
			
	var offset = 0; 
	
	var selection = window.getSelection();
	if (selection.rangeCount > 0){
		var range = selection.getRangeAt(0);
		
		// test if range was selected left to right or vice versa
		var container;
		if (range.startContainer.compareDocumentPosition(range.endContainer) & Node.DOCUMENT_POSITION_PRECEDING ){		
			if(range.startContainer.nodeType === Node.ELEMENT_NODE){
				container = range.startContainer.childNodes[range.startOffset];
				offset = 0;
			} else {
				container = range.startContainer;
				offset = range.startOffset;
			}
		} else {
			if(range.endContainer.nodeType === Node.ELEMENT_NODE){
				container = range.endContainer.childNodes[range.endOffset];
				offset = 0;
			} else {
				container = range.endContainer;
				offset = range.endOffset;
			}			
		}	
		
		var iterator = this.__container.ownerDocument.createNodeIterator(
				this.__container,
				NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT,
				null,
				false);
		var current = iterator.nextNode();	
		
		// zeichen zhlen, absatz zhlen wir als ein zeichen
		while (current && container != current){
			
			switch (current.nodeName){
				case "#text" :
					offset += current.length;
				break;
				case "P" :				
					offset += 1;
			}
			
			current = iterator.nextNode();
		}	
		iterator.detach();
		offset -=1;
	}	
	
	return offset;
};

VerySimpleEditor.prototype.setCaret = function (offset){
	var selection = window.getSelection();
	var range = document.createRange();
	
	var iterator = this.__container.ownerDocument.createNodeIterator(
			this.__container,
			NodeFilter.SHOW_ELEMENT | NodeFilter.SHOW_TEXT,
			null,
			false);
	
	var current = iterator.nextNode();	
	
	while (current){
		
		if (current.nodeName == "#text" ){
			if ((offset + 1) - current.length <= 0){
				range.setStart(current, (offset + 1));
				range.collapse(true);
				break;
			} else {
				offset -= current.length ;
			}
		} else if	(current.nodeName == "P" ){				
			if ((offset + 1) - 1 <= 0){
				range.setStart(current,0);
				range.collapse(true);
				break;
			} else {
				offset--;
			}
		}
		
		current = iterator.nextNode();
	}	
	
	selection.removeAllRanges();
	selection.addRange(range);
	
	this.__container.focus();
		
};

VerySimpleEditor.prototype.hasSelection = function (){	
	var selection = window.getSelection();	
	
	if (selection.rangeCount > 0){
		var range = selection.getRangeAt(0);
		return (
				this.__container.compareDocumentPosition(range.commonAncestorContainer) & Node.DOCUMENT_POSITION_CONTAINED_BY) > 0 ? true : false; // 0x10
	}
	
	return false;
};

VerySimpleEditor.prototype.isCaretAtEOL = function (){
	
	var offset = 0; 
	
	var selection = window.getSelection();
	if (selection.rangeCount > 0){
		var range = selection.getRangeAt(0);
		
		// teste ob die range von links nach rechts oder rechts nach links erstellt wurde
		var container;
		if (range.startContainer.compareDocumentPosition(range.endContainer) & Node.DOCUMENT_POSITION_PRECEDING ){		
			if(range.startContainer.nodeType === Node.ELEMENT_NODE){
				container = range.startContainer.childNodes[range.startOffset];
				offset = 0;
			} else {
				container = range.startContainer;
				offset = range.startOffset;
			}
		} else {
			if(range.endContainer.nodeType === Node.ELEMENT_NODE){
				container = range.endContainer.childNodes[range.endOffset];
				offset = 0;
			} else {
				container = range.endContainer;
				offset = range.endOffset;
			}			
		}	
				
		switch (container.nodeName){
			case "#text" :
				return container.length == offset && 
				(!container.nextSibling || container.nextSibling.nodeName == "BR"); 			
			case "P" :				
				return true;
		}		
	}	
	
	return false;
};