// ========================================================================
// Copyright 1999-2005 Mort Bay Consulting Pty. Ltd.
// ------------------------------------------------------------------------
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// http://www.apache.org/licenses/LICENSE-2.0
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// ========================================================================

package ix;

import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.List;

/**
 * Allows multiple exceptions to be thrown as a single exception,
 * copied from Jetty and refactored.
 */
public class MultiException extends RuntimeException {

    private final List<Throwable> _nested = new ArrayList<Throwable>();

    public MultiException() {
        super("Multiple exceptions");
    }

    public void add(Throwable throwable) {
        if (throwable != null) {
            if (throwable instanceof MultiException) {
                final MultiException me = (MultiException) throwable;
                // noinspection AccessingNonPublicFieldOfAnotherObject
                _nested.addAll(me._nested);
            } else {
                _nested.add(throwable);
            }
        }
    }

    /**
     * If this multi exception is empty then no action is taken,
     * if it contains a single <code>Throwable</code> that is thrown,
     * otherwise this <code>MultiException</code> is thrown.
     */
    public void throwExceptionIfNotEmpty() throws Exception {
        if (_nested.isEmpty()) {
            // Do nothing
        } else if (_nested.size() == 1) {
            final Throwable throwable = _nested.get(0);
            if (throwable instanceof Error) {
                throw (Error) throwable;
            } else if (throwable instanceof Exception) {
                throw (Exception) throwable;
            } else {
                throw new Exception(throwable);
            }
        } else {
            throw this;
        }
    }

    /**
     * If this multi exception is empty then no action is taken,
     * if it contains a single <code>RuntimeException<code> that is thrown,
     * if it contains a single <code>Error<code> that is thrown,
     * otherwise a new <code>RuntimeException</code> with a proper cause is thrown.
     */
    public void throwRuntimeExceptionIfNotEmpty() throws RuntimeException {
        if (_nested.isEmpty()) {
            // Do nothing
        } else if (_nested.size() == 1) {
            final Throwable throwable = _nested.get(0);
            if (throwable instanceof RuntimeException) {
                throw (RuntimeException) throwable;
            } else if (throwable instanceof Error) {
                throw (Error) throwable;
            } else {
                throw new RuntimeException(throwable);
            }
        } else {
            throw new RuntimeException(this);
        }
    }

    @Override
    public void printStackTrace(PrintStream out) {
        synchronized (out) {
            super.printStackTrace(out);
            for (Throwable e : _nested) {
                out.println();
                e.printStackTrace(out);
            }
        }
    }

    @Override
    public void printStackTrace(PrintWriter out) {
        synchronized (out) {
            super.printStackTrace(out);
            for (Throwable e : _nested) {
                out.println();
                e.printStackTrace(out);
            }
        }
    }

    public String toString() {
        int n = _nested.size();
        if (n == 0) {
            return getClass().getName() + " with no nested exception";
        } else if (n == 1) {
            return getClass().getName() + " with 1 nested exception:\n" + _nested.get(0).toString();
        } else {
            StringBuilder sb = new StringBuilder();
            sb.append(getClass().getName()).append(" with ").append(n).append(" nested exceptions:");
            for (Throwable e : _nested) {
                sb.append('\n').append(e);
            }
            return sb.toString();
        }
    }
}
