/* ==========================================================================
 *
 * (c) 2010 Christoph Leisegang
 *
 * ========================================================================== */
package model.neo4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.Relationship;
import org.neo4j.graphdb.ReturnableEvaluator;
import org.neo4j.graphdb.StopEvaluator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.Traverser;

class UserImpl extends BaseImpl implements User {

	private static final String KEY_NAME = "name";

	public UserImpl(Node theNode, Neo4jDatabaseService dbService) {
		super(theNode, dbService);
	}

	private Iterator<Label> getLabelsIter() {
		return new Iterator<Label>() {
			private final Iterator<Node> iterator = getTheNode().traverse(
					Traverser.Order.BREADTH_FIRST, StopEvaluator.DEPTH_ONE,
					ReturnableEvaluator.ALL_BUT_START_NODE,
					ServiceRelationshipTypes.USER_TO_LABEL, Direction.OUTGOING)
					.iterator();

			public boolean hasNext() {
				return iterator.hasNext();
			}

			public Label next() {
				return new LabelImpl(iterator.next(), getDatabaseService());
			}

			public void remove() {
				iterator.remove();
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.User#getLabels()
	 */
	@Override
	public Collection<Label> getLabels() {
		final Collection<Label> labels = new ArrayList<Label>();

		Transaction tx = getDatabaseService().beginTx();

		Iterator<Label> iter = getLabelsIter();

		while (iter.hasNext()) {
			labels.add(iter.next());
		}

		tx.success();
		tx.finish();

		return labels;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.User#addLabel(model.Label)
	 */
	@Override
	public void add(Label label) {

		if (!isConnected(label)) {
			Transaction tx = getDatabaseService().beginTx();

			try {
				getTheNode().createRelationshipTo(
						((LabelImpl) label).getTheNode(),
						ServiceRelationshipTypes.USER_TO_LABEL);
				tx.success();
			} catch (Exception e) {
				tx.finish();
			} finally {
				tx.finish();
			}
		}

	}

	private boolean isConnected(Label label) {

		boolean isConnected = false;
		Iterable<Relationship> relIter = getTheNode().getRelationships(
				ServiceRelationshipTypes.USER_TO_LABEL);
		for (Relationship relationship : relIter) {
			if (relationship.getEndNode().equals(
					((LabelImpl) label).getTheNode())) {
				isConnected = true;
				break;
			}
		}

		return isConnected;
	}

	private Iterator<Bookmark> getBookmarksIter() {
		return new Iterator<Bookmark>() {
			private final Iterator<Node> iterator = getTheNode().traverse(
					Traverser.Order.BREADTH_FIRST, StopEvaluator.DEPTH_ONE,
					ReturnableEvaluator.ALL_BUT_START_NODE,
					ServiceRelationshipTypes.USER_TO_BM, Direction.OUTGOING)
					.iterator();

			public boolean hasNext() {
				return iterator.hasNext();
			}

			public Bookmark next() {
				return new BookmarkImpl(iterator.next(), getDatabaseService());
			}

			public void remove() {
				iterator.remove();
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.User#getBookmarks()
	 */
	@Override
	public Collection<Bookmark> getBookmarks() {
		final Collection<Bookmark> bookmarks = new ArrayList<Bookmark>();

		Transaction tx = getDatabaseService().beginTx();

		Iterator<Bookmark> iter = getBookmarksIter();

		while (iter.hasNext()) {
			bookmarks.add(iter.next());
		}

		tx.success();
		tx.finish();

		return bookmarks;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.User#addBookmark(model.Bookmark)
	 */
	@Override
	public void add(Bookmark bookmark) {

		if (!isConnected(bookmark)) {
			Transaction tx = getDatabaseService().beginTx();

			try {
				getTheNode().createRelationshipTo(
						((BookmarkImpl) bookmark).getTheNode(),
						ServiceRelationshipTypes.USER_TO_BM);

				tx.success();
			} catch (Exception e) {
				tx.failure();
			} finally {
				tx.finish();
			}
		}
	}

	private boolean isConnected(Bookmark bookmark) {

		boolean isConnected = false;
		Iterable<Relationship> relIter = getTheNode().getRelationships(
				ServiceRelationshipTypes.USER_TO_BM);
		for (Relationship relationship : relIter) {
			if (relationship.getEndNode().equals(bookmark)) {
				isConnected = true;
				break;
			}
		}

		return isConnected;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.User#getName()
	 */
	@Override
	public String getName() {
		return (String) getTheNode().getProperty(KEY_NAME);
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof UserImpl) {
			return getTheNode().equals(((UserImpl) obj).getTheNode());
		}
		return false;
	}

	@Override
	public int hashCode() {
		return getTheNode().hashCode();
	}

	@Override
	public String toString() {
		return String.format("UserImpl [node id=%d, name=%s]", getTheNode()
				.getId(), getName());
	}

	@Override
	public void remove(Label label) {

		Transaction tx = getDatabaseService().beginTx();

		try {
			Iterable<Relationship> relIter = getTheNode().getRelationships(
					ServiceRelationshipTypes.USER_TO_LABEL);
			for (Relationship relationship : relIter) {
				if (relationship.getEndNode().equals(
						((LabelImpl) label).getTheNode())) {
					relationship.delete();
					break;
				}
			}

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	@Override
	public void remove(Bookmark bookmark) {

		Transaction tx = getDatabaseService().beginTx();

		try {
			Iterable<Relationship> relIter = getTheNode().getRelationships(
					ServiceRelationshipTypes.USER_TO_BM);
			for (Relationship relationship : relIter) {
				if (relationship.getEndNode().equals(
						((BookmarkImpl) bookmark).getTheNode())) {
					relationship.delete();
					break;
				}
			}

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}
}
