/* ==========================================================================
 *
 * (c) 2010 Christoph Leisegang
 *
 * ========================================================================== */
package model.neo4j;

import java.util.ArrayList;
import java.util.Collection;

import model.BookmarkDto;
import model.LabelDto;
import model.ModelService;
import model.UserDto;

import org.bushe.swing.event.EventService;
import org.bushe.swing.event.generics.TypeReference;
import org.neo4j.graphdb.NotFoundException;

import presenter.dto.BookmarkViewDto;
import presenter.dto.LabelViewDto;

import com.google.inject.Inject;

public class Neo4jModelService implements ModelService {

	private final Neo4jDatabaseService dbService;

	@Inject
	private EventService eventService;

	private User getOrCreateUser(final UserDto userDto) {

		User user;

		try {
			user = dbService.getUserById(((UserDtoImpl) userDto).getId());
		} catch (final NotFoundException e) {
			user = dbService.getOrCreateUser(userDto.getName());
		}

		return user;
	}

	private Bookmark getOrCreateBookmark(final BookmarkDto bookmarkDto) {

		Bookmark bookmark;

		try {
			bookmark = dbService
					.getBookmarkById(((BookmarkDtoImpl) bookmarkDto).getId());
		} catch (final NotFoundException e) {
			bookmark = dbService.createBookmark(bookmarkDto.getTitle(),
					bookmarkDto.getUri(), bookmarkDto.getDescription());
		}

		return bookmark;
	}

	private Label getOrCreateLabel(final LabelDto labelDto) {

		Label label;

		try {
			label = dbService.getLabelById(((LabelDtoImpl) labelDto).getId());
		} catch (final NotFoundException e) {
			label = dbService.getOrCreateLabel(labelDto.getName());
		}

		return label;
	}

	public Neo4jModelService() {
		this(new Neo4jDatabaseService());
	}

	public Neo4jModelService(final Neo4jDatabaseService dbService) {
		this.dbService = dbService;
	}

	@Override
	public void addBookmark(final BookmarkDto bookmarkDto, final UserDto userDto) {
		getOrCreateUser(userDto).add(getOrCreateBookmark(bookmarkDto));
	}

	@Override
	public void addLabel(final LabelDto labelDto, final UserDto userDto) {
		getOrCreateUser(userDto).add(getOrCreateLabel(labelDto));
	}

	@Override
	public void getBookmarks(final UserDto userDto) {

		final User user = getOrCreateUser(userDto);
		final Collection<Bookmark> bookmarks = user.getBookmarks();

		final Collection<BookmarkDto> bookmarksDto = new ArrayList<BookmarkDto>();

		for (final Bookmark bookmark : bookmarks) {
			bookmarksDto.add(new BookmarkDtoImpl(bookmark, user));
		}

		eventService.publish(new TypeReference<Collection<BookmarkDto>>() {
		}.getType(), bookmarksDto);
	}

	@Override
	public void getBookmarks(final UserDto userDto, final LabelDto labelDto) {

		final User user = getOrCreateUser(userDto);
		final Collection<Bookmark> bookmarks = getOrCreateLabel(labelDto)
				.getBookmarks(user);

		final Collection<BookmarkDto> bookmarksDto = new ArrayList<BookmarkDto>();

		for (final Bookmark bookmark : bookmarks) {
			bookmarksDto.add(new BookmarkDtoImpl(bookmark, user));
		}

		eventService.publish(new TypeReference<Collection<BookmarkDto>>() {
		}.getType(), bookmarksDto);
	}

	@Override
	public void getLabels(final UserDto userDto) {
		final User user = getOrCreateUser(userDto);
		final Collection<Label> labels = getOrCreateUser(userDto).getLabels();

		final Collection<LabelDto> labelsDto = new ArrayList<LabelDto>();

		for (final Label label : labels) {
			labelsDto.add(new LabelDtoImpl(label, user));
		}

		eventService.publish(new TypeReference<Collection<LabelDto>>() {
		}.getType(), labelsDto);
	}

	@Override
	public void getLabels(final BookmarkDto bookmarkDto, final UserDto userDto) {
		final User user = getOrCreateUser(userDto);
		final Collection<Label> labels = getOrCreateBookmark(bookmarkDto)
				.getLabels(getOrCreateUser(userDto));

		final Collection<LabelDto> labelsDto = new ArrayList<LabelDto>();

		for (final Label label : labels) {
			labelsDto.add(new LabelDtoImpl(label, user));
		}

		eventService.publish(new TypeReference<Collection<LabelDto>>() {
		}.getType(), labelsDto);
	}

	@Override
	public void addAndConnectBookmarkAndLabel(final BookmarkDto bookmarkDto,
			final LabelDto labelDto, final UserDto userDto) {

		final User user = getOrCreateUser(userDto);
		final Label label = getOrCreateLabel(labelDto);
		final Bookmark bookmark = getOrCreateBookmark(bookmarkDto);

		user.add(bookmark);
		user.add(label);
		bookmark.add(label);
	}

	@Override
	public void update(BookmarkViewDto bookmarkViewDto, UserDto userDto) {

		final User user = getOrCreateUser(userDto);
		final BookmarkDto bookmarkDto = new BookmarkDtoImpl(bookmarkViewDto);
		BookmarkImpl bookmark = (BookmarkImpl) getOrCreateBookmark(bookmarkDto);

		if (!bookmarkDto.getTitle().equals(bookmark.getTitle())) {
			bookmark.setTitle(bookmarkDto.getTitle());
		}

		if (!bookmarkDto.getUri().equals(bookmark.getUri())) {
			bookmark.setUri(bookmarkDto.getUri());
		}

		if (!bookmarkDto.getDescription().equals(bookmark.getDescription())) {
			bookmark.setDescription(bookmarkDto.getDescription());
		}

		// Handle new labels
		Collection<Label> newLabels = getLabels(bookmarkDto);
		newLabels.removeAll(bookmark.getLabels(user));
		for (Label label : newLabels) {
			user.add(label);
			bookmark.add(label);
		}

		// Handle removed labels
		Collection<Label> removeLabels = bookmark.getLabels(user);
		removeLabels.removeAll(getLabels(bookmarkDto));
		for (Label label : removeLabels) {
			bookmark.remove(label);
			if (label.getCountTotal() == 0) {
				user.remove(label);
				// label.delete();
			}
		}

		eventService.publish(BookmarkDto.class,
				getOrCreateBookmark(bookmarkDto));

		getLabels(userDto);
	}

	private Collection<Label> getLabels(BookmarkDto bookmarkDto) {
		Collection<Label> labelsNew = new ArrayList<Label>();
		for (LabelDto labelDto : bookmarkDto.getLabels()) {
			labelsNew.add(getOrCreateLabel(labelDto));
		}
		return labelsNew;
	}

	@Override
	public void update(LabelViewDto labelViewDto, UserDto userDto) {
		// TODO Auto-generated method stub

	}

	@Override
	public void update(UserDto userDto) {
		// TODO Auto-generated method stub

	}

	@Override
	public void delete(BookmarkDto bookmarkDto, UserDto userDto) {
		// TODO Auto-generated method stub

	}

	@Override
	public void delete(LabelDto label, UserDto userDto) {
		// TODO Auto-generated method stub

	}

	@Override
	public void delete(UserDto userDto) {
		// TODO Auto-generated method stub

	}

	@Override
	public void start() {
		dbService.start();
	}

	@Override
	public void stop() {
		dbService.shutdown();
	}

	@Override
	public BookmarkDto createBookmarkDto(final String title, final String uri,
			final String description) {
		final Bookmark bookmark = getOrCreateBookmark(new BookmarkDtoImpl(
				title, uri, description));
		return new BookmarkDtoImpl(bookmark);
	}

	@Override
	public LabelDto createLabelDto(final String name) {
		final Label label = getOrCreateLabel(new LabelDtoImpl(name, 0));
		return new LabelDtoImpl(label);
	}

	Neo4jDatabaseService getDbService() {
		return dbService;
	}
}
