/* ==========================================================================
 *
 * (c) 2010 Christoph Leisegang
 *
 * ========================================================================== */
package model.neo4j;

import java.util.Properties;

import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.GraphDatabaseService;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.NotFoundException;
import org.neo4j.graphdb.RelationshipType;
import org.neo4j.graphdb.ReturnableEvaluator;
import org.neo4j.graphdb.StopEvaluator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.TraversalPosition;
import org.neo4j.graphdb.Traverser;
import org.neo4j.kernel.EmbeddedGraphDatabase;

public class Neo4jDatabaseService {

	private final static String GRAPHDB_DEFAULT_PATH = "var/graphdb";
	private final static String DEFAULT_USER = "public";

	private final static String NAME_PROPERTY_KEY = "name";
	private final static String TITLE_PROPERTY_KEY = "title";
	private final static String URI_PROPERTY_KEY = "uri";
	private static final String URI_PROPERTY_DESCRIPTION = "description";

	private final GraphDatabaseService graphDb;

	private Node bookmarksRoot;
	private Node labelsRoot;
	private Node usersRoot;
	private Node publicUser;

	public Neo4jDatabaseService() {
		this(GRAPHDB_DEFAULT_PATH);
	}

	public Neo4jDatabaseService(String dbName) {
		this(new EmbeddedGraphDatabase(dbName));
	}

	public Neo4jDatabaseService(GraphDatabaseService graphDb) {
		this.graphDb = graphDb;

		Transaction tx = graphDb.beginTx();
		try {
			Node root = graphDb.getReferenceNode();
			this.bookmarksRoot = getOrCreateBookmarksRoot(root);
			this.labelsRoot = getOrCreateLabelsRoot(root);
			this.usersRoot = getOrCreateUsersRoot(root);
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

	private Node getOrCreateBookmarksRoot(Node root) {

		final Node bookmarks;

		if (root.hasRelationship(ServiceRelationshipTypes.BOOKMARKS)) {
			bookmarks = root.getSingleRelationship(
					ServiceRelationshipTypes.BOOKMARKS, Direction.OUTGOING)
					.getEndNode();
		} else {
			bookmarks = graphDb.createNode();
			bookmarks.setProperty(NAME_PROPERTY_KEY, "bookmarks");

			root.createRelationshipTo(bookmarks,
					ServiceRelationshipTypes.BOOKMARKS);
		}

		return bookmarks;
	}

	private Node getOrCreateLabelsRoot(Node root) {

		final Node labels;

		if (root.hasRelationship(ServiceRelationshipTypes.LABELS)) {
			labels = root.getSingleRelationship(
					ServiceRelationshipTypes.LABELS, Direction.OUTGOING)
					.getEndNode();
		} else {
			labels = graphDb.createNode();

			labels.setProperty(NAME_PROPERTY_KEY, "labels");

			root.createRelationshipTo(labels, ServiceRelationshipTypes.LABELS);
		}

		return labels;
	}

	private Node getOrCreateUsersRoot(Node root) {

		final Node users;

		if (root.hasRelationship(ServiceRelationshipTypes.USERS)) {
			users = root.getSingleRelationship(ServiceRelationshipTypes.USERS,
					Direction.OUTGOING).getEndNode();
		} else {
			users = graphDb.createNode();
			users.setProperty(NAME_PROPERTY_KEY, "users");

			root.createRelationshipTo(users, ServiceRelationshipTypes.USERS);

			publicUser = graphDb.createNode();
			publicUser.setProperty("name", DEFAULT_USER);
			users.createRelationshipTo(publicUser,
					ServiceRelationshipTypes.USER);
		}

		return users;
	}

	private Node lookupNode(final Properties properties, Node startNode,
			RelationshipType relType) {
		ReturnableEvaluator findByProperties = new ReturnableEvaluator() {
			@Override
			public boolean isReturnableNode(TraversalPosition position) {

				Boolean found = true;

				for (Object key : properties.keySet()) {
					found &= position.currentNode().getProperty((String) key,
							"").equals(properties.getProperty((String) key));
				}

				return found;
			}
		};

		final Traverser traverser = startNode.traverse(
				Traverser.Order.BREADTH_FIRST, StopEvaluator.DEPTH_ONE,
				findByProperties, relType, Direction.OUTGOING);

		if (traverser.iterator().hasNext()) {
			return traverser.iterator().next();
		} else
			return null;
	}

	public User getPublicUser() {

		return new UserImpl(publicUser, this);
	}

	public Bookmark getBookmarkById(long id) throws NotFoundException {

		Bookmark bookmark;

		final Transaction tx = graphDb.beginTx();
		try {
			bookmark = new BookmarkImpl(graphDb.getNodeById(id), this);
			tx.success();
		} catch (NotFoundException e) {
			bookmark = null;
			tx.failure();
			throw e;
		} finally {
			tx.finish();
		}

		return bookmark;
	}

	public Bookmark createBookmark(final String title, final String uri,
			String description) {

		Node bmark = null;

		final Transaction tx = graphDb.beginTx();
		try {
			bmark = graphDb.createNode();
			bmark.setProperty(TITLE_PROPERTY_KEY, title);
			bmark.setProperty(URI_PROPERTY_KEY, uri);
			bmark.setProperty(URI_PROPERTY_DESCRIPTION, description);
			this.bookmarksRoot.createRelationshipTo(bmark,
					ServiceRelationshipTypes.BOOKMARK);

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}

		return new BookmarkImpl(bmark, this);
	}

	public Label getLabelById(long id) throws NotFoundException {

		Label label;

		final Transaction tx = graphDb.beginTx();
		try {
			label = new LabelImpl(graphDb.getNodeById(id), this);
			tx.success();
		} catch (NotFoundException e) {
			label = null;
			tx.failure();
			throw e;
		} finally {
			tx.finish();
		}

		return label;
	}

	public Label getOrCreateLabel(final String name) {

		final Properties properties = new Properties();
		properties.setProperty(NAME_PROPERTY_KEY, name);

		final Transaction tx = graphDb.beginTx();

		Node label = lookupNode(properties, this.labelsRoot,
				ServiceRelationshipTypes.LABEL);

		if (label == null) {
			label = graphDb.createNode();
			label.setProperty(NAME_PROPERTY_KEY, name);
			this.labelsRoot.createRelationshipTo(label,
					ServiceRelationshipTypes.LABEL);
		}

		tx.success();
		tx.finish();

		return new LabelImpl(label, this);
	}

	public User getUserById(long id) throws NotFoundException {

		User user;

		final Transaction tx = graphDb.beginTx();
		try {
			user = new UserImpl(graphDb.getNodeById(id), this);
			tx.success();
		} catch (NotFoundException e) {
			user = null;
			tx.failure();
			throw e;
		} finally {
			tx.finish();
		}

		return user;
	}

	public User getOrCreateUser(final String name) {

		final Properties properties = new Properties();
		properties.setProperty(NAME_PROPERTY_KEY, name);

		final Transaction tx = graphDb.beginTx();

		Node user = lookupNode(properties, this.usersRoot,
				ServiceRelationshipTypes.USER);

		if (user == null) {
			user = graphDb.createNode();
			user.setProperty(NAME_PROPERTY_KEY, name);
			this.usersRoot.createRelationshipTo(user,
					ServiceRelationshipTypes.USER);
		}

		tx.success();
		tx.finish();

		return new UserImpl(user, this);
	}

	public void shutdown() {
		graphDb.shutdown();
	}

	public void start() {
		// TODO Auto-generated method stub
	}

	protected Transaction beginTx() {
		return graphDb.beginTx();
	}

}
