/* ==========================================================================
 *
 * (c) 2010 Christoph Leisegang
 *
 * ========================================================================== */
package model.neo4j;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.neo4j.graphdb.Direction;
import org.neo4j.graphdb.Node;
import org.neo4j.graphdb.ReturnableEvaluator;
import org.neo4j.graphdb.StopEvaluator;
import org.neo4j.graphdb.Transaction;
import org.neo4j.graphdb.TraversalPosition;
import org.neo4j.graphdb.Traverser;

class LabelImpl extends BaseImpl implements Label {

	private static final String KEY_NAME = "name";

	public LabelImpl(Node theNode, Neo4jDatabaseService dbService) {
		super(theNode, dbService);
	}

	private Iterator<Bookmark> getBookmarksIter(final User user) {
		return new Iterator<Bookmark>() {
			private final Iterator<Node> iterator = getTheNode()
					.traverse(
							Traverser.Order.BREADTH_FIRST,
							StopEvaluator.DEPTH_ONE,
							new ReturnableEvaluator() {
								public boolean isReturnableNode(
										TraversalPosition pos) {
									final Node bmark = pos.currentNode();

									Traverser traverser = bmark.traverse(
											Traverser.Order.BREADTH_FIRST,
											StopEvaluator.DEPTH_ONE,
											new ReturnableEvaluator() {
												public boolean isReturnableNode(
														TraversalPosition pos) {
													return pos
															.currentNode()
															.equals(
																	((UserImpl) user)
																			.getTheNode());
												}
											},
											ServiceRelationshipTypes.USER_TO_BM,
											Direction.INCOMING);

									return !pos.isStartNode()
											&& traverser.iterator().hasNext();
								}
							}, ServiceRelationshipTypes.BM_TO_LABEL,
							Direction.INCOMING).iterator();

			public boolean hasNext() {
				return iterator.hasNext();
			}

			public Bookmark next() {
				Node nextNode = iterator.next();
				return new BookmarkImpl(nextNode, getDatabaseService());
			}

			public void remove() {
				iterator.remove();
			}
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Label#getBookmarks(model.User)
	 */
	@Override
	public Collection<Bookmark> getBookmarks(User user) {

		final Collection<Bookmark> bookmarks = new ArrayList<Bookmark>();

		Transaction tx = getDatabaseService().beginTx();

		try {
			Iterator<Bookmark> iter = getBookmarksIter(user);

			while (iter.hasNext()) {
				bookmarks.add(iter.next());
			}

			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}

		return bookmarks;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see model.neo4j.Label#getName()
	 */
	@Override
	public String getName() {

		Transaction tx = getDatabaseService().beginTx();
		String name;

		try {
			name = (String) getTheNode().getProperty(KEY_NAME, "NOT FOUND");
			tx.success();
		} catch (Exception e) {
			name = "";
			tx.failure();
		} finally {
			tx.finish();
		}

		return name;
	}

	public void setName(final String name) {
		getTheNode().setProperty(KEY_NAME, name);
	}

	@Override
	public int getCountTotal() {
		Transaction tx = getDatabaseService().beginTx();

		int count = 0;
		try {
			count = getTheNode().traverse(Traverser.Order.BREADTH_FIRST,
					StopEvaluator.DEPTH_ONE,
					ReturnableEvaluator.ALL_BUT_START_NODE,
					ServiceRelationshipTypes.BM_TO_LABEL, Direction.INCOMING)
					.getAllNodes().size();
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
		return count;
	}

	@Override
	public int getCount(final User user) {
		Transaction tx = getDatabaseService().beginTx();

		int count = 0;
		try {
			count = getTheNode()
					.traverse(
							Traverser.Order.BREADTH_FIRST,
							StopEvaluator.DEPTH_ONE,
							new ReturnableEvaluator() {
								public boolean isReturnableNode(
										TraversalPosition pos) {
									final Node bmark = pos.currentNode();

									Traverser traverser = bmark.traverse(
											Traverser.Order.BREADTH_FIRST,
											StopEvaluator.DEPTH_ONE,
											new ReturnableEvaluator() {
												public boolean isReturnableNode(
														TraversalPosition pos) {
													return pos
															.currentNode()
															.equals(
																	((UserImpl) user)
																			.getTheNode());
												}
											},
											ServiceRelationshipTypes.USER_TO_BM,
											Direction.INCOMING);

									return !pos.isStartNode()
											&& traverser.iterator().hasNext();
								}
							}, ServiceRelationshipTypes.BM_TO_LABEL,
							Direction.INCOMING).getAllNodes().size();
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
		return count;
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof LabelImpl) {
			return getTheNode().equals(((LabelImpl) obj).getTheNode());
		}
		return false;
	}

	@Override
	public int hashCode() {
		return getTheNode().hashCode();
	}

	@Override
	public String toString() {
		return String.format("LabelImpl [node id=%d, name=%s]", getTheNode()
				.getId(), getName());
	}

	@Override
	public void delete() {
		Transaction tx = getDatabaseService().beginTx();

		try {
			getTheNode().delete();
			tx.success();
		} catch (Exception e) {
			tx.failure();
		} finally {
			tx.finish();
		}
	}

}
