﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Threading.Tasks;
using System.Collections.Concurrent;
using System.Drawing.Imaging;

namespace ImageProcessing {
    class ImageProcessor {

        ////////////////////////////////////////////////////////////////////////////////
        //
        // Konvertiert das Bild in ein Hue-Grauwert
        // (extrem langsam, zeigt aber kurz und knapp den Algoritmus)
        //
        // Nebenbemerkung:
        // Diese Varinate lässt sich auch nicht vernuenftig parallelisieren. Da
        // Bitmap nicht threadsafe ist, muss jeder Zugriff darauf mit einem Lock
        // geschuetzt werden. Ansonsten fängt man sich eine Exception.
        // Daher arbeiten wir im Folgenden mit einem Byte-Array für die Pixel.
        //
        public static Bitmap ToHueSlow(Bitmap b) {
            for (int y = 0; y < b.Height; y++) {
                for (int x = 0; x < b.Width; x++) {
                    Color c = b.GetPixel(x, y);
                    int hue, saturation, value;
                    ColorToHSV(c, out hue, out saturation, out value);
                    b.SetPixel(x, y, Color.FromArgb(hue, hue, hue));
                }
            }
            return b;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // Konvertiert das Bild in ein Hue-Grauwert via Pixel-Array
        //
        public static Bitmap ToHueNew(Bitmap b) {

            PixelAccess pa = new PixelAccess(b);
            byte[] rgbValues = pa.GetRbgBytes();

            for (int y = 0; y < b.Height; y++) {
                for (int x = 0; x < b.Width; x++) {
                    int index = y * b.Width + x;
                    index *= 3;                         // 3 Bytes pro Pixel
                    Color c = Color.FromArgb(rgbValues[index + 2], rgbValues[index + 1], rgbValues[index]);
                    int hue, saturation, value;
                    ColorToHSV(c, out hue, out saturation, out value);
                    rgbValues[index] = (byte)hue;       // blau
                    rgbValues[index + 1] = (byte)hue;   // gruen
                    rgbValues[index + 2] = (byte)hue;   // rot
                }
            }

            pa.WriteBackRbgBytes(rgbValues);

            return b;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // Konvertiert das Bild in ein Hue-Grauwert via Pixel-Array
        //
        public static Bitmap ToHue(Bitmap b) {

            int totalPixels = b.Height * b.Width;

            PixelAccess pa = new PixelAccess(b);
            byte[] rgbValues = pa.GetRbgBytes();

            for (int i = 0; i < totalPixels; i++) {
                int index = i * 3;

                Color c = Color.FromArgb(rgbValues[index + 2], rgbValues[index + 1], rgbValues[index]);
                int hue, saturation, value;
                ColorToHSV(c, out hue, out saturation, out value);
                rgbValues[index] = (byte)hue;       // blau
                rgbValues[index + 1] = (byte)hue;   // gruen
                rgbValues[index + 2] = (byte)hue;   // rot
            }

            pa.WriteBackRbgBytes(rgbValues);

            return b;
        }


        ////////////////////////////////////////////////////////////////////////////////
        //
        // Konvertiert das Bild in ein Hue-Grauwert, parallel ausgeführt
        //
        public static Bitmap ToHueParallel(Bitmap b) {

            int totalPixels = b.Height * b.Width;

            PixelAccess pa = new PixelAccess(b);
            byte[] rgbValues = pa.GetRbgBytes();


            Parallel.ForEach(Partitioner.Create(0, totalPixels), range => {
                try {
                    for (int i = range.Item1; i < range.Item2; i++) {
                        int index = i * 3;
                        Color c = Color.FromArgb(rgbValues[index+2], rgbValues[index+1], rgbValues[index]);
                        int hue, saturation, value;
                        ColorToHSV(c, out hue, out saturation, out value);
                        rgbValues[index] = (byte)hue;       // blau
                        rgbValues[index + 1] = (byte)hue;   // gruen
                        rgbValues[index + 2] = (byte)hue;   // rot
                    }
                } catch (Exception e) {
                    Console.WriteLine(e.StackTrace);
                }
            });

            pa.WriteBackRbgBytes(rgbValues);

            return b;
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // Hilfsfunktion
        // berechnet Hue, Saturation und Value aus einer Color
        //
        private static void ColorToHSV(Color color, out int h, out int s, out int v) {
            int max = Math.Max(color.R, Math.Max(color.G, color.B));
            int min = Math.Min(color.R, Math.Min(color.G, color.B));
            v = max;
            h = (int)color.GetHue();
            h = h * 255 / 360;
            s = (int)((max == 0) ? 0 : 255 * (1d - ((double)min / max)));
        }

        ////////////////////////////////////////////////////////////////////////////////
        //
        // Hilfsklasse
        // kopiert das Pixel-Array in ein Byte-Array und zurück.
        //
        private class PixelAccess {

            private Bitmap b;
            private System.Drawing.Imaging.BitmapData bmpData;
            private int bytes;
            private IntPtr ptr;

            public PixelAccess(Bitmap bitmap) {
                b = bitmap;
            }

            public byte[] GetRbgBytes() {
                // Bitmap-Bits locken
                Rectangle rect = new Rectangle(0, 0, b.Width, b.Height);
                bmpData =
                    b.LockBits(rect, System.Drawing.Imaging.ImageLockMode.ReadWrite,
                    PixelFormat.Format24bppRgb);

                // Adresse der ersten Zeile
                ptr = bmpData.Scan0;

                // Byte-Array für die Bitmap Daten.
                bytes = bmpData.Stride * b.Height;
                byte[] rgbValues = new byte[bytes];

                // RGB-Werte in den managed Code kopieren (kann man sich sparen, wenn man usafe weiter arbeitet)
                System.Runtime.InteropServices.Marshal.Copy(ptr, rgbValues, 0, bytes);

                return rgbValues;
            }

            public void WriteBackRbgBytes(byte[] rgbValues) {
                // Zurück kopieren nach unmanaged...
                System.Runtime.InteropServices.Marshal.Copy(rgbValues, 0, ptr, bytes);

                // Unlock.
                b.UnlockBits(bmpData);
            }
        }
    }
}
