/********************************************************************
 * $Id: LibraryClientFrame.java 91 2009-10-05 01:11:23Z rvandenh $
 *
 * (c) Copyright 2009 TH Wildau
 * All rights reserved
 ********************************************************************/
package de.thwildau.tm.library.client;

import javax.swing.JPanel;
import javax.swing.JFrame;

import java.awt.Button;
import java.awt.Dimension;
import java.awt.Label;
import java.awt.TextField;
import java.awt.Toolkit;
import javax.swing.BoxLayout;
import java.awt.List;

import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.util.Vector;

import de.thwildau.tm.library.api.Book;
import de.thwildau.tm.library.api.ILibraryService;


/**
 * JFrame class for the library client
 * @author Ralf Vandenhouten
 * @version $Revision: 91 $
 */
public class LibraryClientFrame extends JFrame {

    private static final long USER_ID = 999;

    private static final long serialVersionUID = 1L;
    private JPanel jContentPane = null;
    private Label bookLabel = null;
    private List bookList = null;
    private Button buttonBorrow = null;
    private Label borrowLabel = null;
    private List borrowedList = null;
    private Button buttonReturn = null;
    private TextField statusField = null;
    
    private ILibraryService libraryService;  //  @jve:decl-index=0:
    
    private java.util.List<Book> books = null;
    private java.util.List<Book> borrowedBooks = new Vector<Book>();
    private Book selectedLibraryBook = null;
    private int lendSelectionIndex;
    private long userId = USER_ID;

    /**
     * This is the default constructor
     */
    public LibraryClientFrame() {
        super();
        initialize();
        userId = Long.parseLong(
                System.getProperty("de.thwildau.tm.library.userid", ""+USER_ID));
    }

    /**
     * This method initializes this Frame
     * 
     * @return void
     */
    private void initialize() {
        Dimension frameSize = new Dimension(400, 300);
        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        int top = (screenSize.height - frameSize.height) / 2;
        int left = (screenSize.width - frameSize.width) / 2;
        setSize(frameSize);
        setLocation(left, top);
        this.setIconImage(Toolkit.getDefaultToolkit().getImage(getClass().getResource("/de/thwildau/tm/library/client/book.png")));
        this.setContentPane(getJContentPane());
        this.setTitle("Library Client");
    }

    /**
     * This method initializes bookList	
     */
    private List getBookList() {
        if (bookList == null) {
            bookList = new List();
            bookList.setPreferredSize(new Dimension(110, 600));
            bookList.addItemListener(new ItemListener() {
                public void itemStateChanged(ItemEvent e) {
                    selectedLibraryBook = books.get(bookList.getSelectedIndex());
                }
            });
        }
        return bookList;
    }

    /**
     * This method initializes buttonBorrow	
     */
    private Button getButtonBorrow() {
        if (buttonBorrow == null) {
            buttonBorrow = new Button();
            buttonBorrow.setLabel("Borrow book");
            buttonBorrow.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent e) {
                    lendSelectedBook();
                }
            });
        }
        return buttonBorrow;
    }

    /**
     * This method initializes borrowedList	
     */
    private List getBorrowedList() {
        if (borrowedList == null) {
            borrowedList = new List();
            borrowedList.addItemListener(new ItemListener() {
                public void itemStateChanged(ItemEvent e) {
                    lendSelectionIndex = borrowedList.getSelectedIndex();
                }
            });
        }
        return borrowedList;
    }

    /**
     * This method initializes buttonReturn	
     */
    private Button getButtonReturn() {
        if (buttonReturn == null) {
            buttonReturn = new Button();
            buttonReturn.setLabel("Return book");
            buttonReturn.addActionListener(new java.awt.event.ActionListener() {
                public void actionPerformed(java.awt.event.ActionEvent e) {
                    returnSelectedBook();
                }
            });
        }
        return buttonReturn;
    }

    /**
     * This method initializes statusField	
     */
    private TextField getStatusField() {
        if (statusField == null) {
            statusField = new TextField();
            statusField.setEditable(false);
        }
        return statusField;
    }

    /**
     * This method initializes jContentPane
     * 
     * @return javax.swing.JPanel
     */
    private JPanel getJContentPane() {
        if (jContentPane == null) {
            borrowLabel = new Label();
            borrowLabel.setText("Books borrowed from the library:");
            bookLabel = new Label();
            bookLabel.setText("All books in the library:");
            jContentPane = new JPanel();
            jContentPane.setLayout(new BoxLayout(getJContentPane(), BoxLayout.Y_AXIS));
            jContentPane.setName("");
            jContentPane.add(bookLabel, null);
            jContentPane.add(getBookList(), null);
            jContentPane.add(getButtonBorrow(), null);
            jContentPane.add(borrowLabel, null);
            jContentPane.add(getBorrowedList(), null);
            jContentPane.add(getButtonReturn(), null);
            jContentPane.add(getStatusField(), null);
        }
        return jContentPane;
    }
    
    /**
     * @return the field libraryService.
     */
    public ILibraryService getLibraryService() {
        return libraryService;
    }

    
    /**
     * @param libraryService the field libraryService.
     */
    public void setLibraryService(ILibraryService libraryService) {
        this.libraryService = libraryService;
        fillBookList();
    }

    /**
     * Fills the book list with the books read from the remote service
     */
    public void fillBookList() {
        if (libraryService!=null) {
            try {
                books = libraryService.getSortedBookList();
            } catch (Exception e) {
                statusField.setText("Library service temporarily not available.");
                return;
            }
            bookList.removeAll();
            for (Book b: books)
                bookList.add(b.toString());
            try {
                borrowedBooks = libraryService.getBookList(userId);
            } catch (Exception e) {
                statusField.setText("Library service temporarily not available.");
                return;
            }
            borrowedList.removeAll();
            for (Book b: borrowedBooks)
                borrowedList.add(b.toString());
        }
    }
    
    /**
     * Lends the currently selected book from the library list
     */
    public void lendSelectedBook() {
        if (libraryService!=null) {
            if (selectedLibraryBook != null) {
                long bookId = selectedLibraryBook.getId();
                boolean lendSuccess;
                try {
                    lendSuccess = libraryService.lendBook(userId, bookId);
                } catch (Exception e) {
                    statusField.setText("Library service temporarily not available.");
                    return;
                } 
                if (lendSuccess) {
                    borrowedBooks.add(selectedLibraryBook);
                    borrowedList.add(selectedLibraryBook.toString());
                    statusField.setText("Borrowed selected book successfully.");
                } else {
                    statusField.setText("This book is not available!");
                }
            }
        } else {
            statusField.setText("Library service temporarily not available.");
        }
    }
    
    /**
     * Returns the currently selected book from the lending list to the library
     */
    public void returnSelectedBook() {
        if (libraryService!=null) {
            if (lendSelectionIndex>=0 && lendSelectionIndex<borrowedBooks.size()) {
                Book book = borrowedBooks.get(lendSelectionIndex);
                try {
                    libraryService.returnBook(book.getId());
                } catch (Exception e) {
                    statusField.setText("Library service temporarily not available.");
                    return;
                }
                borrowedList.remove(lendSelectionIndex);
                borrowedBooks.remove(lendSelectionIndex);
                statusField.setText("Returned selected book successfully.");
            } else {
                statusField.setText("No book selected to return!");
            }
        } else {
            statusField.setText("Library service temporarily not available.");
        }
    }

    /**
     * Displays the "book of the minute" in the status bar
     * @param bookId - the ID of the book to be displayed
     */
    public void displayBookOfTheMinute(long bookId) {
        if (books != null) {
            for (final Book b: books)
                if (b.getId()==bookId) {
                    statusField.setText("Book of the minute is '" + b.getTitle() 
                            + "' by " + b.getAuthor() + ".");
                    return;
                }
        }
        statusField.setText("");
    }
}
