/********************************************************************
 * $Id: LibraryClient.java 88 2009-10-05 00:25:12Z rvandenh $
 *
 * (c) Copyright 2009 TH Wildau
 * All rights reserved
 ********************************************************************/
package de.thwildau.tm.library.client;

import java.io.IOException;
import java.util.Arrays;
import java.util.Hashtable;

import javax.swing.JFrame;

import org.osgi.framework.ServiceRegistration;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventConstants;
import org.osgi.service.event.EventHandler;

import ch.ethz.iks.r_osgi.RemoteOSGiException;
import ch.ethz.iks.r_osgi.RemoteOSGiService;
import ch.ethz.iks.r_osgi.RemoteServiceReference;
import ch.ethz.iks.r_osgi.URI;
import ch.ethz.iks.r_osgi.service_discovery.ServiceDiscoveryListener;
import de.thwildau.tm.library.api.ILibraryService;
 

/**
 * Client for the library service
 * @author Thomas Kistel, Ralf Vandenhouten
 * @version $Revision: 88 $
 */
public class LibraryClient implements ServiceDiscoveryListener, EventHandler {
    
    private static final String TOPIC = "service/library/BOOKOFTHEMINUTE";
    
    private static LibraryClientFrame frame = null;
    
    private RemoteOSGiService remote;
    private ServiceRegistration eventReg;

    /*
     * (non-javadoc)
     * @see ServiceDiscoveryListener#announceService(java.lang.String, ch.ethz.iks.r_osgi.URI)
     */
    public void announceService(String serviceInterface, URI uri) {
        System.out.println("FOUND SERVICE " + serviceInterface + ", URI=" + uri);
        if (serviceInterface.equals(ILibraryService.class.getName())) {
            initializeService(uri);
        }
    }

    /*
     * (non-javadoc)
     * @see ServiceDiscoveryListener#discardService(java.lang.String, ch.ethz.iks.r_osgi.URI)
     */
    public void discardService(String serviceInterface, URI uri) {
        System.out.println("LOST SERVICE " + serviceInterface + ", URI=" + uri);
        if (serviceInterface.equals(ILibraryService.class.getName())) {
            disconnectRemoteFramework(uri);
        }
    }

    /*
     * (non-javadoc)
     * @see org.osgi.service.event.EventHandler#handleEvent(org.osgi.service.event.Event)
     */
    public void handleEvent(Event event) {
        long bookId = (Long)event.getProperty("bookId");
        if (frame!=null)
            frame.displayBookOfTheMinute(bookId);
    }

    /**
     * Called from the SCR when the RemoteOSGiService is available. This method binds the Service to this
     * component.
     * 
     * @param remote the RemoteOSGiService
     */
    public void bind(final RemoteOSGiService remote) {
        System.out.println("Binding RemoteOSGiService " + remote);
        this.remote = remote;
    }

    /**
     * Called from the SCR when the RemoteOSGiService is no longer available.
     * 
     * @param remote the RemoteOSGiService
     */
    public void unbind(final RemoteOSGiService remote) {
        System.out.println("Unbinding RemoteOSGiService " + remote);
        this.remote = null;
    }

    /**
     * Called from the SCR when this Component is activated. It tests whether the DiscoveryService is not
     * available. If so, it connects the remote framework and opens the frame.
     * 
     * @param componentContext the Context of this Component
     */
    protected void activate(ComponentContext componentContext) {
        System.out.println("Activating ServiceComponent " + componentContext);
        final Hashtable<String, Object> props = new Hashtable<String, Object>();
        props.put(EventConstants.EVENT_TOPIC, new String[] { TOPIC });
        eventReg = componentContext.getBundleContext().registerService(
                EventHandler.class.getName(), this, props);
        if (!isServiceDiscoveryAvailable()) {
            final URI uri = getServiceUri();
            System.out.println("URI: " + uri);
            initializeService(uri);
        }
    }

    /**
     * Called from the SCR when this Component is deactivated.
     * 
     * @param componentContext the Context of this Component
     */
    protected void deactivate(ComponentContext componentContext) {
        System.out.println("Deactivating ServiceComponent " + componentContext);
        if (eventReg != null)
            eventReg.unregister();
        if (frame!=null) {
            frame.setLibraryService(null);
            frame.setVisible(false);
            frame.dispose();
            frame = null;
        }
        if (!isServiceDiscoveryAvailable()) {
            disconnectRemoteFramework(getServiceUri());
        }
    }

    /**
     * Connects to the remote {@link ILibraryService} and opens the frame
     * if not already open.
     * @param uri - the {@link URI} to connect to
     */
    private void initializeService(final URI uri) {
        boolean success = connectRemoteFramework(uri);
        final ILibraryService service;
        if (success) {
            final RemoteServiceReference[] refs = 
                remote.getRemoteServiceReferences(uri,
                        ILibraryService.class.getName(), null);
            System.out.println("REFERENCES " + Arrays.asList(refs));
            service = (ILibraryService)remote.getRemoteService(refs[0]);
        } else {
            service = null;
        }
        
        if (frame==null) {
            new Thread(new Runnable() {
                public void run() {
                    frame = new LibraryClientFrame();
                    frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                    frame.setVisible(true);
                    frame.setLibraryService(service);
                }
            }).start();
        } else {
            frame.setLibraryService(service);
        }
    }
    
    /**
     * Tests the property 'ch.ethz.iks.r_osgi.service.discovery'.
     * @return <code>true</code> if the service discovery is available,
     *         otherwise <code>false</code>
     */
    private boolean isServiceDiscoveryAvailable() {
        return Boolean.getBoolean("ch.ethz.iks.r_osgi.service.discovery");
    }

  
    /**
     * Get the URI for connecting to the remote service
     * @return the URI of the remote service
     */
    private URI getServiceUri() {
        return new URI(System.getProperty("ch.ethz.iks.r_osgi.service.uri", 
                "r-osgi://localhost:9278"));
    }

    /**
     * Uses the RemoteOSGiService to connect a remote framework under the given uri.
     * 
     * @param uri the URI of the remote framework
     * @return <code>true</code> if successful, <code>false</code> when the framework could not be connected
     *         due to exceptions
     */
    private boolean connectRemoteFramework(URI uri) {
        if (remote!=null) {
            try {
                remote.connect(uri);
                return true;
            } catch (RemoteOSGiException e) {
                e.printStackTrace();
                return false;
            } catch (IOException e) {
                e.printStackTrace();
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * Disconnects the remote framework.
     * 
     * @param uri the URI of the remote framework
     * @return true if successful, false if Exception occur
     */
    private boolean disconnectRemoteFramework(URI uri) {
        if (remote!=null) {
            try {
                remote.disconnect(uri);
                return true;
            } catch (RemoteOSGiException e) {
                e.printStackTrace();
                return false;
            }
        } else {
            return false;
        }
    }
}
