import groovy.xml.StreamingMarkupBuilder
import java.text.SimpleDateFormat

/**
 * Testing the REST API.
 * Depends on a running instance of the application.
 * @author Dierk Knig
 */

class RestClientAsTest extends GroovyTestCase {

    static final Rest REST = new Rest("http://localhost:8080/ix/nutzer/")

    void testInitialGetOfFirstEntryFromBootStrap() {
        assertEquals 'Dierk', REST.get(1).firstname.text()
    }

    void testInitialListFromBootStrap() {
        assert REST.get().nutzer.@id.contains('1')
    }

    void testInsertANewEntryGetItAndDeleteIt() {
        def initialCount = currentNutzerCount
        def newEntry =  REST.post {
            nutzer {
                birth      new SimpleDateFormat().format(new Date(68,3,19))
                firstname 'Dierk'
                lastname  'Koenig'
                mail      'some.other@mail.address'
            }
        }
        Long id = newEntry.@id.toLong()
        assert initialCount < currentNutzerCount

        def fetchedEntry = REST.get(id)
        assertEquals id, fetchedEntry.@id.toLong()

        def deletedEntry = REST.delete(id)
        assertEquals id, deletedEntry.@id.toLong()
        assertEquals initialCount, currentNutzerCount
    }

    void testPutMethod() {
        def oldNutzer = REST.get(1)
        def update   =  REST.put(1) {
            nutzer {
                mail( oldNutzer.mail.text()-'com'+'org')
            }
        }
        assertEquals 'org', update.mail.text()[-3..-1]
    }

    void testSaveException() {
        def result =  REST.post {}
        assertErrorCount result, 1
    }

    void testSaveError() {
        def result =  REST.post {
            nutzer {
                birth      new SimpleDateFormat().format(new Date())
                mail      'false-format'
            }
        }
        assertErrorCount result, 4
    }

    void testPutException() {
        def result =  REST.put (0) {}
        assertErrorCount result, 1
    }

    void testPutError() {
        def result = REST.put(1) {
            nutzer {mail 'false.format'}
        }
        assertErrorCount result, 1
    }

    void testDeleteException() {
        def result = REST.delete(0)
        assertErrorCount result, 1
    }

    private void assertErrorCount(Node result, int count) {
        assertEquals 'errors', result.name()
        assertEquals count, result.error.size()
        println 'Expected errors:'
        println '- ' + result.error.@message.join("\n- ")
    }

    private int getCurrentNutzerCount(){
        REST.get().nutzer.size()
    }
}

class Rest {
    Rest(String target) { this.@target = target }
    private target  = null
    private parser  = new XmlParser()
    private builder = new StreamingMarkupBuilder()

    def get   (id=0)     { call("GET",    id) }
    def put   (id, body) { call("PUT",    id, body) }
    def post  (body)     { call("POST",    0, body) }
    def delete(id)       { call("DELETE", id) }

    def call(String method, Long id, Closure body=null) {

        HttpURLConnection httpConnection = new URL(target + id ).openConnection()
        httpConnection.requestMethod = method
        httpConnection.doInput = true
        httpConnection.addRequestProperty("Accept", "text/xml,application/xml;q=0.9")
        httpConnection.addRequestProperty("Content-Type", "text/xml")
        if (body) {
            httpConnection.doOutput = true
            httpConnection.outputStream.withWriter("ASCII") {
                it << builder.bind(body)
            }
        }
        if (httpConnection.responseCode == httpConnection.HTTP_OK) {
            return parser.parse(httpConnection.inputStream)
        }
        def msg = "Operation failed: ${httpConnection.responseCode} " +
                  "for URL: ${httpConnection.URL}"
        throw new GroovyRuntimeException(msg)
    }
}