package addressbook.web;

import addressbook.service.Address;
import addressbook.service.AddressDao;
import addressbook.service.AddressDaoException;
import addressbook.service.XmlAddressDao;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Controller für die Adressverwaltung, der Methoden zur Ansteuerung der
 * Geschäftslogik zur Verfügung stellt.
 */
public class Controller {

  private static final Log LOG = LogFactory.getLog(Controller.class);

  private static final String OUTCOME_LIST = "list";
  private static final String OUTCOME_EDITOR = "editor";

  private AddressDao addressDao = new XmlAddressDao();

  /**
   * Die aktuelle, zu editierende Adresse.
   */
  private Address currentAddress = new Address();

  /**
   * Modell aller Adressen für den DataTable.
   */
  private List<AddressRow> addressRows = new ArrayList<AddressRow>();

  /**
   * Suchkriterium zum Einschränken der Adressen in der Tabelle.
   */
  private String searchCriterion;

  public Controller() {
    try {
      initAddressRows();
    } catch (AddressDaoException e) {
      LOG.error(null, e);
    }
  }

  /**
   * Aktualisiert die Adressliste auf Basis des Suchkriteriums.
   */
  public String search() throws AddressDaoException {
    initAddressRows();
    return OUTCOME_LIST;
  }

  /**
   * Legt eine neue Adresse an.
   */
  public String createAddress() {
    currentAddress = new Address();
    return OUTCOME_EDITOR;
  }

  /**
   * Stellt die erste selektierte Adresse im Editor dar.
   */
  public String editAddress() {
    currentAddress = null;
    for (AddressRow row : addressRows) {
      if (row.isSelected()) {
        currentAddress = row.getAddress();
        break;
      }
    }
    if (currentAddress == null) {
      return createAddress();
    }
    return OUTCOME_EDITOR;
  }

  /**
   * Löscht alle selektierten Adressen.
   */
  public String deleteAddresses() throws AddressDaoException {
    for (AddressRow row : addressRows) {
      if (row.isSelected()) {
        addressDao.removeAddress(row.getAddress());
      }
    }
    return search();
  }

  /**
   * Persistieren der aktuellen Adresse im Editor.
   */
  public String store() throws AddressDaoException {
    addressDao.updateAddress(currentAddress);
    searchCriterion = null;
    return search();
  }

  /**
   * Bricht das Editieren der aktuellen Adresse ab.
   */
  public String cancel() throws AddressDaoException {
    searchCriterion = null;
    return search();
  }

  public Address getCurrentAddress() {
    return currentAddress;
  }

  public void setCurrentAddress(Address currentAddress) {
    this.currentAddress = currentAddress;
  }

  public List<AddressRow> getAddressRows() {
    return addressRows;
  }

  public void setAddressRows(List<AddressRow> addressRows) {
    this.addressRows = addressRows;
  }

  public String getSearchCriterion() {
    return searchCriterion;
  }

  public void setSearchCriterion(String searchCriterion) {
    this.searchCriterion = searchCriterion;
  }

  /**
   * Initialisiert die Adressliste in Abhängigkeit des Suchkriteriums.
   */
  private void initAddressRows() throws AddressDaoException {
    // Aufsammeln der aktuellen Selektierungen.
    List<Integer> selectedIds = new ArrayList<Integer>();
    for (AddressRow row : addressRows) {
      if (row.isSelected()) {
        selectedIds.add(row.getAddress().getId());
      }
    }
    addressRows.clear();
    List<Address> addresses = addressDao.findAddresses(searchCriterion);
    for (Address address : addresses) {
      AddressRow row = new AddressRow();
      row.setAddress(address);
      if (selectedIds.contains(address.getId())) {
        row.setSelected(true);
      }
      addressRows.add(row);
    }
  }

}
