using System;

namespace Delegates
{
	/// <summary>
	/// Delegate for reporting the change of the status the worker is in
	/// </summary>
	public delegate void StatusChangeEventHandler (CWorker sender, StatusChangeEventArgs e);
	/// <summary>
	/// Delegate for reporting the progress of the work an getting information
	/// about the satisfaction about the work
	/// </summary>
	public delegate EFeedback ProgressChangeEventHandler (CWorker sender, ProgressChangeEventArgs e);
	/// <summary>
	/// A worker, which starts working when he is requested to do so and sends notofications to registred people.
	/// </summary>
	public class CWorker : CPerson
	{
		#region MEMBERS
		private long m_lngLastStartTime;
		private long m_lngLastStopTime;
		private TimeSpan m_tsLastExecutionTime;
		/// <summary>
		/// Current status of the work
		/// </summary>
		private EStatus m_enmStatus;
		/// <summary>
		/// Timer that fires each second as long as work is in progress
		/// </summary>
		private System.Timers.Timer m_timer;
		private const int TIMER_INTERVAL=10000;
        /// <summary>
		/// The current progress of the work (0..100 percent)
		/// </summary>
		private byte m_bytProgress=0;
		#endregion

		#region CONSTRUCTORS
		public CWorker():this("", "")
		{}
		public CWorker(string strFirstName, string strLastName):this(strFirstName, strLastName, DateTime.Today)
		{}
		public CWorker(string strFirstName, string strLastName, DateTime dtDateOfBirth):base(strFirstName, strLastName, dtDateOfBirth)
		{
			m_enmStatus=EStatus.enmIdle; 
			m_timer=new System.Timers.Timer(TIMER_INTERVAL); //A timer that fires all second
			m_timer.AutoReset=true;
			m_timer.Elapsed+=new System.Timers.ElapsedEventHandler(OnTimerElapsed);
		}
		#endregion

		public event ProgressChangeEventHandler ProgressChange;
		public event StatusChangeEventHandler StatusChange;

		/// <summary>
		/// Tells the worker to start working
		/// </summary>
		public void StartWork()
		{ 
			m_lngLastStartTime=DateTime.Now.Ticks;
			//If the old task was finished, begin from scratch, 
			//otherwise continue where you stopped previously.
			if (m_enmStatus==EStatus.enmWorkCompleted) 
				m_bytProgress=0;
			m_enmStatus=EStatus.enmWorkStarted;
			//Tell everybody in the receiver list the about the change of status
			//From now on, the progress counter should be increased
			//by using an internal timer
			m_timer.Start(); 
		}
		/// <summary>
		/// Tells the worker to stop working
		/// </summary>
		public void StopWork()
		{
			m_timer.Stop();
			m_lngLastStopTime=DateTime.Now.Ticks;
			m_tsLastExecutionTime=TimeSpan.FromTicks(m_lngLastStopTime-m_lngLastStartTime);
			m_enmStatus= m_bytProgress <100 ? EStatus.enmIdle : EStatus.enmWorkCompleted;
			//Call the function that will fire the notification event for status changes
			OnStatusChange(new StatusChangeEventArgs(m_enmStatus));
			//Write out the execution Time
			Console.WriteLine ("Execution Time until work completion " +
							   "(including all Notifications): " + 
							   m_tsLastExecutionTime.ToString());
				
		}
		/// <summary>
		/// A timer simulates the progress of the work. 
		/// This method implements a handler for the timer event: 
		/// Shows the progress of work by incrementing the percentage by 10%
		/// each time the timer elapses, until 100% are reached.
		/// </summary>
		/// <param name="sender">The timer that has raised the event</param>
		/// <param name="e">The event arguments provided by the sending timer </param>
		private void OnTimerElapsed(object sender, System.Timers.ElapsedEventArgs e)
		{
			m_enmStatus=EStatus.enmWorkInProgress;
			if (m_bytProgress < 100) m_bytProgress+=10; //Increase the progress percentage
			//Call the Method that will fire the notification event for work progress
			OnProgressChange(new ProgressChangeEventArgs(m_bytProgress));
			
			//Automatically stop working if work is done ;-)
			if (m_bytProgress>=100 && m_timer.Enabled) this.StopWork();  
		}
		
		/// <summary>
		/// Called whenever the worker's work progress changes
		/// </summary>
		/// <param name="e">Contains the new percentage complete</param>
		protected virtual void OnProgressChange(ProgressChangeEventArgs e)
		{
			long lngStartTime = DateTime.Now.Ticks;
			//Notify everybody in the receiver list about the new percentage 		
			if (ProgressChange != null)
			{
				//Iterate manually through the receiver list
				foreach (ProgressChangeEventHandler prog in ProgressChange.GetInvocationList ())
				{
					Console.WriteLine ("{0} {1} will notify a {2} about the progress " +
						"being at {3} Percent", this.FirstName, this.LastName, 
						prog.Method.DeclaringType.Name, m_bytProgress);
					prog.BeginInvoke (this, e, null, null);

				}
			}
			else
			{
				Console.WriteLine ("Noone wants to be informed about the progress of the work");
			}
			long lngEndTime = DateTime.Now.Ticks;
			Console.WriteLine ("Notification time for progress {0}: {1}", e.PercentComplete, 
				TimeSpan.FromTicks(lngEndTime - lngStartTime));
		}

		/// <summary>
		/// Called whenever the worker's status changes
		/// </summary>
		/// <param name="e">Contains the new Status</param>
		protected virtual void OnStatusChange(StatusChangeEventArgs e)
		{
			//Tell everybody in the receiver list the change of status.
			if (StatusChange != null)
				StatusChange (this, e);
			else
				Console.WriteLine ("Noone wants to be informed about the status of the worker");
		}
		
	}
}
