// (C) 2002 by U. Eisenecker
// Komponenten, DSL und Generator fr Konto-Beispiel

#include <iostream>
using namespace std;

#include "meta.h"
using namespace meta;

// Deutschland, USA, Schweiz

enum Land {
	Deutschland,
	USA,
	Schweiz
};

ostream& operator<<(ostream& os,const Land& l) {
	switch (l) {
	case Deutschland: os << "Deutschland"; break;
	case USA:         os << "USA"; break;
	case Schweiz:     os << "Schweiz"; break;
	}
	return os;
}

// EUR, USD, CHF

enum Waehrung {
	EUR,
	USD,
	CHF
};

ostream& operator<<(ostream& os, const Waehrung& w) {
	switch (w) {
		case EUR: os << "EUR"; break;
		case USD: os << "USD"; break;
		case CHF: os << "CHF"; break;
	}
	return os;
}

// Umrechnungen ***************************************************************************

template <bool tausch>
struct UnbekannteUmrechnung {
	enum {
		ok
	};
};

template <>
struct UnbekannteUmrechnung<true> {
};

template <Waehrung ausgangswaehrung,Waehrung zielwaehrung>
double umrechnung(const double& betrag) {
	enum {
		ok = UnbekannteUmrechnung<(ausgangswaehrung != zielwaehrung)>::ok
	};
	return betrag;
}

template <>
double umrechnung<EUR,USD>(const double& betrag) {
	return betrag / 1.2;
}

template <>
double umrechnung<USD,EUR>(const double& betrag) {
	return betrag * 1.2;
}

template <>
double umrechnung<EUR,CHF>(const double& betrag) {
	return betrag * 1.6;
}

template <>
double umrechnung<CHF,EUR>(const double& betrag) {
	return betrag / 1.6;
}

template <>
double umrechnung<USD,CHF>(const double& betrag) {
	return betrag * 1.8;
}

template <>
double umrechnung<CHF,USD>(const double& betrag) {
	return betrag / 1.8;
}

// Ermittlung der Landeswaehrung

template <Land land>
struct Landeswaehrung {};

template <>
struct Landeswaehrung<Deutschland> {
	enum {
		Waehrung = EUR
	};
};

template <>
struct Landeswaehrung<USA> {
	enum {
		Waehrung = USD
	};
};

template <>
struct Landeswaehrung<Schweiz> {
	enum {
		Waehrung = CHF
	};
};

// Ueberweisungskomponenten **************************************************************

template <class Generator>
struct Ueberweisung {
	typedef typename Generator::Config Config;
	typedef typename Config::Auftraggeberkonto Auftraggeberkonto;
	typedef typename Config::Empfaengerkonto Empfaengerkonto;
	typedef typename Auftraggeberkonto::Config  ConfigAuftraggeberkonto;
	enum {
		WaehrungAuftraggeberkonto = ConfigAuftraggeberkonto::waehrung,
	};
	static void ausfuehren(Auftraggeberkonto& a,Empfaengerkonto& e,const double& b) {
		cout << "Normale Ueberweisung: " << Waehrung(WaehrungAuftraggeberkonto) << ' ' << b << endl;
		e.stand += b;
		a.stand -= b;
	}
};

template <class ueberweisung>
struct Umtausch: ueberweisung {
	typedef typename ueberweisung::Config       Config;
	typedef typename Config::Auftraggeberkonto  Auftraggeberkonto;
	typedef typename Config::Empfaengerkonto    Empfaengerkonto;
	typedef typename Auftraggeberkonto::Config  ConfigAuftraggeberkonto;
	typedef typename Empfaengerkonto::Config    ConfigEmpfaengerkonto;
	enum {
		WaehrungAuftraggeberkonto = ConfigAuftraggeberkonto::waehrung,
		WaehrungEmpfaengerkonto   = ConfigEmpfaengerkonto::waehrung
	};
	static void ausfuehren(Auftraggeberkonto& a,Empfaengerkonto& e,const double& b) {
		cout << "Umtausch von " << Waehrung(WaehrungAuftraggeberkonto) << ' ' << b << " in " << Waehrung(WaehrungEmpfaengerkonto) << ' ';
		double b2 = umrechnung<Waehrung(WaehrungAuftraggeberkonto),Waehrung(WaehrungEmpfaengerkonto)>(b);
		cout << b2 << endl;
		ueberweisung::ausfuehren(a,e,b2);
	}
};

template <class ueberweisung>
struct Meldung: ueberweisung {
	typedef typename ueberweisung::Config        Config;
	typedef typename Config::Auftraggeberkonto   Auftraggeberkonto;
	typedef typename Config::Empfaengerkonto     Empfaengerkonto;
	typedef typename Auftraggeberkonto::Config   ConfigAuftraggeberkonto;
	typedef typename Empfaengerkonto::Config     ConfigEmpfaengerkonto;
	enum {
		LandAuftraggeberkonto     = ConfigAuftraggeberkonto::land,
		LandEmpfaengerkonto       = ConfigEmpfaengerkonto::land,
		WaehrungAuftraggeberkonto = ConfigAuftraggeberkonto::waehrung,
		WaehrungEmpfaengerkonto   = ConfigEmpfaengerkonto::waehrung
	};
	static void ausfuehren(Auftraggeberkonto& a,Empfaengerkonto& e,const double& b) {
		cout << "Auslandsueberweisung: " << Waehrung(WaehrungAuftraggeberkonto) << ' ' << b << " aus " << Land(LandAuftraggeberkonto) << " auf " << Waehrung(WaehrungEmpfaengerkonto) << "-Konto in " << Land(LandEmpfaengerkonto) << endl;
		ueberweisung::ausfuehren(a,e,b);
	}
};

// Ueberweisungsgenerator ****************************************************************

template <class auftraggeberkonto,class empfaengerkonto>
struct UEBERWEISUNGS_GENERATOR {
	// Kurzname fuer Generator
	typedef UEBERWEISUNGS_GENERATOR<auftraggeberkonto,empfaengerkonto> Generator;
	// DSL analysieren
	typedef typename auftraggeberkonto::Config   ConfigAuftraggeberkonto;
	typedef typename empfaengerkonto::Config     ConfigEmpfaengerkonto;
	enum {
		LandAuftraggeberkonto     = ConfigAuftraggeberkonto::land,
		LandEmpfaengerkonto       = ConfigEmpfaengerkonto::land,
		WaehrungAuftraggeberkonto = ConfigAuftraggeberkonto::waehrung,
		WaehrungEmpfaengerkonto   = ConfigEmpfaengerkonto::waehrung
	};
	// Komponenten montieren
	typedef Ueberweisung<Generator> BasisUeberweisung;
	typedef IF<WaehrungAuftraggeberkonto != WaehrungEmpfaengerkonto,
				Umtausch<BasisUeberweisung>,
				BasisUeberweisung
			  >::RET UeberweisungMitOptionalemUmtausch;
	typedef IF<LandAuftraggeberkonto != LandEmpfaengerkonto,
				Meldung<UeberweisungMitOptionalemUmtausch>,
				UeberweisungMitOptionalemUmtausch
			  >::RET UeberweisungMitOptionalerMeldung;
	// Name und Standardname fuer Endprodukt vergeben
	typedef UeberweisungMitOptionalerMeldung Ueberweisung;
	typedef UeberweisungMitOptionalerMeldung RET;
	// Config erzeugen
	struct Config {
		typedef RET Ueberweisung;
		typedef auftraggeberkonto Auftraggeberkonto;
		typedef empfaengerkonto Empfaengerkonto;
		enum {
			Waehrungsumtausch = WaehrungAuftraggeberkonto != WaehrungEmpfaengerkonto,
			Meldung = LandAuftraggeberkonto != LandEmpfaengerkonto
		};
	};
};


// Konto-Komponenten *********************************************************************

template <class Generator>
struct Basiskonto {
	typedef typename Generator::Config Config;
	typedef typename Config::Konto DiesesKonto;
	enum {
		waehrung = Config::waehrung,
		landeswaehrung = Config::landeswaehrung
	};
	double stand;
	Basiskonto():stand(0.0) {
	}
	template <Waehrung ausgangswaehrung>
		void einzahlen(const double& betrag) {
		stand += umrechnung<ausgangswaehrung,Waehrung(waehrung)>(betrag);
	}
	template <class Empfaengerkonto>
	void ueberweisen(Empfaengerkonto& e,const double& b) {
		UEBERWEISUNGS_GENERATOR<DiesesKonto,Empfaengerkonto>::Ueberweisung::ausfuehren(static_cast<DiesesKonto&>(*this),e,b);
	}
};

template <class konto>
struct Stand: konto {
	typedef typename konto::Config Config;
	void abfrage() const {
		cout << "Kontostand: " << Waehrung(waehrung) << ' ' << stand << endl;
	}
};

template <class konto>
struct StandFremdwaehrung: Stand<konto> {
	typedef typename konto::Config Config;
	enum {
		waehrung       = Config::waehrung,
		landeswaehrung = Config::landeswaehrung
	};
	void abfrage() const {
		Stand<konto>::abfrage();
		cout << "Entspricht: " << Waehrung(landeswaehrung) << ' ' << umrechnung<Waehrung(waehrung),Waehrung(landeswaehrung)>(stand) << endl;
	}
};

// Konto-DSL *****************************************************************************

struct deutsch {
	enum {
		land          = Deutschland,
		waehrung      = EUR,
		fremdwaehrung = false,
		pruefung      = true
	};
};

struct schweizerisch {
	enum {
		land          = Schweiz,
		waehrung      = CHF,
		fremdwaehrung = false,
		pruefung      = true
	};
};

struct us_amerikanisch {
	enum {
		land          = USA,
		waehrung      = USD,
		fremdwaehrung = false,
		pruefung      = true
	};
};

template <Land land_,Waehrung waehrung_>
struct fremd {
	enum {
		land          = land_,
		waehrung      = waehrung_,
		fremdwaehrung = true,
		pruefung      = (land_ == Deutschland && waehrung_ != EUR) || (land_ == USA && waehrung_ != USD) || (land_ == Schweiz && waehrung_ != CHF)
	};
};

// Konto-Generator ***********************************************************************

// Fehlerpruefung
template <bool pruefung>
struct UnzulaessigesFremdwaehrungskonto {
	enum {
		ok
	};
};

template <>
struct UnzulaessigesFremdwaehrungskonto<false> {
};

template <class Art = deutsch>
struct KONTO_GENERATOR {
	// Kurzname fuer Generator
	typedef KONTO_GENERATOR<Art> Generator;
	// DSL analysieren
	enum {
		ok = UnzulaessigesFremdwaehrungskonto<Art::pruefung>::ok,
		fremdwaehrung = Art::fremdwaehrung,
	};
	// Komponenten montieren
	typedef Basiskonto<Generator> KontoBasis;
	typedef IF <fremdwaehrung,
				StandFremdwaehrung<KontoBasis>,
				Stand<KontoBasis>
			   >::RET KontoMitStand;
	// Name und Standardname fuer Endprodukt vergeben
	typedef KontoMitStand Konto;
	typedef KontoMitStand RET;
	// Config erzeugen
	struct Config {
		typedef RET Konto;
		enum {
			waehrung       = Art::waehrung,
			land           = Art::land,
			fremdwaehrung  = Art::fremdwaehrung,
			landeswaehrung = Landeswaehrung<Land(land)>::Waehrung
		};
	};
};

void main() {
	KONTO_GENERATOR<deutsch>::Konto k01;
	k01.einzahlen<EUR>(1000.0);
	k01.abfrage();
	KONTO_GENERATOR<us_amerikanisch>::Konto k02;
	k02.einzahlen<USD>(1000);
	k02.abfrage();
	KONTO_GENERATOR<schweizerisch>::Konto k03;
	k03.einzahlen<CHF>(1000);
	k02.abfrage();
	KONTO_GENERATOR<fremd<Deutschland,USD> >::Konto k04;
	k04.einzahlen<USD>(1000);
	k04.abfrage();
	KONTO_GENERATOR<>::RET k05;
	k05.einzahlen<EUR>(1000);
	k05.abfrage();
	KONTO_GENERATOR<fremd<USA,EUR> >::RET k06;
	k06.einzahlen<EUR>(1000);
	k06.abfrage();
	//KONTO_GENERATOR<fremd<USA,USD> >::RET k07; // Das muss einen Fehler geben!
	k01.ueberweisen(k01,10.0);
	k01.ueberweisen(k05,100.0);
	k05.ueberweisen(k06,100.0);
	k06.ueberweisen(k01,100.0);
	k06.ueberweisen(k02,100.0);

}
