/**
 * Copyright @ 2000 Peter Robach (pr@webapp.de)
 * 
 * Source is only for non commercial and coaching usage.
 * 
 * Not Warranty to use it.
 */
package de.ix.jspTutorial.database ;

import java.sql.*;
import java.util.*;


/**
  * Verbindung zur Datenbank.
  *
  *
  * @author Peter Rossbach (<a href="mailto://pr@webapp.de">pr@webapp.de</a>)
  * @version $Id:$
 */
public class DatabaseConnection {

   /**
    * Version  des Source
    */
  public static String vcid = "$Id:$";

	/** Driver object name to use for JDBC connection */
	private String myDatabaseDriver ;

	/** URL to use when connecting via the given driver */
	private String myConnectionString ;

	/** Description of this connection */
	private String myDescription;

	/** Is this connection object actually connected to the database yet? */
	private boolean myIsConnected = false;

	/** JDBC connection object */
	private Connection myConnection;

	/** Parameter to connect to the database */
	private Properties myProperties ;

	/** When was this connection last used? */
	private long myLastUsed ;

	// no default constructor
	private DatabaseConnection() {}
	
	/**
	 */
	public DatabaseConnection(
		String aDatabaseDriver, 
		String aConnectionString)
			throws DatabaseException
	{
 		this(aDatabaseDriver,aConnectionString,"no description") ;
	}
		
	/**
	 */
	public DatabaseConnection(
		String aDatabaseDriver, 
		String aConnectionString,
		String aDescription )
			throws DatabaseException {
			
		myDatabaseDriver = aDatabaseDriver;
		myConnectionString = aConnectionString;
		myDescription = aDescription ;
		try {
			Class.forName(myDatabaseDriver);
		} catch(Exception se) {
			throw new DatabaseException("Can't load Database Driver " + myDatabaseDriver);
		}
   myConnection = null ;
   myIsConnected = false ;
		used();
	}

	public boolean isConnected() { return myIsConnected ; }
	
	/**
	 */
	public long getLastUsed() {
		return myLastUsed;
	}


	/**
	 */
	private void used() {
		myLastUsed = System.currentTimeMillis();
	} 


	/**
	 */
	public synchronized void setDescription(String newDescription) {
		if (newDescription != null)
			myDescription = newDescription;
		used();
	}

	/**
	 */
	public String getDescription() {
		return myDescription;
	}

	/**
	 */
	public String getConnectionString() {
		return myConnectionString;
	}

	/**
	 */
	public String getLogin() {
		return myProperties.getProperty("user") ;
	}


	/**
	 */
	public String getPassword() {
		return myProperties.getProperty("password");
	} 

	/**
	 */
	public String getDatabaseDriver() {
		return myDatabaseDriver;
	} 

	/**
	 */
	public synchronized void 
		connect(String aLogin, String aPassword)
		throws DatabaseException {
		
		Properties theProperties = new Properties();
		theProperties.put("user", aLogin);
		theProperties.put("password", aPassword);
		connect(theProperties) ;
	}

	/**
	 */
	public synchronized void 
		connect(Properties aProperties)
		throws DatabaseException {
		
		if(myConnection != null) {
			try {
				disconnect() ;
			} catch (DatabaseException e) {}
		}
		try {
			  myProperties = aProperties ;
				myConnection =
					DriverManager.getConnection(myConnectionString,	myProperties);
				if (myConnection == null) {
					throw new DatabaseException (
						"Can't get connection to database via driver '" + myDatabaseDriver
						+ "' and server " +  myConnectionString 
						+ "JDBC returned a null connection. (" + myDescription 
						+ ")");
				}
		
				setAutoCommit(false);
						
		} catch(SQLException se) {
			if(myConnection != null)
				try {
						myConnection.close() ;
				} catch (SQLException sql) {}
			myConnection = null ;
			throw new DatabaseException(
				"Can't get connection to database via driver '" + myDatabaseDriver
				+ "' and server " +  myConnectionString 
				+ "' (" + myDescription + ")", se);
		}
		myIsConnected = true;
		used();
	}

	/**
	 */
	public synchronized void disconnect() throws DatabaseException {
		try {
			if(myConnection != null)
					myConnection.close() ;
		} catch (SQLException sql)
		{
					throw new DatabaseException (
						"Can't disconnect to database via driver '" + myDatabaseDriver
						+ "' and server " +  myConnectionString 
						+ "JDBC returned a null connection. (" + myDescription 
						+ ")", sql);
		} finally {
			myConnection = null ;
			myIsConnected = false ;
			used();
		}
	
	}
	
	/**
 	  */
	public void finalize() {
		if(isConnected())
			try {
				disconnect();
			}
			catch(DatabaseException de) {
				 de.printStackTrace() ;
			}
	}

   public synchronized boolean begin() throws DatabaseException
   {
      return isConnected();
   }

	/**
	 */
	public synchronized void commit() 
		throws DatabaseException
 {
		if(!isConnected())
			 return ;

		try {
			myConnection.commit();
		} catch(SQLException se) {
			throw new DatabaseException(
				"Could not commit (" + myDescription + ")", 
				se);
		}
		used();
	}


	/**
	 */
	public synchronized void rollback() throws DatabaseException {
		if(!isConnected())
			 return ;
		try {
			myConnection.rollback();
		} catch(SQLException se) {
			throw new DatabaseException(
				"Could not rollback (" + myDescription + ")", se);
		}
		used();
	} 

	/**
 	  */
	public String nativeSQL(String sql) throws SQLException {
		if(isConnected())
			return myConnection.nativeSQL(sql);
		else
		  return null ;
	}

	/**
 	  */
	public Statement createStatement() throws SQLException {
		if(isConnected())
			return myConnection.createStatement();
		else
		  return null ;
	}
	
	/**
 	  */
	public CallableStatement prepareCall(String sql) throws SQLException {
		if(isConnected())
			return myConnection.prepareCall(sql);
		else
		  return null ;
	}

	/**
 	  */
	public PreparedStatement prepareStatement(String sql) throws SQLException {
		if(isConnected())
			return myConnection.prepareStatement(sql);
		else
		  return null ;
	}

	/**
 	  */
	public synchronized void  setAutoCommit(boolean autoCommit) throws SQLException {
		if(isConnected())
			myConnection.setAutoCommit(autoCommit);
	}

	public synchronized boolean getAutoCommit() throws SQLException {
		if(isConnected())
			return myConnection.getAutoCommit();
		else
		  return false ;
	}

} // end of class


//
// History:
//
// $Log:$
//
//
