package de.ebuchner.jv;

import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.io.File;
import java.util.List;
import java.util.zip.ZipException;

/**
 * Adapter fuer Drop-Events des Baums
 */
public class TreeDTAdapter implements DropTargetListener {

  // Welche Drop-Actions unterstuetzt dieser Adapter?
  final int dropActions = DnDConstants.ACTION_COPY_OR_MOVE;
  
  /**
   * Konstruktor
   * @param tree Baum, fuer den der Adapter zustaendig ist
   */
  public TreeDTAdapter(JvTree tree) {

    // Tree als DropTarget fuer JarFiles
    new DropTarget(tree, dropActions, this);
  }

  /**
   * Anwender zieht etwas in den Baum hinein
   */
  public void dragEnter(DropTargetDragEvent dtde) {
    checkDrag(dtde);
  }

  /**
   * Anwender zieht etwas ueber den Baum
   */
  public void dragOver(DropTargetDragEvent dtde) {
    checkDrag(dtde);
  }

  /**
   * Anwender aendert die Aktion
   */
  public void dropActionChanged(DropTargetDragEvent dtde) {
    checkDrag(dtde);
  }

  /**
   * Anwender verlaesst den Baum wieder
   */
  public void dragExit(DropTargetEvent dte) {
  }

  /**
   * Anwender laesst etwas auf dem Baum fallen
   */
  public void drop(DropTargetDropEvent dtde) {
    // GUI-Parent
    final JvTree tree = (JvTree)dtde.getDropTargetContext().getComponent();

    // Nicht innerhalb des Baums
    if(dtde.isLocalTransfer()) {
      System.err.println("Local transfer is not supported");
      dtde.rejectDrop();
      return;
    }

    try {
      // Vor dem Datenzugriff
      dtde.acceptDrop(dtde.getDropAction());

      // Datenzugriff
      Transferable t = dtde.getTransferable();
      List files = (List)t.getTransferData(DataFlavor.javaFileListFlavor);

      // Nur eine Datei
      if(files.size()!=1) {
        System.err.println("Only one archive may be opened");
        dtde.dropComplete(false);
        return;
      }

      File f = (File)files.get(0);
      tree.show(f);
      dtde.dropComplete(true);
      
    } catch(final ZipException ze) {
      System.err.println("Not a valid JAR file");
      dtde.dropComplete(false);
    } catch(Exception e) {
      e.printStackTrace();
      dtde.dropComplete(false);
    }
  }

  /**
   * Hilfsroutine: Prueft, ob der Baum das Datenformat und die Aktion
   * unterstuetzt
   */
  private void checkDrag(DropTargetDragEvent dtde) {
    // Dateiliste?
    boolean accept = dtde.isDataFlavorSupported(DataFlavor.javaFileListFlavor);
    
    // Copy oder Move?
    int dropAction = dtde.getDropAction();

    if(accept && (dropAction & dropActions)>0)
      dtde.acceptDrag(dropAction);
    else
      dtde.rejectDrag();
  }
}
