package de.ebuchner.jv;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import javax.swing.*;
import javax.swing.tree.*;

import de.ebuchner.jar.JarUtils;

/**
 * Zeigt den Inhalt eines Java Archivs als Baum an
 */
public class JvTree extends JTree {

  JarFile jarFile=null;

  /**
   * Erzeugt einen leeren Baum
   */
  public JvTree() {
    super(new JvTreeModel());

    setRootVisible(false);
    setShowsRootHandles(true);
  }

  /**
   * Erzeugt einen Baum aus dem Inhalt des JarFiles
   */
  public JvTree(JarFile jarFile) {
    this();
    show(jarFile);
  }

  /**
   * Laedt ein Jar-File
   */
  public void show(File file) throws IOException {
    show(new JarFile(file));
  }

  /**
   * Laedt ein Jar-File
   */
  public void show(JarFile jarFile) {
    this.jarFile=jarFile;
    JvTreeModel treeModel = (JvTreeModel)this.getModel();
    treeModel.show(jarFile);
  }

  /**
   * Liefert das geoeffnete Jar-File
   */
  public JarFile getJarFile() {
    return jarFile;
  }

  /**
   * Welchem Eintrag im Jar-File entspricht die aktuelle Selektion?
   * @return Eintrag oder null
   */
  public JarEntry getSelectedEntry() {
    TreePath path = getSelectionPath();
    if(path==null)
      return null;
    JvTreeNode tn = (JvTreeNode)path.getLastPathComponent();
    return tn.entry();
  }

  /**
   * Ermittelt alle markierten Archiv-Elemente
   */
  public JarEntry [] getSelectionEntries() {
    TreePath [] selPaths = getSelectionPaths();
    if(selPaths==null)
      return null;

    JarEntry [] selEntries = new JarEntry[selPaths.length];
    for(int i=0; i<selPaths.length; i++)
      selEntries[i] = entryOf(selPaths[i]);
    return selEntries;
  }

  private JarEntry entryOf(TreePath path) {
    JarEntry entry = ((JvTreeNode)path.getLastPathComponent()).entry();
    // Kein Element, das sich ziehen liesse
    if(entry==null || entry.isDirectory())
      return null;

    return entry;
  }
}

/**
 * TreeDaten auf Basis eines JarFiles
 */
class JvTreeModel extends DefaultTreeModel {

  static JvTreeNode root = new JvTreeNode("(no file loaded)");

  /**
   * Ein leeres Treemodel
   */
  public JvTreeModel() {
    this(null);
  }

  /**
   * aus einer Datei
   */
  public JvTreeModel(JarFile file) {
    super(root);
    if(file!=null)
      show(file);
  }

  /**
   * Laedt eine neue Datei
   */
  public void show(JarFile file) {
    root.setUserObject(file.getName());

    // alte Kinder entfernen
    while(getChildCount(root)>0)
      removeNodeFromParent((DefaultMutableTreeNode)getChild(root, 0));

    // Neue Kinder einfuegen
    for(Enumeration en = file.entries(); en.hasMoreElements(); ) {
      JarEntry entry = (JarEntry)en.nextElement();
      addEntry(root, entry);
    }
    // Model benachrichtigen
    super.nodeStructureChanged(root);
  }

  /**
   * Fuegt ein Entry an der richtigen Stelle im Baum ein
   */
  private void addEntry(DefaultMutableTreeNode root, JarEntry entry) {
    Enumeration enPath = JarUtils.splitName(entry);
    if(enPath.hasMoreElements())
      addEntry(root, entry, enPath);
  }

  private void addEntry(DefaultMutableTreeNode parent, JarEntry entry, Enumeration enPath) {
    String childName = (String)enPath.nextElement();
    DefaultMutableTreeNode child = findNode(parent, childName);

    // Verzeichnis?
    if(enPath.hasMoreElements()) {
      if(child==null)
        child = new JvTreeNode(childName);
      parent.add(child);
      addEntry(child, entry, enPath);
    } else
      parent.add(new JvTreeNode(entry, childName));
  }

  private DefaultMutableTreeNode findNode(DefaultMutableTreeNode parent, String name) {
    for(Enumeration en = parent.children(); en.hasMoreElements(); ) {
      DefaultMutableTreeNode treeNode = (DefaultMutableTreeNode)en.nextElement();
      if(treeNode.getUserObject().equals(name))
        return treeNode;
    }
    return null;
  }
}
