package de.ebuchner.jv;

import java.awt.dnd.*;
import java.io.*;
import java.util.jar.*;
import javax.swing.tree.*;

import de.ebuchner.datatransfer.FileSelection;
import de.ebuchner.dnd.DragSourceAdapter;
import de.ebuchner.jar.JarUtils;

/**
 * Dieser DragGestureRecognizer wird informiert, wenn der
 * Benutzer ein Element des Jar-Files aus der Baumansicht
 * zu ziehen beginnt.
 * <p>
 * Wird eine Drag-Geste erkannt, erzeugt dieser Adapter
 * das Datenobjekt, das dem DnD-System bergeben wird.
 */
public class TreeDGAdapter implements DragGestureListener {

  // Welche DragActions unterstuetzt dieser Adapter
  final int dragActions = DnDConstants.ACTION_COPY_OR_MOVE;

  /**
   * Konstruktor
   * @param tree Baum, fuer den der Adapter zustaendig ist
   */
  public TreeDGAdapter(JvTree tree) {

    // Initialize a DragGestureRecognizer
    DragSource dragSource = DragSource.getDefaultDragSource();
    dragSource.createDefaultDragGestureRecognizer(
      tree,
      dragActions,
      this
    );
  }

  /**
   * Eine Drag-Geste wurde erkannt
   */
  public void dragGestureRecognized(DragGestureEvent dge) {
    // Nur bestimmte Aktionen zulassen
    if((dge.getDragAction() & dragActions) == 0)
      return;

    // Alle markierten Objekte des Baums einsammeln
    JvTree   tree        = (JvTree)dge.getComponent();
    JarEntry selEntries [] = tree.getSelectionEntries();
    if(selEntries==null) // nichts selektiert
      return;

    JarFile jarFile = tree.getJarFile();

    try {
      // Transferable
      FileSelection fsel = new FileSelection();

      // Markierte Elemente auspacken und Dateinamen in
      // Transferable aufnehmen
      for(int i=0; i<selEntries.length; i++) {
        JarEntry entry = selEntries[i];
        if(entry==null)
          continue;
        fsel.add(JarUtils.extract(jarFile, entry));
      }

      // Startet den Drag-Vorgang
      dge.startDrag(
        DragSource.DefaultCopyNoDrop,
        fsel,
        new DragSourceAdapter()
      );

    } catch(IOException ioe) {
      ioe.printStackTrace();
    }
  }
}

