package com.example.spring3cities.controller;

import java.io.IOException;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.validation.BindingResult;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.servlet.ModelAndView;
import org.apache.lucene.queryParser.ParseException;

import com.example.spring3cities.dao.CityDao;
import com.example.spring3cities.model.City;
import com.example.spring3cities.service.CityService;

@Controller
@RequestMapping
public class CityController {
	private static final Logger logger = LoggerFactory.getLogger(CityController.class);
	
	@Autowired
	private CityDao cityDao;
	
	@Autowired
	private CityService cityService;
	
	// 50 größte Städte auslesen
	@RequestMapping(method=RequestMethod.GET, value="/")
	public ModelAndView showCities() {
		
		List<City> cities = cityDao.getCities(50);
		logger.info("Insgesamt sind " + cities.size() + " Städte vorhanden");
		logger.info("Name der größten Stadt ist " + cities.get(0).getName());
		
		return cityService.buildListView(cities);
	}
	
	// Neue Stadt anlegen
	@RequestMapping(method=RequestMethod.POST, value="/city/add")
	public String addCity(@ModelAttribute("city") City city, BindingResult result) {
		logger.info("Lege neue Stadt " + city.getName() + " an");
		
		cityDao.addCity(city);
		
		return "redirect:/";
	}
	
	// Löschen per URL und via AJAX
	@RequestMapping(method=RequestMethod.GET, value="/city/delete/{id}")
	public String deleteCity(@PathVariable("id") int id) {
		logger.info("Lösche Datensatz " + id);
		
		cityDao.deleteCityById(id);
		
		return "redirect:/";
	}
	
	// Index neu erstellen
	@RequestMapping(method=RequestMethod.GET, value="/createsearchindex")
	public String createSearchIndex() throws InterruptedException {
		logger.info("Schreibe den Lucene-Index neu");
		
		cityDao.createSearchIndex();
		
		return "redirect:/";
	}
	
	// Normale Suchfunktion GET
	@RequestMapping(method=RequestMethod.GET, value="/city/search")
	public ModelAndView getSearchResult(@RequestParam("term") String term) {		
		List<City> cities = cityDao.searchCity(term);
		logger.info("Es wurden " + cities.size() + " Städte gefunden");

		return cityService.buildListView(cities);
	}
	
	// Suchfunktion Autocomplete POST
	@RequestMapping(method=RequestMethod.POST, value="/city/search")
	public ResponseEntity<String> getAutocompleteResult(@RequestParam("term") String term) throws IOException, ParseException {		
		
		// Neue Response-Header nötig, da Ergebnis sonst ISO-codiert
		HttpHeaders responseHeaders = new HttpHeaders();
		responseHeaders.add("Content-Type", "text/html; charset=utf-8");
		
		List<String> cities = cityDao.searchCityName(term);
		
		if(cities.size() == 0) {
			logger.info("Kein Treffer in Datenbank - versuche Autokorrektur");
			cities = cityDao.searchSimilarCities(term);
		} else {
			logger.info("Es wurden " + cities.size() + " Städte gefunden - Autocomplete");
		}
		String json = cityService.convertJson(cities);
		return new ResponseEntity<String>(json, responseHeaders, HttpStatus.CREATED);
	}
}

