#include <stdio.h>
#include <stdlib.h>
#include <QtGui/QWidget>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QLabel>
#include <QtGui/QGroupBox>
#include <QtGui/QPushButton>
#include <QtGui/QScrollArea>
#include <QtCore/QPoint>
#include <QtCore/QTimer>
#include "data.h"
#include "hiresdialog.h"

HiresDialog::HiresDialog(Data *p_data, QWidget *parent, Qt::WindowFlags f) : QWidget(parent, f) {

  data = p_data;
  hiresWidth = data->getHiresWidth();
  hiresHeight = data->getHiresHeight();
  QVBoxLayout *mainLayout = new QVBoxLayout;

  scroll = new QScrollArea;
  hiresView = new HiresView(data, this, data->getHiresWidth(), data->getHiresHeight());
  scroll->setWidget(hiresView);
  QObject::connect(hiresView, SIGNAL(moveView(QPoint)), this, SLOT(moveHiresView(QPoint)));
  mainLayout->addWidget(scroll);

  QHBoxLayout *controlsLayout = new QHBoxLayout;
  QWidget *controlsFrame = new QWidget;
  controlsFrame->setLayout(controlsLayout);

  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);

  controlsLayout->addWidget(new QLabel("Fractal Type:"));
  fractalSelector = new QComboBox();
  fractalSelector->insertItem(0, "Mandelbrot Set");
  fractalSelector->insertItem(1, "Julia Set");
  QObject::connect(fractalSelector, SIGNAL(currentIndexChanged(int)), this, SLOT(setFractal(int)));
  controlsLayout->addWidget(fractalSelector);
  
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);

  controlsLayout->addWidget(new QLabel("Width:"));
  resolutionSpin[0] = new QSpinBox();
  resolutionSpin[0]->setMinimum(640);
  resolutionSpin[0]->setMaximum(MAX_HIRES_WIDTH);
  resolutionSpin[0]->setKeyboardTracking(false);
  resolutionSpin[0]->setValue(hiresWidth);
  controlsLayout->addWidget(resolutionSpin[0]);
  QObject::connect(resolutionSpin[0], SIGNAL(valueChanged(int)), this, SLOT(setWidth(int)));

  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);

  controlsLayout->addWidget(new QLabel("Height:"));
  resolutionSpin[1] = new QSpinBox();
  resolutionSpin[1]->setMinimum(480);
  resolutionSpin[1]->setMaximum(MAX_HIRES_HEIGHT);
  resolutionSpin[1]->setKeyboardTracking(false);
  resolutionSpin[1]->setValue(hiresHeight);
  controlsLayout->addWidget(resolutionSpin[1]);
  QObject::connect(resolutionSpin[1], SIGNAL(valueChanged(int)), this, SLOT(setHeight(int)));

  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  
  statusLabel = new QLabel("Status:                       ");
  controlsLayout->addWidget(statusLabel);
  
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);
  controlsLayout->addWidget(new QWidget);

  mainLayout->addWidget(controlsFrame);

  QHBoxLayout *buttonLayout = new QHBoxLayout;
  QWidget *buttonFrame = new QWidget;
  buttonFrame->setLayout(buttonLayout);
  closeButton = new QPushButton("Close");
  QObject::connect(closeButton, SIGNAL(clicked()), this, SLOT(hide()));
  buttonLayout->addWidget(new QWidget);
  buttonLayout->addWidget(closeButton);
  buttonLayout->addWidget(new QWidget);
  refreshButton = new QPushButton("Refresh");
  QObject::connect(refreshButton, SIGNAL(clicked()), this, SLOT(refreshClicked()));
  buttonLayout->addWidget(new QWidget);
  buttonLayout->addWidget(refreshButton);
  buttonLayout->addWidget(new QWidget);
  mainLayout->addWidget(buttonFrame);
  
  setLayout(mainLayout);
  
  timer = new QTimer(this);
  timer->setSingleShot(true);
  timer->setInterval(100);
  QObject::connect(timer, SIGNAL(timeout()), this, SLOT(updateStatus()));
  timer->start();
  sx = data->getHiresWidth()>>1;
  sy = data->getHiresHeight()>>1;
  scroll->ensureVisible(sx, sy);
}

HiresDialog::~HiresDialog() {

}

void HiresDialog::setWidth(int value) {

  hiresWidth = value;
}

void HiresDialog::setHeight(int value) {

  hiresHeight = value;
}

void HiresDialog::setFractal(int value) {

  data->setHiresJulia(value);
}

void HiresDialog::refreshClicked() {

  if (!data->getImageBusy(2)) {
    sx = data->getHiresWidth()>>1;
    sy = data->getHiresHeight()>>1;
    scroll->ensureVisible(sx, sy);
    timer->start();
    data->setHiresWidth(hiresWidth);
    data->setHiresHeight(hiresHeight);
    hiresView->setMinimumSize(data->getHiresWidth(), data->getHiresHeight());
    hiresView->setMaximumSize(data->getHiresWidth(), data->getHiresHeight());
    emit refreshHires();
  }  
}

void HiresDialog::refreshHiresView() {

  hiresView->repaint();
}

void HiresDialog::updateStatus() {

   if (data->getChunkCount() >= 0) {
     statusLabel->setText("Status: Computing [" + QString::number(100 - (int)(100.0 * (long double)data->getChunkCount() * (long double)data->getChunkLen() / (long double)data->getHiresHeight())) + "%]");
     //statusLabel->setText("chunkCount: " + QString::number(data->getChunkCount())
     //                     + "  chunkLen: " + QString::number(data->getChunkLen()));
     timer->start();
   } else {
     statusLabel->setText("Status: Ready                    ");
   }   
}

void HiresDialog::moveHiresView(QPoint dp) {

//  printf("%d %d   %d %d\n", dp.x(), dp.y(), sx, sy);
  if (abs(dp.x() < 30)) {
    sx += dp.x()<<1;
  }  
  if (abs(dp.y() < 30)) {
    sy += dp.y()<<1;
  }  
  if (sx < 0) sx = 0;
  if (sx > data->getHiresWidth()) sx = data->getHiresWidth();
  if (sy < 0) sy = 0;
  if (sy > data->getHiresHeight()) sy = data->getHiresHeight();
  scroll->ensureVisible(sx, sy);  
}
