#include <math.h>
#include <QtGui/QRgb>
#include "data.h"

Data::Data(QObject *parent) : QObject(parent) {

  int i, tmp;
  
  mandelZoom = 1.0; 
  juliaZoom = 1.0;
  imageWidth = WIDTH;
  imageHeight = HEIGHT;
  hiresWidth = 1920;
  hiresHeight = 1440;
  hiresJuliaFlag = 0;
  antiAliasing = 0;
  chunkCount = 0;
  chunkLen = CHUNK_LEN;
  c_re = 0.4;
  c_im = 0.2;
  xScale[0] = 4.0 * mandelZoom / (long double)imageWidth;
  xScale[1] = 8.0 * juliaZoom / (long double)imageWidth;
  yScale[0] = 4.0 * mandelZoom / (long double)imageHeight;
  yScale[1] = 8.0 * juliaZoom / (long double)imageHeight;
  xOfs[0] = -0.64 * imageWidth;
  xOfs[1] = -imageWidth>>1;
  yOfs[0] = -imageHeight>>1; 
  yOfs[1] = -imageHeight>>1; 
  threadCount = QThread::idealThreadCount();
  for (i = 0; i < 3; i++) {
    threadsReady[i] = 0;
    imageBusyFlag[i] = true;
  }  
  for (i = 0; i < 2; i++) {
    maxIterations[i] = 1000;
    maxAbsoluteValue[i] = 100;
    max[i] = 0;
    tiles[i] = new QPoint[(imageWidth / TILE_LEN) * (imageHeight / TILE_LEN)];
    showZoomRect[i] = false;
    tmpZoom[i] = 1.0;
    paletteScale[i] = 0.5;
    colorCount[i] = 0;
    addColor(i, 0, QColor(0, 0, 0), tmp);
    addColor(i, 16, QColor(0, 50, 100), tmp);
    addColor(i, 32, QColor(150, 200, 255), tmp);
    addColor(i, 64, QColor(0, 150, 0), tmp);
    addColor(i, 128, QColor(0, 200, 100), tmp);
    addColor(i, 256, QColor(200, 200, 0), tmp);
    addColor(i, 512, QColor(200, 0, 0), tmp);
    addColor(i, maxIterations[i], QColor(0, 100, 0), tmp);
  }  
  mandelData = (int *)malloc(imageWidth * imageHeight * sizeof(int));
  juliaData = (int *)malloc(imageWidth * imageHeight * sizeof(int));
  mandelImage = new QImage(imageWidth, imageHeight, QImage::Format_RGB32);
  juliaImage = new QImage(imageWidth, imageHeight, QImage::Format_RGB32);
  hiresImage = new QImage(hiresWidth, hiresHeight, QImage::Format_RGB32);
  rescalePalette = false;
  normalize = false;
  shutdownFlag = false;
}

Data::~Data() {

  free(mandelData);
  free(juliaData);  
  delete mandelImage;
  delete juliaImage;
  delete hiresImage;
}

void Data::setThreadCount(int value) {

  threadCount = value;
}
  
int Data::getThreadCount() {
  
  return(threadCount);
}
    
int Data::getThreadsReady(int index) {
      
  return(threadsReady[index]);
}
        
void Data::threadReady(int index) {
        
  threadsReady[index]++; 
}
          
void Data::imageBusy(int index) {
          
  imageBusyFlag[index] = true;
}

bool Data::getImageBusy(int index) { 
          
  return(imageBusyFlag[index]);       
}

void Data::imageReady(int index) {
          
  imageBusyFlag[index] = false;          
}

void Data::calcTiles(int index, int dx, int dy) {

  int i1, i2, left, right, up, down;

  tileMutex[index].lock();  
  imageBusy(index);
  threadsReady[index] = 0;
  tileCount[index] = 0;
  if (dx >= 0) {
    left = 0;
    right = (int)ceil((long double)dx / (long double)TILE_LEN);
  } else {
    left = (int)floor((long double)(imageWidth + dx) / (long double)TILE_LEN);
    right = imageWidth / TILE_LEN;
  }
  if (dy >= 0) {
    up = 0;
    down = (int)ceil((long double)dy / (long double)TILE_LEN);
  } else {
    up = (int)floor((long double)(imageHeight + dy) / (long double)TILE_LEN);
    down = imageHeight / TILE_LEN;
  }
  if (left < 0) left = 0;  
  if (right < 0) right = 0;
  if (up < 0) up = 0;
  if (down < 0) down = 0;
  if (left > imageWidth / TILE_LEN) left = imageWidth / TILE_LEN;  
  if (right > imageWidth / TILE_LEN) right = imageWidth / TILE_LEN;
  if (up > imageHeight / TILE_LEN) up = imageHeight / TILE_LEN;
  if (down > imageHeight / TILE_LEN) down = imageHeight / TILE_LEN;
  if (up) {
    for (i1 = left; i1 < right; i1++) {
      for (i2 = 0; i2 < up; i2++) {
        tiles[index][tileCount[index]].setX(i1 * TILE_LEN);
        tiles[index][tileCount[index]].setY(i2 * TILE_LEN);
        tileCount[index]++;
      }
    }
    for (i1 = 0; i1 < imageWidth / TILE_LEN; i1++) {
      for (i2 = up; i2 < down; i2++) {
        tiles[index][tileCount[index]].setX(i1 * TILE_LEN);
        tiles[index][tileCount[index]].setY(i2 * TILE_LEN);
        tileCount[index]++;
      }
    }
  } else {
    for (i1 = 0; i1 < imageWidth / TILE_LEN; i1++) {
      for (i2 = 0; i2 < down; i2++) {
        tiles[index][tileCount[index]].setX(i1 * TILE_LEN);
        tiles[index][tileCount[index]].setY(i2 * TILE_LEN);
        tileCount[index]++;
      }
    }
    for (i1 = left; i1 < right; i1++) {
      for (i2 = down; i2 < imageHeight / TILE_LEN; i2++) {
        tiles[index][tileCount[index]].setX(i1 * TILE_LEN);
        tiles[index][tileCount[index]].setY(i2 * TILE_LEN);
        tileCount[index]++;
      }
    }
  } 
  tileMutex[index].unlock();
}

void Data::calcTiles(int index) {

  int i1, i2;

  tileMutex[index].lock();  
  imageBusy(index);
  threadsReady[index] = 0;
  tileCount[index] = 0;
  for (i1 = 0; i1 < getImageWidth() / TILE_LEN; i1++) {
    for (i2 = 0; i2 < getImageHeight() / TILE_LEN; i2++) {
      tiles[index][tileCount[index]].setX(i1 * TILE_LEN);
      tiles[index][tileCount[index]].setY(i2 * TILE_LEN);
      tileCount[index]++;
    }
  }
  tileMutex[index].unlock();
}

int Data::getTileCount(int index) {

  return(tileCount[index]);
}
        
bool Data::getNextTile(int index, QPoint &tile) {

  tileMutex[index].lock();
  if (tileCount[index]) {
    tile = tiles[index][tileCount[index] - 1];
    tileCount[index]--;
    tileMutex[index].unlock();
    return(true);
  } else {  
    tileMutex[index].unlock();
    return(false);
  }
}

void Data::calcChunks(int len) { 

  imageBusy(2);
  threadsReady[2] = 0;
  chunkLen = len;
  chunkCount = (int)ceil((long double)hiresHeight / (long double)len);
}

int Data::getChunkCount() { 

  return(chunkCount);
}  

int Data::getChunkLen() { 

  return(chunkLen);
}  

bool Data::getNextChunk(int &pos, int &len) {

  bool ok;

  chunkMutex.lock();
  len = chunkLen;
  chunkCount--;
  if ((chunkCount * chunkLen) > (hiresHeight - chunkLen)) {
    len = hiresHeight - chunkCount * chunkLen;
  }
  ok = chunkCount >= 0;
  pos = chunkCount * chunkLen;
  chunkMutex.unlock();
  return(ok);
}

void Data::setHiresWidth(int value) { 

  QImage *oldPointer;
  
  hiresWidth = value;
  oldPointer = hiresImage;
  hiresImage = new QImage(hiresWidth, hiresHeight, QImage::Format_RGB32);
  delete oldPointer;
}

int Data::getHiresWidth() { 

  return(hiresWidth);
}

void Data::setHiresHeight(int value) { 

  QImage *oldPointer;
  oldPointer = hiresImage;

  hiresHeight = value;
  hiresImage = new QImage(hiresWidth, hiresHeight, QImage::Format_RGB32);
  delete oldPointer;
}

int Data::getHiresHeight() { 

  return(hiresHeight);
}

void Data::setHiresJulia(bool value) {

  hiresJuliaFlag = value;
}

bool Data::hiresJulia() {

  return(hiresJuliaFlag);
}

void Data::setMandelZoomRaw(long double value) {

  mandelZoom = value;
  xScale[0] = 4.0 * mandelZoom / (long double)imageWidth;
  yScale[0] = 4.0 * mandelZoom / (long double)imageHeight;
}

void Data::setJuliaZoomRaw(long double value) {

  juliaZoom = value;
  xScale[1] = 8.0 * juliaZoom / (long double)imageWidth;
  yScale[1] = 8.0 * juliaZoom / (long double)imageHeight;
}

void Data::setMandelZoom(long double value) {

  xOfs[0] = ((long double)(imageWidth>>1) * (mandelZoom - value) + xOfs[0] * mandelZoom) / value;
  yOfs[0] = ((long double)(imageHeight>>1) * (mandelZoom - value) + yOfs[0] * mandelZoom) / value;
  mandelZoom = value;
  xScale[0] = 4.0 * mandelZoom / (long double)imageWidth;
  yScale[0] = 4.0 * mandelZoom / (long double)imageHeight;
}

long double Data::getMandelZoom() {

  return(mandelZoom);
}

void Data::setJuliaZoom(long double value) {

  xOfs[1] = ((long double)(imageWidth>>1) * (juliaZoom - value) + xOfs[1] * juliaZoom) / value;
  yOfs[1] = ((long double)(imageHeight>>1) * (juliaZoom - value) + yOfs[1] * juliaZoom) / value;
  juliaZoom = value;
  xScale[1] = 8.0 * juliaZoom / (long double)imageWidth;
  yScale[1] = 8.0 * juliaZoom / (long double)imageHeight;
}

long double Data::getJuliaZoom() {  
 
  return(juliaZoom);
}

void Data::setImageWidth(int value) {

  imageWidth= value;
  xScale[0] = 4.0 * mandelZoom / (long double)imageWidth;
  xScale[1] = 8.0 * juliaZoom / (long double)imageWidth;
  xOfs[0] = -imageWidth>>1; 
  xOfs[1] = -imageWidth>>1; 
}
        
int Data::getImageWidth() {

  return(imageWidth);
}
            
void Data::setImageHeight(int value) {

  imageHeight = value;
  yScale[0] = 4.0 * mandelZoom / (long double)imageHeight;
  yScale[1] = 8.0 * juliaZoom / (long double)imageHeight;
  yOfs[0] = -imageHeight>>1;
  yOfs[1] = -imageHeight>>1;
}

int Data::getImageHeight() {

  return(imageHeight);
}

void Data::getScale(int index, long double &x, long double &y) {

  x = xScale[index];
  y = yScale[index];
} 

void Data::setOfs(int index, int x, int y) {

  xOfs[index] = x;
  yOfs[index] = y;
}

void Data::getOfs(int index, int &x, int &y) {

  x = (int)xOfs[index];
  y = (int)yOfs[index];
      
}

void Data::moveOfs(int index, QPoint dp) {

  xOfs[index] += dp.x();
  yOfs[index] += dp.y();
}

long double Data::getXScale(int index) {

  return(xScale[index]);
}
   
long double Data::getYScale(int index) {
   
  return(yScale[index]);
}
     
int Data::getXOfs(int index) {
     
  return((int)xOfs[index]);
} 
       
int Data::getYOfs(int index) {
        
  return((int)yOfs[index]);
}

void Data::setC(long double p_c_re, long double p_c_im) {

  c_re = p_c_re;
  c_im = p_c_im;
}

void Data::getC(long double &p_c_re, long double &p_c_im) {

  p_c_re = c_re;
  p_c_im = c_im;
}

void Data::clearPalette(int index) {

  colorCount[index] = 0;
}

void Data::addColor(int index, int pos, QColor p_color, int &colorIndex) {

  int i;

  colorIndex = 0;
  while ((color[index][colorIndex].pos < pos) && (colorIndex < colorCount[index])) {
    colorIndex++;
  }  
  for (i = colorCount[index]; i > colorIndex; i--) {        
    color[index][i].color = color[index][i - 1].color;
    color[index][i].pos = color[index][i - 1].pos;
  }
  color[index][colorIndex].color = p_color;
  color[index][colorIndex].pos = pos;
  colorCount[index]++;
  updatePalette(index);
}

void Data::delColor(int index, int colorIndex) {

  int i;

  for (i = colorIndex; i < colorCount[index] - 1; i++) {        
    color[index][i].color = color[index][i + 1].color;
    color[index][i].pos = color[index][i + 1].pos;
  }
  colorCount[index]--;
  updatePalette(index);
}

void Data::setColor(int index, int colorIndex, QColor p_color) {

  color[index][colorIndex].color = p_color;
  updatePalette(index);
}

QColor Data::getColor(int index, int colorIndex, int &pos) {

  pos = color[index][colorIndex].pos;
  return(color[index][colorIndex].color);
}

void Data::setColorPos(int index, int colorIndex, int pos) {

  color[index][colorIndex].pos = pos;
  updatePalette(index);
}

int Data::getColorPos(int index, int colorIndex) {

  return(color[index][colorIndex].pos);
}

int Data::getColorCount(int index) {

  return(colorCount[index]);
}

QColor Data::getPalette(int index, int paletteIndex) {

  return(palette[index][paletteIndex]);
}

void Data::updatePalette(int index) {

  int i1, i2;
  long double rgbScale[3];

  for (i1 = 0; i1 < colorCount[index] - 1; i1++) {
    rgbScale[0] = (long double)(color[index][i1 + 1].color.red() - color[index][i1].color.red()) / (long double)(color[index][i1 + 1].pos - color[index][i1].pos);
    rgbScale[1] = (long double)(color[index][i1 + 1].color.green() - color[index][i1].color.green()) / (long double)(color[index][i1 + 1].pos - color[index][i1].pos);
    rgbScale[2] = (long double)(color[index][i1 + 1].color.blue() - color[index][i1].color.blue()) / (long double)(color[index][i1 + 1].pos - color[index][i1].pos);
    for (i2 = color[index][i1].pos; i2 < color[index][i1 + 1].pos; i2++) {
      palette[index][i2] = QColor((int)((long double)color[index][i1].color.red() + rgbScale[0] * (long double)(i2 - color[index][i1].pos)), 
                                  (int)((long double)color[index][i1].color.green() + rgbScale[1] * (long double)(i2 - color[index][i1].pos)),
                                  (int)((long double)color[index][i1].color.blue() + rgbScale[2] * (long double)(i2 - color[index][i1].pos)));
    }
    palette[index][maxIterations[index]] = QColor((int)((long double)color[index][i1].color.red() + rgbScale[0] * (long double)(i2 - color[index][i1].pos)),
                                           (int)((long double)color[index][i1].color.green() + rgbScale[1] * (long double)(i2 - color[index][i1].pos)),
                                           (int)((long double)color[index][i1].color.blue() + rgbScale[2] * (long double)(i2 - color[index][i1].pos)));
  }  
}
 
void Data::setRescalePalette(bool on) {

  rescalePalette = on;
}
 
bool Data::getRescalePalette() { 

  return(rescalePalette);
}

void Data::setPaletteScale(int index, double value) {

  paletteScale[index] = value;
}

double Data::getPaletteScale(int index) {

  return(paletteScale[index]);
}

void Data::setNormalize(bool on) {

  normalize = on;
}

bool Data::getNormalize() {

  return(normalize);
}

void Data::setMaxIterations(int index, int value) {

  long double scale;
  int i;

  scale = (long double)value / (long double)maxIterations[index];  
  maxIterations[index] = value;
  if (rescalePalette) {
    for (i = 0; i < colorCount[index] - 1; i++) {
      color[index][i].pos = (int)((long double)color[index][i].pos * scale);
    }  
  } 
  color[index][colorCount[index]-1].pos = maxIterations[index];
}

int Data::getMaxIterations(int index) {

  return(maxIterations[index]);
}

void Data::setMaxAbsoluteValue(int index, double value) {

  maxAbsoluteValue[index] = value;
}

double Data::getMaxAbsoluteValue(int index) {

  return(maxAbsoluteValue[index]);
}

void Data::calcMax(int index) {
  
  int i1, i2, val;

  if (index) {
    max[index] = 0;
    for(i1 = 0; i1 < imageHeight; i1++) {
      for(i2 = 0; i2 < imageWidth; i2++) {
        val = juliaData[i1 * imageWidth + i2];
        if (val > max[index])  max[index] = val;
      }
    }  
  } else {
    max[index] = 0;
    for(i1 = 0; i1 < imageHeight; i1++) {
      for(i2 = 0; i2 < imageWidth; i2++) {
        val = mandelData[i1 * imageWidth + i2];
        if (val > max[index])  max[index] = val;
      }
    }  
  }
}

int Data::getMax(int index) {
  
  return(max[index]);
}

void Data::setAntiAliasing(int value) {

  antiAliasing = value;
}

int Data::getAntiAliasing() {

  return(antiAliasing);
}
