/*  Capacitance and ESR meter
 *  Based on Capacitance meter by Paul Badger 2008, see: http://arduino.cc/it/Tutorial/CapacitanceMeter
 *  ESR extension rp (2014-12), tesr < 1ms
 *
 * Capacitance:
 *    A capcitor will charge, through a resistor, in one time constant, defined as T seconds where
 *    TC = R * C
 * 
 *    TC = time constant period in seconds
 *    R = resistance in ohms
 *    C = capacitance in farads (1 microfarad (ufd) = .0000001 farad = 10^-6 farads ) 
 *
 *    The capacitor's voltage at one time constant is defined as 63.2% of the charging voltage.
 *    U(t) = U0 * (1- exp(-t/TC); if t = TC -> U(t)/U= = 0.632
 *
 *
 * ESR
 *
 *    R should be as small as possible (limited by Arduino Imax): 220 Ohm 
 *    tesr: time between charging C and ADC input. 
 *          Insert "delay(tesr)" if tesr should be larger than 1 ms
 *          c(t) correction becomes difficult for large tesr due to limited ADC resolution.
 *        
 * 
 */

const byte analogPin =  0;         // analog pin for measuring capacitor voltage
const byte chargePin =  13;     // pin to charge the capacitor - connected to one end of the charging resistor
const byte dischargePin =  11;         // pin to discharge the capacitor
const float resistorValue = 10000.0F;  // change this to whatever resistor value you are using
//const float resistorValue = 330000.0F;  //use larger resistor for small capacities <1 µF
const float resistorSml = 220.0;       // discharge; used for ESR measurement as well.                        

unsigned long startTime;
unsigned long elapsedTime;        // elapsedTime (in ms) 
float microFarads;                // floating point variable to preserve precision, make calculations
float nanoFarads;

//ESR calculation; 
//R = resistorSml
float Ucesr;          // Uc + Uesr
float Resr;           // equivalent serial resistance ESR
const float tesr = 0.2; // estimated time delay between begin of charging and ADC reading
const float U0 = 5;     // 5 V supply



void setup(){
  dischargec();
  pinMode(chargePin, OUTPUT);     // set chargePin to output
  digitalWrite(chargePin, LOW);  
  pinMode(dischargePin, INPUT);            // set discharge pin back to input

  Serial.begin(115200);             // initialize serial transmission for debugging
  Serial.println("(Messzeit); Kapazitaet; ESR +-20 Ohm");
}

void loop(){
  delay(4000);
  digitalWrite(chargePin, HIGH);  // set chargePin HIGH and capacitor charging
  startTime = millis();
  
  /* Measure Capacity */
  while(analogRead(analogPin) < 648){       // 647 = 63.2% of 1023, which corresponds to full-scale voltage 
  }

  elapsedTime= millis() - startTime;
  if (elapsedTime>1) {  //output only if time was measured
  
    // convert milliseconds to seconds ( 10^-3 ) and Farads to microFarads ( 10^6 ),  net 10^3 (1000)  
    microFarads = ((float)elapsedTime / resistorValue) * 1000;  
    Serial.print("("); 
    Serial.print(elapsedTime);       // print the value to serial port
    Serial.print(" ms); ");         

    if (microFarads > 1){
      Serial.print((long)microFarads);       // print the value to serial port
      Serial.print(" mikro F ");         // print units and carriage return
      }
      else {
      // if value is smaller than one microFarad, convert to nanoFarads (10^-9 Farad). 
      nanoFarads = microFarads * 1000.0;      // multiply by 1000 to convert to nanoFarads (10^-9 Farads)
      Serial.print((long)nanoFarads);         // print the value to serial port
      Serial.print(" nano F");          // print units and carriage return
    }
    //delay(300);
    dischargec(); /* dicharge the capacitor  */

    /* Measure ESR */
    
    if (microFarads <5) {
      Serial.println(""); //  ESR skipped (C too small)
    } else {
      pinMode(chargePin, INPUT);            // disable chargepin
      pinMode(dischargePin, OUTPUT);
      delay(50);
      digitalWrite(dischargePin, HIGH);  // switch on and read voltage immediately.
      //delay(tesr);                     // skip delay in order to be as quick as possible
      Ucesr = analogRead(analogPin);     // time delay: approx 0.1 ms (required for ADC)
      Ucesr = Ucesr * 5 / 1024;
      Resr = 0;                   // first approximation
      Resr= get_esr();            // might be required if tesr >> 1ms
      Resr= get_esr();
      if (Resr <0)                // remove overcompensation of correction; might happen for small C and Ucesr near resolution limit
        Resr = 0;
      Serial.print("; ESR: ");
      Resr = (int) Resr/10;
      Resr *= 10;
      Serial.print((int)Resr);
      Serial.print(" Ohm");
      if (Resr>1000){
        Serial.println("; Check C, ESR quite large!");
      } else {
        Serial.println("");
      }
    }
  } //end of elapsedTime > 0
  dischargec();
  pinMode(dischargePin, INPUT);            // restore: set discharge_pin back to input
  pinMode(chargePin, OUTPUT);              //     and charge_pin to output
} 


void dischargec(){
  //discharge capacity
  digitalWrite(chargePin, LOW);             // set charge pin to  LOW, discharge via resistorValue 
  pinMode(dischargePin, OUTPUT);            // set discharge pin to output 
  digitalWrite(dischargePin, LOW);          // set discharge pin LOW, discharge via discharge resistor
  while(analogRead(analogPin) > 0){         // wait until capacitor is completely discharged
  }
  delay(100);
}


float get_esr(){
  //calculate R_ESR: (Ucesr - Uc) : (U0 - Ucesr) = R_ESR : R  
  //for small tesr correction is negligible
  float Uc = ct(tesr, (resistorSml+Resr) * microFarads /1000); 
  return (resistorSml * (Ucesr-Uc)/(U0-Ucesr));
}

  
float ct(float t, float RC){
  //calculate Uc(t, RC)
  return (U0 * (1-exp(-t/RC)));
}
