//-----------------------------------------------------------------------------
//  ProcessM.c
/// @file
///
/// @brief Central module of the application
///
/// This is the central module of the application.h The main task 'ProcessM'
/// is implemented in this task.
///
/// @copyright 2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard include files
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "null.h"
#include "string.h"

//-----------------------------------------------------------------------------
// Project specific include files
//-----------------------------------------------------------------------------
#include "OS.h"
#include "ProcessM.h"
#include "MeasureM.h"
#include "MessageDefs.h"
#include "Tools.h"
#include "CpuRegHAL.h"

#include "Events.h"

#include "GPIO1.h"
#include "GPIO2.h"
#include "IO_Map.h"

#include "Application.h"
#include "MPL3115A2.h"

//-----------------------------------------------------------------------------
// Privat definitions, macros and constants
//-----------------------------------------------------------------------------
/// default value for the measure period
#define DEFAULT_IDLE_MEASURE_PERIOD_MS       200

/// softwareversion of the application
#define SW_VERSION                           0
/// softwarerevision of the application
#define SW_REVISION                          1
/// softwaresubrevision
#define SW_SUBREVISION                       6

/// hardware version
#define HW_VERSION                           2
/// hardware revision
#define HW_REVISION                          0

/// os type (1 - FreeRTOS)
#define OS_TYPE_FREE_RTOS                    1

/// os version
#define OS_VERSION                           7
/// os revision
#define OS_REVISION                          2
/// os subrevision
#define OS_SUBREVISION                       0

/// number of led flashes at application start
#define LED_NUM_OF_FLASHES_AT_APP_START      5
/// led flash frequency at application start in hz
#define LED_FLASH_FREQUENCY_AT_APP_START     1
/// number  of led flashes at checking key actions 
#define LED_NUM_OF_FLASHES_AT_CHK_KACTIONS   1
/// led flash frequency at checking key actions in hz
#define LED_FLASH_FREQUENCY_AT_CHK_KACTIONS  1

//-----------------------------------------------------------------------------
// Private data types
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// External references
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global member for the modul
//-----------------------------------------------------------------------------
/// Device info data of the system
MSG_DATA_DEVICE_INFO  m_DeviceInfo;
/// Device config data of the system
MSG_DATA_DEVICE_CONFIG m_ConfigData;
/// Timeoutvalue for measurement (disable the interrupts, when you modify the value)
UWORD m_MeasureTimeout;
/// Measure mode
MEASURE_MODE m_MeasureMode;
/// Number of files in the flash file system
UWORD m_ProcessM_NumOfFilesInFFS;

//-----------------------------------------------------------------------------
// Prototypes of private functions
//-----------------------------------------------------------------------------

// Send a message for display a float value to the led manager
void ProcessM_ShowValue (VALUE enValue );
// check if a key action is valid or not
bool IsKeyActionValid (KEY_ACTION * pKeyAction);

// handler for the message MSG_INIT_SYSTEM
void ProcessM_MsgInitSystem (void);
// handler for the message MSG_KEY_ACTION
void ProcessM_MsgKeyAction (KEY_ACTION * pKeyAction);
// handler for the message MSG_FALSHMEM_INFO
void ProcessM_MsgFlashMemInfo (MSG_DATA_FALSHMEM_INFO * pData);
// handler for the message MSG_READ_DEVICE_CONFIG_DATA
void ProcessM_MsgReadDeviceConfigData (MSG_DATA_DEVICE_CONFIG * pConfigData);
// handler for the message MSG_REQ_DEVICE_INFO
void ProcessM_MsgReqDeviceInfo();
// handler for the message MSG_REQ_DEVICE_CONFIG
void ProcessM_MsgReqDeviceConfig();
// handler for the message MSG_REQ_ACTUAL_MVALUES
void ProcessM_MsgReqActualMValues();
// handler for the message MSG_DEVICE_CONFIG
void ProcessM_MsgDeviceConfig (MSG_DATA_DEVICE_CONFIG * pData);
// handler for the message MSG_RESET_MVALUES
void ProcessM_MsgResetMValues();
// handler for the message MSG_MEASURE_MODE
void ProcessM_MsgMeasureMode (MSG_DATA_MEASURE_MODE * pMeasureMode);
// Sets the config data
void ProcessM_SetConfigData (MSG_DATA_DEVICE_CONFIG * pConfigData);
// Set the measure mode
void ProcessM_SetMeasureMode (MEASURE_MODE MeasureMode);
// Inform all other task that the device configuration is changed
void ProcessM_SendConfigChangedMsg (void);
// handler for the message MSG_MEASURE_MODE
void ProcessM_MsgReqMeasureMode();
// handler for the message MSG_CLEAR_ALL_DATA_FILES
void ProcessM_MsgClearAllDataFiles();
// handler for the message MSG_NUM_OF_FILES_IN_FFS
void ProcessM_MsgNumOfFilesInFFS (MSG_DATA_NUM_OF_FILES_IN_FFS * pNumOfFilesInFFS);
// handler for the message MSG_REQ_FFS_FILE
void ProcessM_MsgReqFFSFile (MSG_DATA_REQ_MEASURE_FILE * pReqMeasureFile);
// andler for the message MSG_START_FFS_FILE_TRANSFER
void ProcessM_MsgStartFFSFileTransfer (MSG_DATA_START_FFS_FILE_TRANSFER * pStartFFSFileTransfer);
// handler for the message MSG_FFS_FILE_BLOCK
void ProcessM_MsgFFSFileBlock (MSG_DATA_FFS_FILE_BLOCK * pFFSFileBlock);
// handler for the message MSG_FFS_FILE_EOF
void ProcessM_MsgFFSFileEof();
// enable or disable the logging mode
void ProcessM_KASwitchLogging (UBYTE EnableLogging);
// switching the power supply of the bluetooth module
void ProcessM_KASwitchBluetooth (UBYTE EnableBluetooth);
// handler for the message MSG_GET_NEXT_FFS_FILE_BLOCK
void ProcessM_MsgGetNextFFSFileBlock();

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function  ProcessM_TaskFunction
///
/// @brief Taskfunction for the ProcessM-Task
///
/// @param pvParameters (@c [in] void *) - not used
///
/// This is the Taskfunction for the ProcessM-Task. The main function is reading
/// messages from a message queue and calling the processing functions for the
/// messages. Please do not process a message directly in this method!
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_TaskFunction (void * pvParameters)
{
   MESSAGE Message;

   while (1)
   {
      if (OS_ReceiveMessage (TASK_ID_PROCESS_M, &Message, WAIT_MAX) == ERR_OS_OK)
      {
         switch (Message.MessageID)
         {
            case MSG_INIT_SYSTEM:               ProcessM_MsgInitSystem();                                                              break;
            case MSG_KEY_ACTION:                ProcessM_MsgKeyAction ((KEY_ACTION *) Message.pMessageData);                           break;
            case MSG_FALSHMEM_INFO:             ProcessM_MsgFlashMemInfo ((MSG_DATA_FALSHMEM_INFO * ) Message.pMessageData);           break;
            case MSG_CONFIG_DATA:               ProcessM_MsgReadDeviceConfigData ((MSG_DATA_DEVICE_CONFIG *) Message.pMessageData);    break;
            case MSG_REQ_DEVICE_INFO:           ProcessM_MsgReqDeviceInfo();                                                           break;
            case MSG_REQ_DEVICE_CONFIG:         ProcessM_MsgReqDeviceConfig();                                                         break;
            case MSG_DEVICE_CONFIG:             ProcessM_MsgDeviceConfig ((MSG_DATA_DEVICE_CONFIG *) Message.pMessageData);            break;
            case MSG_REQ_ACTUAL_MVALUES:        ProcessM_MsgReqActualMValues();                                                        break;
            case MSG_RESET_MVALUES:             ProcessM_MsgResetMValues();                                                            break;
            case MSG_READ_DEVICE_CONFIG_DATA:   ProcessM_MsgReadDeviceConfigData ((MSG_DATA_DEVICE_CONFIG *) Message.pMessageData);    break;
            case MSG_MEASURE_MODE:              ProcessM_MsgMeasureMode ((MSG_DATA_MEASURE_MODE *) Message.pMessageData);              break;
            case MSG_REQ_MEASURE_MODE:          ProcessM_MsgReqMeasureMode();                                                          break;
            case MSG_CLEAR_ALL_DATA_FILES:      ProcessM_MsgClearAllDataFiles();                                                       break;
            case MSG_NUM_OF_FILES_IN_FFS:       ProcessM_MsgNumOfFilesInFFS ((MSG_DATA_NUM_OF_FILES_IN_FFS *) Message.pMessageData);   break;
            case MSG_REQ_FFS_FILE:              ProcessM_MsgReqFFSFile ((MSG_DATA_REQ_MEASURE_FILE *) Message.pMessageData);           break;
            case MSG_START_FFS_FILE_TRANSFER:   ProcessM_MsgStartFFSFileTransfer ((MSG_DATA_START_FFS_FILE_TRANSFER *) Message.pMessageData); break;
            case MSG_FFS_FILE_BLOCK:            ProcessM_MsgFFSFileBlock ((MSG_DATA_FFS_FILE_BLOCK *) Message.pMessageData);           break;
            case MSG_GET_NEXT_FFS_FILE_BLOCK:   ProcessM_MsgGetNextFFSFileBlock ();                                                    break;
            case MSG_FFS_FILE_EOF:              ProcessM_MsgFFSFileEof();                                                              break;
            
            default:
               break;
         }

         // Speicher wenn notwendig freigeben
         if (Message.pMessageData != NULL)
         {
            FreeMemory (Message.pMessageData);
         }
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgInitSystem
///
/// @brief Handler for the message MSG_INIT_SYSTEM
///
/// This is the handler function for the message MSG_INIT_SYSTEM. The function
/// will call all central initialisiation functions and send the init message
/// to the other task of the application.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-09-13 Eike Mueller(TABO)
///            Enable the very low power modes at system start
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgInitSystem (void)
{
   MESSAGE Msg;

   // init the member (the system starts in normal mode)
   m_ConfigData.MeasurePeriodNormal  = DEFAULT_IDLE_MEASURE_PERIOD_MS;
   m_ConfigData.MeasurePeriodLogging = DEFAULT_IDLE_MEASURE_PERIOD_MS;

   m_ProcessM_NumOfFilesInFFS = 0;

   m_MeasureTimeout = (m_ConfigData.MeasurePeriodNormal / 10) -1;

   m_MeasureMode = MM_NORMAL;

   TU1_Init (NULL);

   CRHal_EnableVeryLowPowerModes();
   
   GPIO1_SetPortBits (NULL, GPIO1_MMA_SA0_MASK);

   // send the init message to the measure manager ...
   MeasureM_SetJob (MMJ_INIT_MEASURE_SYSTEM);

   //-------------------------------------------------------------------------
   // ... and to the communication manager ...
   //-------------------------------------------------------------------------
   Msg.MessageID    = MSG_INIT_SYSTEM;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX);
   
   //-------------------------------------------------------------------------
   // ... and to the flash manager
   //-------------------------------------------------------------------------
   (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);

   //-------------------------------------------------------------------------
   // ... and to the led manager
   //-------------------------------------------------------------------------
   (void)OS_SendMessage (TASK_ID_LED_M, &Msg, WAIT_MAX);   
   
   //-------------------------------------------------------------------------
   // set the init flashes (5 x green @ 2 hz)
   //-------------------------------------------------------------------------
   Msg.MessageID    = MSG_SET_LED_JOB;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_SET_LED_JOB));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->LEDName        = LN_GREEN;
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->LEDJob         = LS_FLASH;
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->NumOfFlashes   = LED_NUM_OF_FLASHES_AT_APP_START;
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->FlashFrequency = LED_FLASH_FREQUENCY_AT_APP_START;

      if (OS_SendMessage (TASK_ID_LED_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }

   //-------------------------------------------------------------------------
   // request the flash memeory info from the flash manager
   //-------------------------------------------------------------------------
   Msg.MessageID    = MSG_REQ_FALSHMEM_INFO;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);

   //-------------------------------------------------------------------------
   // request the device config data from the flash manager
   //-------------------------------------------------------------------------
   Msg.MessageID    = MSG_REQ_READ_DEVICE_CONFIG_DATA;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReqDeviceInfo
///
/// @brief Handler for the message MSG_REQ_DEVICE_INFO
///
/// This is the handler function for the message MSG_REQ_DEVICE_INFO.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReqDeviceInfo (void)
{
   MESSAGE Msg;

   Msg.MessageID     = MSG_DEVICE_INFO;
   Msg.pMessageData  = (UBYTE *) AllocateMemory (sizeof(MSG_DATA_DEVICE_INFO));
   
   if (Msg.pMessageData != NULL)
   {      
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->FlashMemInfo.FlashDeviceType  = m_DeviceInfo.FlashMemInfo.FlashDeviceType;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->FlashMemInfo.FlashSize        = m_DeviceInfo.FlashMemInfo.FlashSize;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->CPU_DeviceID = CRHal_GetCPUDeviceID();;

      CRHal_GetCPUUniqueID (((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->CPU_UnigueID);
      
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->SWVersion      = SW_VERSION;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->SWRevision     = SW_REVISION;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->SWSubRevision  = SW_SUBREVISION;

      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->HWVersion      = HW_VERSION;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->HWRevision     = HW_REVISION;

      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->BT_FWVersion   = 0x03;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->BT_FWRevision  = 0x04;
      
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->OSType         = OS_TYPE_FREE_RTOS;
      
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->OSVersion      = OS_VERSION;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->OSRevision     = OS_REVISION;
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->OSSubRevision  = OS_SUBREVISION;   
      
      ((MSG_DATA_DEVICE_INFO *) Msg.pMessageData)->NumOfFiles     = m_ProcessM_NumOfFilesInFFS;
      
      if (OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReqDeviceConfig
///
/// @brief Handler for the message MSG_REQ_DEVICE_CONFIG
///
/// This is the handler function for the message MSG_REQ_DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReqDeviceConfig (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_DEVICE_CONFIG;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_DEVICE_CONFIG));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_DEVICE_CONFIG *) Msg.pMessageData)->MeasurePeriodNormal  = m_ConfigData.MeasurePeriodNormal;
      ((MSG_DATA_DEVICE_CONFIG *) Msg.pMessageData)->MeasurePeriodLogging = m_ConfigData.MeasurePeriodLogging;

      if (OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReqActualMValues
///
/// @brief Handler for the message MSG_REQ_ACTUAL_MVALUES
///
/// This is the handler function for the message MSG_REQ_ACTUAL_MVALUES.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReqActualMValues (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_ACTUAL_MVALUES;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_ACTUAL_MVALUES));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->TimestampHi          = MeasureM_GetTimestamp (VAL_TIMESTAMP_HI);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->TimestampLo          = MeasureM_GetTimestamp (VAL_TIMESTAMP_LO);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualAltitude       = MeasureM_GetMeasureValue (VAL_ACTUAL_ALTITUDE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualPressure       = MeasureM_GetMeasureValue (VAL_ACTUAL_PRESSURE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualTemperature    = MeasureM_GetMeasureValue (VAL_ACTUAL_TEMPERATURE);      
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualAccXDir        = MeasureM_GetMeasureValue (VAL_ACTUAL_ACC_X_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualAccYDir        = MeasureM_GetMeasureValue (VAL_ACTUAL_ACC_Y_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->ActualAccZDir        = MeasureM_GetMeasureValue (VAL_ACTUAL_ACC_Z_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalAltitude      = MeasureM_GetMeasureValue (VAL_MINIMAL_ALTITUDE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalPressure      = MeasureM_GetMeasureValue (VAL_MINIMAL_PRESSURE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalTemperature   = MeasureM_GetMeasureValue (VAL_MINIMAL_TEMPERATURE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalAccXDir       = MeasureM_GetMeasureValue (VAL_MINIMAL_ACC_X_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalAccYDir       = MeasureM_GetMeasureValue (VAL_MINIMAL_ACC_Y_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MinimalAccZDir       = MeasureM_GetMeasureValue (VAL_MINIMAL_ACC_Z_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalAltitude      = MeasureM_GetMeasureValue (VAL_MAXIMAL_ALTITUDE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalPressure      = MeasureM_GetMeasureValue (VAL_MAXIMAL_PRESSURE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalTemperature   = MeasureM_GetMeasureValue (VAL_MAXIMAL_TEMPERATURE);
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalAccXDir       = MeasureM_GetMeasureValue (VAL_MAXIMAL_ACC_X_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalAccYDir       = MeasureM_GetMeasureValue (VAL_MAXIMAL_ACC_Y_POS); 
      ((MSG_DATA_ACTUAL_MVALUES *) Msg.pMessageData)->MaximalAccZDir       = MeasureM_GetMeasureValue (VAL_MAXIMAL_ACC_Z_POS);

      if (OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgResetMValues
///
/// @brief Handler for the message MSG_RESET_MVALUES
///
/// This is the handler function for the message MSG_RESET_MVALUES.
///
/// @status  Draft
/// @date    2013-06-14
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-14 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgResetMValues (void)
{
   MeasureM_ClearMeasureValues();
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgDeviceConfig
///
/// @brief Handler for the message MSG_DEVICE_CONFIG
///
/// @param pData (@c [in] MSG_DATA_DEVICE_CONFIG *) - Data of the message
///
/// This is the handler function for the message MSG_DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgDeviceConfig( MSG_DATA_DEVICE_CONFIG * pData )
{
   MESSAGE Msg;

   if (pData == NULL)
      return;

   // get the new config data ...
   ProcessM_SetConfigData (pData);

   ProcessM_SendConfigChangedMsg();

   // ... and send a message to the flash manager to store them 
   Msg.MessageID    = MSG_WRITE_DEVICE_CONFIG_DATA;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_DEVICE_CONFIG));

   if (Msg.pMessageData != NULL)
   {
      memcpy (Msg.pMessageData, &m_ConfigData, sizeof(MSG_DATA_DEVICE_CONFIG));

      if (OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgFlashMemInfo
///
/// @brief Handler for the message MSG_FLASH_MEMINFO
///
/// @param pData (@c [in] MSG_DATA_FALSHMEM_INFO *) - Data of the message
///
/// This is the handler function for the message MSG_FLASH_MEMINFO.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgFlashMemInfo (MSG_DATA_FALSHMEM_INFO * pData)
{
   if (pData == NULL)
      return;

   m_DeviceInfo.FlashMemInfo.FlashSize       = pData->FlashSize;
   m_DeviceInfo.FlashMemInfo.FlashDeviceType = pData->FlashDeviceType;
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgKeyAction
///
/// @brief Handler for the message MSG_KEY_ACTION
///
/// @param pKeyAction (@c [in] KEY_ACTION *) - Data of the message
///
/// This is the handler function for the message MSG_KEY_ACTION.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgKeyAction (KEY_ACTION * pKeyAction)
{
   MESSAGE Msg;
   bool    RetVal;

   Msg.MessageID    = MSG_SET_LED_JOB;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_SET_LED_JOB));

   if (Msg.pMessageData != NULL)
   {
      RetVal = IsKeyActionValid (pKeyAction);

      if (RetVal == TRUE)
      {
         ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->LEDName = LN_GREEN;
      }
      else
      {
         ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->LEDName = LN_RED;
      }

      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->LEDJob         = LS_FLASH;
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->NumOfFlashes   = LED_NUM_OF_FLASHES_AT_CHK_KACTIONS;
      ((MSG_DATA_SET_LED_JOB *) Msg.pMessageData)->FlashFrequency = LED_FLASH_FREQUENCY_AT_CHK_KACTIONS;

      if (OS_SendMessage (TASK_ID_LED_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }

      if (RetVal == TRUE)
      {
         OS_Sleep (1500);
      }
      else
      {
         return;
      }
   }

   switch (*pKeyAction)
   {
      case KA_SHOW_ACTUAL_ALTITUDE:    ProcessM_ShowValue (VAL_ACTUAL_ALTITUDE);    break;
      case KA_SHOW_MAX_ALTITUDE:       ProcessM_ShowValue (VAL_MAXIMAL_ALTITUDE);   break;
      case KA_SHOW_MIN_ALTITUDE:       ProcessM_ShowValue (VAL_MINIMAL_ALTITUDE);   break;
      case KA_SHOW_ACTUAL_TEMP:        ProcessM_ShowValue (VAL_ACTUAL_TEMPERATURE); break;
      case KA_SHOW_RESET_MAX_MIN_VAL:  MeasureM_ClearMeasureValues ();              break;
      case KA_START_LOGGING:           ProcessM_KASwitchLogging (1);                break;
      case KA_STOP_LOGGING:            ProcessM_KASwitchLogging (0);                break;
      case KA_ENABLE_BLUETOOTH:        ProcessM_KASwitchBluetooth (1);              break;
      case KA_DISABLE_BLUETOOTH:       ProcessM_KASwitchBluetooth (0);              break;
      default:                                                                      break;
   }
}

//-----------------------------------------------------------------------------
//  Function IsKeyActionValid 
///
/// @brief Checks if a key action is valid
///
/// @param pKeyAction (@c [in] KEY_ACTION *) - Key action
///
/// @return bool true - Key action is valid, false - Key action is invalid
///
/// This function checks if a key action is valid.
///
/// @status  Draft
/// @date    2013-08-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-08-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool IsKeyActionValid (KEY_ACTION * pKeyAction)
{
   bool RetVal = FALSE;

   if (pKeyAction == NULL)
      return RetVal;

   if (m_MeasureMode == MM_NORMAL)
   {
      switch (*pKeyAction)
      {
         case KA_SHOW_ACTUAL_ALTITUDE:    
         case KA_SHOW_MAX_ALTITUDE:       
         case KA_SHOW_MIN_ALTITUDE:       
         case KA_SHOW_ACTUAL_TEMP:        
         case KA_SHOW_RESET_MAX_MIN_VAL:
         case KA_ENABLE_BLUETOOTH:
         case KA_DISABLE_BLUETOOTH:
         case KA_START_LOGGING:
         default:
            RetVal = TRUE;
            break;

         case KA_STOP_LOGGING:
            RetVal = FALSE;
            break;
      }
   }
   else
   {
      switch (*pKeyAction)
      {
         case KA_SHOW_ACTUAL_ALTITUDE:    
         case KA_SHOW_MAX_ALTITUDE:       
         case KA_SHOW_MIN_ALTITUDE:       
         case KA_SHOW_ACTUAL_TEMP:        
         case KA_SHOW_RESET_MAX_MIN_VAL:
         case KA_ENABLE_BLUETOOTH:
         case KA_DISABLE_BLUETOOTH:
         case KA_START_LOGGING:
         default:
            RetVal = FALSE;
            break;

         case KA_STOP_LOGGING:
            RetVal = TRUE;
            break;
      }
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function ProcessM_KASwitchLogging 
///
/// @brief Enable or disable the logging mode
///
/// @param EnableLogging (@c [in] UBYTE) - Flag enable the logging mode
///
/// This function enables or disables the logging mode.
///
/// @status  Draft
/// @date    2013-07-17
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-17 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_KASwitchLogging (UBYTE EnableLogging)
{
   MESSAGE Msg;
   
   Msg.MessageID    = MSG_MEASURE_MODE;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_MEASURE_MODE));
   
   if (Msg.pMessageData != NULL)
   {
      if (EnableLogging == 1)
      {
         ((MSG_DATA_MEASURE_MODE *) Msg.pMessageData)->MeasureMode = MM_LOGGING;
      }
      else
      {
         ((MSG_DATA_MEASURE_MODE *) Msg.pMessageData)->MeasureMode = MM_NORMAL;         
      }
      
      if (OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_KASwitchBluetooth 
///
/// @brief Switching the power supply of the bluetooth module
///
/// @param EnableBluetooth (@c [in] UBYTE) - Flag switch on or off the power supply
///
/// This function switches the power supply of the bluetooth module.
///
/// @status  Draft
/// @date    2013-07-17
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-17 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_KASwitchBluetooth (UBYTE EnableBluetooth)
{
   MESSAGE Msg;
   
   Msg.MessageID    = MSG_ENABLE_BLUETOOTH;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_ENABLE_BLUETOOTH));
   
   if (Msg.pMessageData != NULL)
   {
      if (EnableBluetooth == 1)
      {
         ((MSG_DATA_ENABLE_BLUETOOTH *) Msg.pMessageData)->EnableBluetooth = 1;
      }
      else
      {
         ((MSG_DATA_ENABLE_BLUETOOTH *) Msg.pMessageData)->EnableBluetooth = 0;         
      }
      
      if (OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }   
}

//-----------------------------------------------------------------------------
//  Function ProcessM_GetMeasureTimeout
///
/// @brief Get the actual measure timeout value 
///
/// @return UWORD Measure timeout value in incremental steps for the timer isr 
///
/// This function gets the actual measure timeout value.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UWORD ProcessM_GetMeasureTimeout (void)
{
   return m_MeasureTimeout;
}

//-----------------------------------------------------------------------------
//  Function ProcessM_ShowValue
///
/// @brief Send a message for display a float value to the led manager
///
/// @param enValue (@c [in] VALUE) - Id of the value to display
///
/// This function sends a message for display a float value to the led manager.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_ShowValue (VALUE enValue)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_SHOW_VALUE;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof( MSG_DATA_SHOW_VALUE ));
   
   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_SHOW_VALUE *) Msg.pMessageData)->Value = MeasureM_GetMeasureValue (enValue);
      
      if (OS_SendMessage (TASK_ID_LED_M, &Msg, WAIT_MAX) == ERR_OS_QUEUE_FULL)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReadDeviceConfigData
///
/// @brief Handler for the message MSG_READ_DEVICE_CONFIG_DATA
///
/// @param pConfigData (@c [in] MSG_DATA_DEVICE_CONFIG *) - Message data
///
/// This function is the handler for the message MSG_READ_DEVICE_CONFIG_DATA.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReadDeviceConfigData( MSG_DATA_DEVICE_CONFIG * pConfigData )
{
   MESSAGE Msg;

   if (pConfigData == NULL)
      return;

   // Ist die aktuelle Konfigration im Flash gltig?
   if (   pConfigData->MeasurePeriodLogging != 0xFFFF 
       && pConfigData->MeasurePeriodNormal  != 0
       && pConfigData->MeasurePeriodLogging != 0)
   {
      // Konfiguration ist gltig und kann bernommen werden
      ProcessM_SetConfigData (pConfigData);

      ProcessM_SendConfigChangedMsg();
   }
   else
   {
      // Defaultkonfiguration festlegen ...
      pConfigData->MeasurePeriodNormal = DEFAULT_IDLE_MEASURE_PERIOD_MS;
      pConfigData->MeasurePeriodLogging = DEFAULT_IDLE_MEASURE_PERIOD_MS;

      ProcessM_SetConfigData (pConfigData);

      ProcessM_SendConfigChangedMsg();

      // ... und an den Flashspeichermanager senden
      Msg.MessageID = MSG_WRITE_DEVICE_CONFIG_DATA;
      Msg.pMessageData = (PUBYTE) AllocateMemory(sizeof(MSG_DATA_DEVICE_CONFIG));

      if (Msg.pMessageData != NULL)
      {
         memcpy (Msg.pMessageData, &m_ConfigData, sizeof(MSG_DATA_DEVICE_CONFIG));

         if (OS_SendMessage( TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX ) == ERR_OS_QUEUE_FULL)
         {
            // Fehlerbehandlung
         }
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_SendConfigChangedMsg 
///
/// @brief Inform all other task that the device configuration is changed
///
/// This function inform all other task that the device configuration is changed.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_SendConfigChangedMsg (void)
{
   // send the change config message to the other task ...
   // ... to the measure manager ...
   MeasureM_SetJob (MMJ_CONFIG_CHANGED);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgMeasureMode 
///
/// @brief Handler for the message MSG_MEASURE_MODE
///
/// @param pMeasureMode (@c [in] MSG_DATA_MEASURE_MODE *) - Message data
///
/// This is the handler for the message MSG_MEASURE_MODE.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgMeasureMode (MSG_DATA_MEASURE_MODE * pMeasureMode)
{
   MESSAGE Msg;
   MESSAGE MsgSetLED;

   if (pMeasureMode == NULL)
      return;

   // set the yellow led to a flash frequency of 2 hz for display the logging mode
   MsgSetLED.MessageID    = MSG_SET_LED_JOB;
   MsgSetLED.pMessageData = (UBYTE *) AllocateMemory (sizeof (MSG_DATA_SET_LED_JOB));

   if (MsgSetLED.pMessageData == NULL)
      return;

   // check the measure mode for file operation
   if (pMeasureMode->MeasureMode == MM_NORMAL)
   {
      // close an open logging file
      Msg.MessageID    = MSG_CLOSE_ACTIVE_FILE;
      Msg.pMessageData = NULL;

      (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);

      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->LEDName        = LN_YELLOW;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->LEDJob         = LS_OFF;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->NumOfFlashes   = 0;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->FlashFrequency = 0;

      (void)OS_SendMessage (TASK_ID_LED_M, &MsgSetLED, WAIT_MAX);
   }
   else
   {
      // open an new logging file
      Msg.MessageID    = MSG_CREATE_NEW_MEASURE_FILE;
      Msg.pMessageData = NULL;

      (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);

      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->LEDName        = LN_YELLOW;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->LEDJob         = LS_FLASH;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->NumOfFlashes   = 0xFFFF;
      ((MSG_DATA_SET_LED_JOB *) MsgSetLED.pMessageData)->FlashFrequency = 2;

      (void)OS_SendMessage (TASK_ID_LED_M, &MsgSetLED, WAIT_MAX);
   }

   ProcessM_SetMeasureMode((MEASURE_MODE) pMeasureMode->MeasureMode);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReqMeasureMode
///
/// @brief Handler for the message MSG_MEASURE_MODE
///
/// This is the mandler for the message MSG_MEASURE_MODE.
///
/// @see ProcessM_MsgMeasureMode()
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReqMeasureMode (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_MEASURE_MODE;
   Msg.pMessageData = (PUBYTE) AllocateMemory(sizeof(MSG_DATA_MEASURE_MODE));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_MEASURE_MODE *) Msg.pMessageData)->MeasureMode = ProcessM_GetMeasureMode();

      (void)OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX);
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgClearAllDataFiles
///
/// @brief Handler for the message MSG_CLEAR_ALL_DATA_FILES
///
/// This is the handler for the message MSG_CLEAR_ALL_DATA_FILES.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgClearAllDataFiles (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_FORMAT_FILE_SYSTEM;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage( TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX );
}

//-----------------------------------------------------------------------------
//  Function ProcessM_SetMeasureMode 
///
/// @brief Set the measure mode
///
/// @param MeasureMode (@c [in] MEASURE_MODE) - New measurement mode value
///
/// This function sets the measure mode.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_SetMeasureMode (MEASURE_MODE MeasureMode)
{
   OS_MutexTake (TASK_ID_PROCESS_M, WAIT_MAX);

   m_MeasureMode = MeasureMode;

   // recalculate the measure timeout value
   OS_DisableInterrupts();

   if (m_MeasureMode == MM_NORMAL)
   {
      m_MeasureTimeout = (m_ConfigData.MeasurePeriodNormal / 10) - 1;
   }
   else
   {
      m_MeasureTimeout = (m_ConfigData.MeasurePeriodLogging / 10) - 1;
   }

   OS_EnableInterrupts();

   OS_MutexGive (TASK_ID_PROCESS_M);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_GetMeasureMode 
///
/// @brief Gets the measure mode
///
/// @return MEASURE_MODE Actual measurement mode
///
/// This function gets the measure mode.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
MEASURE_MODE ProcessM_GetMeasureMode (void)
{
   MEASURE_MODE MeasureMode;

   OS_MutexTake (TASK_ID_PROCESS_M, WAIT_MAX);

   MeasureMode = m_MeasureMode;

   OS_MutexGive (TASK_ID_PROCESS_M);

   return MeasureMode;
}

//-----------------------------------------------------------------------------
//  Function ProcessM_SetConfigData 
///
/// @brief Sets the config data
///
/// @param pConfigData (@c [in] MSG_DATA_DEVICE_CONFIG *) - New configuration data
///
/// This method sets the config data.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_SetConfigData (MSG_DATA_DEVICE_CONFIG * pConfigData)
{
   OS_MutexTake (TASK_ID_PROCESS_M, WAIT_MAX);

   m_ConfigData = *pConfigData;

   OS_MutexGive (TASK_ID_PROCESS_M);

   OS_DisableInterrupts();

   if (m_MeasureMode == MM_NORMAL)
   {
      m_MeasureTimeout = (m_ConfigData.MeasurePeriodNormal / 10) -1;
   }
   else
   {
      m_MeasureTimeout = (m_ConfigData.MeasurePeriodLogging / 10) -1;
   }

   OS_EnableInterrupts();
}

//-----------------------------------------------------------------------------
//  Function ProcessM_GetConfigData 
///
/// @brief Gets the config data
///
/// @param pConfigData (@c [out] MSG_DATA_DEVICE_CONFIG *) - Buffer for getting the config data
///
/// This function gets the config data.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_GetConfigData (MSG_DATA_DEVICE_CONFIG * pConfigData)
{
   if (pConfigData == NULL)
      return;

   OS_MutexTake (TASK_ID_PROCESS_M, WAIT_MAX);

   pConfigData->MeasurePeriodNormal  = m_ConfigData.MeasurePeriodNormal;
   pConfigData->MeasurePeriodLogging = m_ConfigData.MeasurePeriodLogging;

   OS_MutexGive (TASK_ID_PROCESS_M);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgNumOfFilesInFFS 
///
/// @brief Handler for the message MSG_NUM_OF_FILES_IN_FFS
///
/// @param pNumOfFilesInFFS (@c [in] MSG_DATA_NUM_OF_FILES_IN_FFS *) - Message data
///
/// This function is the handler for the message MSG_NUM_OF_FILES_IN_FFS.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgNumOfFilesInFFS (MSG_DATA_NUM_OF_FILES_IN_FFS * pNumOfFilesInFFS)
{
   m_ProcessM_NumOfFilesInFFS = pNumOfFilesInFFS->NumOfFilesInFFS;
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgReqFFSFile 
///
/// @brief Handler for the message MSG_REQ_FFS_FILE
///
/// @param pReqFFSFile (@c [in] MSG_DATA_REQ_MEASURE_FILE *) - Message data
///
/// This is the handler for the message MSG_REQ_FFS_FILE.
///
/// @status  Draft
/// @date    2013-07-18
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-18 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgReqFFSFile (MSG_DATA_REQ_MEASURE_FILE * pReqFFSFile)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_REQ_FFS_FILE;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_REQ_FFS_FILE));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_REQ_FFS_FILE *) Msg.pMessageData)->FileNumber   = pReqFFSFile->FileNumber;
      ((MSG_DATA_REQ_FFS_FILE *) Msg.pMessageData)->MaxBlockSize = pReqFFSFile->MaxBlockSize;

      (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgStartFFSFileTransfer 
///
/// @brief Handler for the message MSG_START_FFS_FILE_TRANSFER
///
/// @param pStartFFSFileTransfer (@c [in] MSG_DATA_START_FFS_FILE_TRANSFER *) - Message data
///
/// Thisis the handler for the message MSG_START_FFS_FILE_TRANSFER.
///
/// @status  Draft
/// @date    2013-07-18
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-18 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgStartFFSFileTransfer (MSG_DATA_START_FFS_FILE_TRANSFER * pStartFFSFileTransfer)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_START_FFS_FILE_TRANSFER;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_START_MEASURE_FILE_TRANSFER));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_START_MEASURE_FILE_TRANSFER *) Msg.pMessageData)->TotalNumOfDataBlocks = pStartFFSFileTransfer->TotalNumOfDataBlocks;
      ((MSG_DATA_START_MEASURE_FILE_TRANSFER *) Msg.pMessageData)->NumOfDataInFile      = pStartFFSFileTransfer->NumOfDataInFile;

      (void)OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX);
   }
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgFFSFileBlock 
///
/// @brief Handler for the message MSG_FFS_FILE_BLOCK
///
/// @param pFFSFileBlock (@c [in] MSG_DATA_FFS_FILE_BLOCK *) - Message data
///
/// This is the handler for the message MSG_FFS_FILE_BLOCK.
///
/// @status  Draft
/// @date    2013-07-18
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-18 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgFFSFileBlock (MSG_DATA_FFS_FILE_BLOCK * pFFSFileBlock)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_FFS_FILE_BLOCK;
   Msg.pMessageData = (PUBYTE) AllocateMemory(sizeof(MSG_DATA_MEASURE_FILE_BLOCK));

   if (Msg.pMessageData != NULL)
   {
      ((MSG_DATA_MEASURE_FILE_BLOCK *) Msg.pMessageData)->MemoryHandle           = pFFSFileBlock->MemoryHandle;
      ((MSG_DATA_MEASURE_FILE_BLOCK *) Msg.pMessageData)->SizeOfMeasureFileBlock = pFFSFileBlock->SizeOfFFSFileBlock;

      (void)OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX);
   }

}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgGetNextFFSFileBlock 
///
/// @brief Handler for the message MSG_GET_NEXT_FFS_FILE_BLOCK
///
/// This is the handler for the message MSG_GET_NEXT_FFS_FILE_BLOCK.
///
/// @status  Draft
/// @date    2013-07-18
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-18 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgGetNextFFSFileBlock (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_GET_NEXT_FFS_FILE_BLOCK;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage (TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX);
}

//-----------------------------------------------------------------------------
//  Function ProcessM_MsgFFSFileEof 
///
/// @brief Handler for the message MSG_FFS_FILE_EOF
///
/// This is the handler for the message MSG_FFS_FILE_EOF.
///
/// @status  Draft
/// @date    2013-07-18
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-18 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ProcessM_MsgFFSFileEof (void)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_FFS_FILE_EOF;
   Msg.pMessageData = NULL;

   (void)OS_SendMessage (TASK_ID_COM_M, &Msg, WAIT_MAX);
}
