//-----------------------------------------------------------------------------
//  OS.c
/// @file
///
/// @brief Module with wrapper function for the operating system FreeRTOS(osal)
///
/// This module contains all wrapper function for the operating system FreeRTOS.
///
/// @copyright 2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard include files
//-----------------------------------------------------------------------------
#include "FreeRTOS.h"
#include "task.h"
#include "queue.h"
#include "semphr.h"

//-----------------------------------------------------------------------------
// Project specific include files
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "OS.h"

#include "ComM.h"
#include "ProcessM.h"
#include "MeasureM.h"
#include "LedM.h"
#include "FlashMemM.h"

//-----------------------------------------------------------------------------
// Privat definitions, macros and constants
//-----------------------------------------------------------------------------
/// definition for the structure for tasks in the system
typedef struct tagMDL_TASK
{
   pdTASK_CODE                TaskCode;         ///< entryfunction for the task
   const signed char * const  pTaskName;        ///< name of the task
   unsigned short             StackSize;        ///< stacksize of the task
   void *                     pParameter;       ///< parameter for the task
   unsigned portBASE_TYPE     Priority;         ///< task priority
   xTaskHandle                TaskHandle;       ///< task handle
   unsigned short             QueueSize;        ///< queue size for the task
   xQueueHandle               QueueHandle;      ///< queue handle
   unsigned char              WithSemaphore;    ///< flag task has a semaphore
   xSemaphoreHandle           SemaphoreHandle;  ///< semaphore handle
   unsigned char              WithMutex;        ///< flag task has a mutex
   xSemaphoreHandle           MutexHandle;      ///< mutex handle
} MDL_TASK;

//-----------------------------------------------------------------------------
// Private data types
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// External references
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global member for the modul
//-----------------------------------------------------------------------------
/// task structure of the application
MDL_TASK m_MDLTasks[] = 
   { { ProcessM_TaskFunction,  (const signed char * const) "ProcessM", 128, (void *) 111, 1U, NULL, 10, NULL, 0, NULL, 1, NULL },
     { MeasureM_TaskFunction,  (const signed char * const) "MeasureM", 128, (void *)  50, 2U, NULL, 0,  NULL, 1, NULL, 1, NULL },
     { LedM_TaskFunction,      (const signed char * const) "LedM",     128, (void *) 121, 1U, NULL, 10, NULL, 1, NULL, 0, NULL },
     { ComM_TaskFunction,      (const signed char * const) "ComM",     128, (void *)  50, 1U, NULL, 10, NULL, 0, NULL, 0, NULL },
     { FlashMemM_TaskFunction, (const signed char * const) "FlMemM",   128, (void *)  50, 1U, NULL, 10, NULL, 0, NULL, 0, NULL }
};

//-----------------------------------------------------------------------------
// Prototypes of private functions
//-----------------------------------------------------------------------------
// Create a task system
UBYTE OS_CreateTaskSystem (TASK_ID TaskID);

//-----------------------------------------------------------------------------
// Implementation
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function  OS_StartOS
///
/// @brief Startfunction for the os
///
/// This is the startfunction for the os. Please do all os specific
/// initialiisations in this function.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void OS_StartOS( void )
{
}

//-----------------------------------------------------------------------------
//  Function  OS_CreateAllTasks
///
/// @brief Creating all tasks for the application
///
/// This function creates and starts all tasks for the application.
///
/// @see OS_StartOS()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void OS_CreateAllTasks( void )
{
   OS_CreateTaskSystem(TASK_ID_PROCESS_M);
   OS_CreateTaskSystem(TASK_ID_MEASURE_M);
   OS_CreateTaskSystem(TASK_ID_LED_M);
   OS_CreateTaskSystem(TASK_ID_COM_M);
   OS_CreateTaskSystem(TASK_ID_FLASHMEM_M);
}

//-----------------------------------------------------------------------------
//  Function  OS_CreateTaskSystem
///
/// @brief Create a task system
///
/// @param TaskID (@c [in] TASK_ID) - ID of the task
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_OK - Functionstate ok
///                  ERR_OS_TASK_NOT_CREATED - Task cannot created
///                  ERR_OS_QUEUE_NOT_CREATED - Queue cannot created
///                  ERR_OS_SEMAPHORE_NOT_CREATED - Semaphore cannot created
///                  ERR_OS_MUTEX_NOT_CREATED - Mutex cannot created
///
/// This function create a task system (task + queue + semaphore + mutex)
/// appropriate of the definition in the m_MDLTasks table.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_CreateTaskSystem (TASK_ID TaskID)
{
   OS_STATE OSState = ERR_OS_OK;

   if (xTaskCreate( m_MDLTasks[TaskID].TaskCode,
                    m_MDLTasks[TaskID].pTaskName,
                    m_MDLTasks[TaskID].StackSize,
                    m_MDLTasks[TaskID].pParameter,
                    m_MDLTasks[TaskID].Priority,
                    &m_MDLTasks[TaskID].TaskHandle ) != pdPASS)
   {
      return ERR_OS_TASK_NOT_CREATED;
   }

   if( m_MDLTasks[TaskID].QueueSize != 0 )
   {
      m_MDLTasks[TaskID].QueueHandle = xQueueCreate (m_MDLTasks[TaskID].QueueSize, sizeof( MESSAGE ));

      if (m_MDLTasks[TaskID].QueueHandle == NULL)
      {
         OSState = ERR_OS_QUEUE_NOT_CREATED;
      }
   }

   if (m_MDLTasks[TaskID].WithSemaphore == 1)
   {
      vSemaphoreCreateBinary (m_MDLTasks[TaskID].SemaphoreHandle);

      if (m_MDLTasks[TaskID].SemaphoreHandle == NULL)
      {
         OSState = ERR_OS_SEMAPHORE_NOT_CREATED;
      }
   }
   
   if (m_MDLTasks[TaskID].WithMutex == 1)
   {
      m_MDLTasks[TaskID].MutexHandle = xSemaphoreCreateMutex();
      
      if (m_MDLTasks[TaskID].MutexHandle == NULL)
      {
         OSState = ERR_OS_MUTEX_NOT_CREATED;
      }
   }

   return OSState;  
}

//-----------------------------------------------------------------------------
//  Function  OS_SendMessage
///
/// @brief Send a message to a task
///
/// @param TaskID (@c [in] TASK_ID) - Id of the receiver task
///
/// @param pMessage (@c [in] MESSAGE *) - Message
///
/// @param TicksToWait (@c [in] UDWORD) - Waittime for the queue
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_FULL - Queue is full
///                  ERR_OS_TASK_HAS_NO_MSG_QUEUE - The receiver task has no queue.
///
/// This function send a message to a task.
///
/// @see OS_SendMessageNB()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_SendMessage (TASK_ID TaskID, MESSAGE * pMessage, UDWORD TicksToWait)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].QueueHandle != NULL)
   {
      if( xQueueSend (m_MDLTasks[TaskID].QueueHandle, pMessage, TicksToWait ) != pdPASS)
      {
         OSState = ERR_OS_QUEUE_FULL;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MSG_QUEUE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function  OS_SendMessageNB
///
/// @brief Send a message to a task (nonblocked)
///
/// @param TaskID (@c [in] TASK_ID) - Id of the receiver task
///
/// @param pMessage (@c [in] MESSAGE *) - Message
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_FULL - Queue is full
///                  ERR_OS_TASK_HAS_NO_MSG_QUEUE - The receiver task has no queue.
///
/// This function send a message to a task. If the queue is full the call return
/// immediately (nonblocking). Use this function for interrupts.
///
/// @see OS_SendMessage()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_SendMessageNB (TASK_ID TaskID, MESSAGE * pMessage)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].QueueHandle != NULL)
   {
      if (xQueueSendFromISR (m_MDLTasks[TaskID].QueueHandle, pMessage, pdFALSE) != pdPASS)
      {
         OSState = ERR_OS_QUEUE_FULL;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MSG_QUEUE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function  OS_ReceiveMessage
///
/// @brief Receive a Message from the message queue
///
/// @param TaskID (@c [in] TASK_ID) - Id of the receiver task
///
/// @param pMessage (@c [in] MESSAGE *) - pointer to the memory for the received message
///
/// @param TicksToWait (@c [in] UDWORD) - Waittime for a message
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_EMPTY - Queue is empty
///                  ERR_OS_TASK_HAS_NO_MSG_QUEUE - The receiver task has no queue.
///
/// This function receive a Message from the message queue.
///
/// @see OS_ReceiveMessageNB()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_ReceiveMessage (TASK_ID TaskID, MESSAGE * pMessage, UDWORD TicksToWait)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].QueueHandle != NULL)
   {
      if (xQueueReceive (m_MDLTasks[TaskID].QueueHandle, pMessage, TicksToWait) != pdPASS)
      {
         OSState = ERR_OS_QUEUE_EMPTY;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MSG_QUEUE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function OS_ReceiveMessageNB
///
/// @brief Receive a Message from the message queue (nonblocking)
///
/// @param TaskID (@c [in] TASK_ID) - Id of the receiver task
///
/// @param pMessage (@c [in] MESSAGE *) - pointer to the memory for the received message
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_EMPTY - Queue is empty
///                  ERR_OS_TASK_HAS_NO_MSG_QUEUE - The receiver task has no queue.
///
/// This function receive a Message from the message queue. If no message is
/// in the queue, the function will return immediately. Use this function for
/// interrupts.
///
/// @see OS_ReceiveMessage()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_ReceiveMessageNB (TASK_ID TaskID, MESSAGE * pMessage)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].QueueHandle != NULL)
   {
      if( xQueueReceiveFromISR (m_MDLTasks[TaskID].QueueHandle, pMessage, pdFALSE) != pdPASS)
      {
         OSState = ERR_OS_QUEUE_EMPTY;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MSG_QUEUE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function OS_SemaphoreGiveFromISR
///
/// @brief Give a semaphore from an ISR (nonblocking)
///
/// @param TaskID (@c [in] TASK_ID) - Id of the task for which is the semaphore defined 
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_FULL - Queue is full
///                  ERR_OS_TASK_HAS_NO_SEMAPHORE - The task with the task id has no semaphore.
///
/// This function gives a semaphore from an ISR. This method is nonblocking and
/// should be used for ISRs.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_SemaphoreGiveFromISR (TASK_ID TaskID)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].SemaphoreHandle != NULL)
   {
      if (xSemaphoreGiveFromISR (m_MDLTasks[TaskID].SemaphoreHandle, pdFALSE) != pdTRUE)
      {
         OSState = ERR_OS_QUEUE_FULL;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_SEMAPHORE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function OS_SemaphoreTake
///
/// @brief Takes a semaphore
///
/// @param TaskID (@c [in] TASK_ID) - Id of the task for which is the semaphore defined
///
/// @param BlockTime (@c [in] UDWORD) - Waittime
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_SEMAPHORE_TAKE_FAILED - Semaphore is not give
///                  ERR_OS_TASK_HAS_NO_SEMAPHORE - The task with the task id has no semaphore.
///
/// This function takes a semaphore.
///
/// @see OS_SemaphoreGiveFromISR()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_SemaphoreTake (TASK_ID TaskID, UDWORD BlockTime)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].SemaphoreHandle != NULL)
   {
      if( xSemaphoreTake (m_MDLTasks[TaskID].SemaphoreHandle, BlockTime) != pdPASS)
      {
         OSState = ERR_OS_SEMAPHORE_TAKE_FAILED;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_SEMAPHORE;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function OS_MutexGive
///
/// @brief Gives a mutex
///
/// @param TaskID (@c [in] TASK_ID) - Id of the task for which is the mutex defined 
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_QUEUE_FULL - Queue is full
///                  ERR_OS_TASK_HAS_NO_MUTEX - The task with the task id has no mutex.
///
/// This function gives a Gives a mutex.
///
/// @see OS_MutexTake()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_MutexGive (TASK_ID TaskID)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].MutexHandle != NULL)
   {
      if (xSemaphoreGive (m_MDLTasks[TaskID].MutexHandle) != pdTRUE)
      {
         OSState = ERR_OS_QUEUE_FULL;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MUTEX;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function OS_MutexTake
///
/// @brief Takes a mutex
///
/// @param TaskID (@c [in] TASK_ID) - Id of the task for which is the mutex defined
///
/// @param BlockTime (@c [in] UDWORD) - Waittime
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_MUTEX_TAKE_FAILED - Mutex is not give
///                  ERR_OS_TASK_HAS_NO_MUTEX - The task with the task id has no
///                  mutex.
///
/// This function takes a mutex.
///
/// @see OS_MutexGive()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_MutexTake (TASK_ID TaskID, UDWORD BlockTime)
{
   OS_STATE OSState = ERR_OS_OK;

   if (m_MDLTasks[TaskID].MutexHandle != NULL)
   {
      if (xSemaphoreTake( m_MDLTasks[TaskID].MutexHandle, BlockTime ) != pdPASS)
      {
         OSState = ERR_OS_MUTEX_TAKE_FAILED;
      }
   }
   else
   {
      OSState = ERR_OS_TASK_HAS_NO_MUTEX;
   }

   return OSState;
}

//-----------------------------------------------------------------------------
//  Function  Sleep
///
/// @brief Sleepfunction
///
/// @param Time_ms (@c [in] UDWORD) - Sleeptime in ms
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_OK - Function is processed ok
///
/// This is a sleepfunction.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_Sleep (UDWORD Time_ms)
{
   vTaskDelay (Time_ms / portTICK_RATE_MS);
   
   return ERR_OS_OK;
}

//-----------------------------------------------------------------------------
//  Function OS_EnableInterrupts
///
/// @brief Enable the interrupts
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_OK - Function is processed ok
///
/// This function enable the interrupts.
///
/// @see OS_DisableInterrupts()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_EnableInterrupts()
{
   taskENABLE_INTERRUPTS();
   
   return ERR_OS_OK;
}

//-----------------------------------------------------------------------------
//  Function OS_DisableInterrupts
///
/// @brief Disable the interrupts
///
/// @return OS_STATE OS-Functionstate:
///                  ERR_OS_OK - Function is processed ok
///
/// This function disable the interrupts.
///
/// @see OS_DisableInterrupts()
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
OS_STATE OS_DisableInterrupts()
{
   taskDISABLE_INTERRUPTS();
   
   return ERR_OS_OK;
}
