//-----------------------------------------------------------------------------
//  LedM.c
/// @file LedM.c
///
/// @brief Implements a simple led user interface
///
/// A simple led user interface is implementet in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "null.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "OS.h"
#include "MessageDefs.h"
#include "Tools.h"

#include "LEDGreen.h"
#include "LEDRed.h"
#include "LEDYellow.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------

/// Defintion of the structure for the led controlling
typedef struct tagLEDM_LED_CONTROL
{
   LED_STATE      LEDState;               ///< Led state
   LED_STATE      ActLEDState;            ///< Actual led state
   UBYTE          LEDFlashFreq;           ///< Led flash frequency
   UBYTE          NumOfFlashes;           ///< Number of led flashes 
   UBYTE          ActNumOfFlashes;        ///< Actual number of flashes
   UBYTE          Tmp;                    ///< Auxilary variable
} LEDM_LED_CONTROL;

/// Maximum value for displaying altitude or pressure values
#define LEDM_SHOW_VALUE_START_VAL      100000
/// Number of position for showing a value
#define LEDM_SHOW_VALUE_NUM_OF_POS          6

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Led controlling member for the 3 led red, green and yellow
LEDM_LED_CONTROL m_LEDControl[LN_LAST+1];

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------
// Handler for the message with the id MSG_INIT_SYSTEM
void LedM_MsgInitSystem (void);
// Handler for the message with the id MSG_SHOW_VALUE
void LedM_MsgShowValue (MSG_DATA_SHOW_VALUE * pMsgDataShowValue);
// Handler for the message with the id MSG_SET_LED_JOB
void LedM_MsgSetLEDJob (MSG_DATA_SET_LED_JOB * pMsgDataSetLEDJob);
// Init an led job
void LedM_InitLEDJob (LED_NAME LEDName, LED_STATE LEDJob, UBYTE NumOfFlashes, UBYTE FlashFrequency);
// Kills an led job
void LedM_KillLEDJob (LED_NAME LEDName);

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function LedM_TaskFunction 
///
/// @brief Taskfunction of the simple user interface task
///
/// @param pvParameters (@c [in] void *) - Taskparameter (unused)
///
/// This is the taskfunction of the simple user interface task.
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_TaskFunction (void * pvParameters)
{
   MESSAGE Message;

   while(1)
   {
      if( OS_ReceiveMessage (TASK_ID_LED_M, &Message, WAIT_MAX) == ERR_OS_OK)
      {
         switch (Message.MessageID)
         {
            case MSG_INIT_SYSTEM:   LedM_MsgInitSystem();                                                break;
            case MSG_SHOW_VALUE:    LedM_MsgShowValue ((MSG_DATA_SHOW_VALUE *) Message.pMessageData);    break;
            case MSG_SET_LED_JOB:   LedM_MsgSetLEDJob ((MSG_DATA_SET_LED_JOB *) Message.pMessageData);   break;
            default:
               break;
         }

         // Speicher wenn notwendig freigeben
         if( Message.pMessageData != NULL )
         {
            FreeMemory( Message.pMessageData );
         }
      }
   }

}

//-----------------------------------------------------------------------------
//  Function LedM_MsgInitSystem 
///
/// @brief Handler for the message with the id MSG_INIT_SYSTEM
///
/// This function is the handler for the message with the id MSG_INIT_SYSTEM.
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_MsgInitSystem (void)
{
   m_LEDControl[LN_RED].LEDState           = LS_OFF;
   m_LEDControl[LN_RED].NumOfFlashes       = 0;
   m_LEDControl[LN_RED].ActNumOfFlashes    = 0;
   m_LEDControl[LN_RED].LEDFlashFreq       = 0;
   m_LEDControl[LN_RED].Tmp                = 0;

   m_LEDControl[LN_GREEN].LEDState         = LS_OFF;
   m_LEDControl[LN_GREEN].NumOfFlashes     = 0;
   m_LEDControl[LN_GREEN].ActNumOfFlashes  = 0;
   m_LEDControl[LN_GREEN].LEDFlashFreq     = 0;
   m_LEDControl[LN_GREEN].Tmp              = 0;

   m_LEDControl[LN_YELLOW].LEDState        = LS_OFF;
   m_LEDControl[LN_YELLOW].NumOfFlashes    = 0;
   m_LEDControl[LN_YELLOW].ActNumOfFlashes = 0;
   m_LEDControl[LN_YELLOW].LEDFlashFreq    = 0;
   m_LEDControl[LN_YELLOW].Tmp             = 0;
}

//-----------------------------------------------------------------------------
//  Function LedM_MsgShowValue 
///
/// @brief Handler for the message with the id MSG_SHOW_VALUE
///
/// @param pMsgDataShowValue (@c [in] MSG_DATA_SHOW_VALUE *) - Data for the command
///
/// This function is the handler for the message with the id MSG_SHOW_VALUE.
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_MsgShowValue (MSG_DATA_SHOW_VALUE * pMsgDataShowValue)
{   
   UDWORD   IntValue;
   UDWORD   ValPos;
   UBYTE    NumOfFlashes;
   UBYTE    iCurPos;
   UBYTE    FirstNum;
   
   if (pMsgDataShowValue == NULL)
      return;
      
   IntValue = pMsgDataShowValue->Value * 10;

   ValPos = LEDM_SHOW_VALUE_START_VAL;

   FirstNum = 0;

   for (iCurPos = 0; iCurPos < LEDM_SHOW_VALUE_NUM_OF_POS; iCurPos++)
   {
      if (IntValue >= ValPos)
      {
         NumOfFlashes = IntValue / ValPos;

         LedM_InitLEDJob (LN_YELLOW, LS_FLASH, NumOfFlashes, 2);

         OS_SemaphoreTake (TASK_ID_LED_M, WAIT_MAX);
         
         IntValue = IntValue - (NumOfFlashes * ValPos);
         
         FirstNum = 1;
      }
      
      // Trennzeichen oder Komma ausgeben
      if (FirstNum == 1)
      {
         if (ValPos > 1)
         {
            OS_Sleep (500);
         
            // Muss ein Trennzeichen oder ein Komma ausgegeben werden
            if (ValPos > 10)
            {
               // Trennzeichen
               LedM_InitLEDJob (LN_GREEN, LS_FLASH, 1, 2);

               OS_SemaphoreTake (TASK_ID_LED_M, WAIT_MAX);
            }
            else
            {
               // Komma
               LedM_InitLEDJob (LN_RED, LS_FLASH, 1, 2);

               OS_SemaphoreTake (TASK_ID_LED_M, WAIT_MAX);
            }
         
            OS_Sleep (500);
         }
      }

      ValPos /= 10;
   }
}

//-----------------------------------------------------------------------------
//  Function LedM_MsgSetLEDJob 
///
/// @brief Set a new Job for a single led
///
/// @param pMsgDataSetLEDJob (@c [in] MSG_DATA_SET_LED_JOB *) - Params for the new led job
///
/// This function sets w new Job for a single led.
///
/// @status  Draft
/// @date    2013-08-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-08-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_MsgSetLEDJob (MSG_DATA_SET_LED_JOB * pMsgDataSetLEDJob)
{
   if (pMsgDataSetLEDJob == NULL)
      return;

   LedM_InitLEDJob (pMsgDataSetLEDJob->LEDName,
                    pMsgDataSetLEDJob->LEDJob,
                    pMsgDataSetLEDJob->NumOfFlashes,
                    pMsgDataSetLEDJob->FlashFrequency);
}

//-----------------------------------------------------------------------------
//  Function LedM_InitLEDJob 
///
/// @brief Init an LED Job
///
/// @param LEDName (@c [in] LED_NAME) - Id of the led
///
/// @param LEDJob (@c [in] LED_STATE) - Led state
///
/// @param NumOfFlashes (@c [in] UBYTE) - Number of flash (only if the state is LED_LS_FLASH)
///
/// @param FlashFrequency (@c [in] UBYTE) - Flash frequency (only if the state is LED_LS_FLASH)
///
/// This function inits an led job.
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_InitLEDJob (LED_NAME LEDName, LED_STATE LEDJob, UBYTE NumOfFlashes, UBYTE FlashFrequency)
{
   if (LEDName > LN_LAST)
      return;

   OS_DisableInterrupts();

   // test to take the semaphore (it is possible that the semaphore is given by
   // setting an led in flash state, the flash state is leaving and nobody has
   // taken the semaphore)
   OS_SemaphoreTake (TASK_ID_LED_M, 1);

   // set the new led job
   m_LEDControl[LEDName].LEDState         = LEDJob;
   m_LEDControl[LEDName].ActLEDState      = LS_OFF;
   m_LEDControl[LEDName].NumOfFlashes     = NumOfFlashes;
   m_LEDControl[LEDName].ActNumOfFlashes  = 0;
   m_LEDControl[LEDName].LEDFlashFreq     = FlashFrequency;
   m_LEDControl[LEDName].Tmp              = 0;

   OS_EnableInterrupts();

   return;
}

//-----------------------------------------------------------------------------
//  Function LedM_KillLEDJob 
///
/// @brief Kills an led job
///
/// @param LEDName (@c [in] LED_NAME) - Id of the led for killing the job
///
/// This function kills an led job.
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_KillLEDJob (LED_NAME LEDName)
{
   if (LEDName > LN_LAST)
      return;

   OS_DisableInterrupts();

   m_LEDControl[LEDName].LEDState         = LS_OFF;
   m_LEDControl[LEDName].ActLEDState      = LS_OFF;
   m_LEDControl[LEDName].NumOfFlashes     = 0;
   m_LEDControl[LEDName].ActNumOfFlashes  = 0;
   m_LEDControl[LEDName].LEDFlashFreq     = 0;
   m_LEDControl[LEDName].Tmp              = 0;

   OS_EnableInterrupts();

   return;
}

//-----------------------------------------------------------------------------
//  Function LedM_CalcLEDState 
///
/// @brief Calculating the led states
///
/// This function is calculating the led state. The function must called from
/// an isr of a timer. Don't do any blocking actions in this function!
///
/// @see LedM_MsgShowValue()
///
/// @status  Draft
/// @date    2013-07-06
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-06 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void LedM_CalcLEDState (void)
{
   UWORD StateTime;
   UBYTE iTmp;
   
   for (iTmp = 0; iTmp <= LN_LAST; iTmp++)
   {
      if (m_LEDControl[iTmp].LEDState == LS_FLASH)
      {
         if (m_LEDControl[iTmp].NumOfFlashes != 0)
         {
            StateTime = 1000 / (m_LEDControl[iTmp].LEDFlashFreq * 2);
      
            if (m_LEDControl[iTmp].Tmp < StateTime / 10)
            {
               m_LEDControl[iTmp].Tmp++;
            }
            else
            {
               m_LEDControl[iTmp].Tmp = 0;
      
               if (m_LEDControl[iTmp].ActLEDState == LS_OFF)
               {
                  m_LEDControl[iTmp].ActLEDState = LS_ON;
               }
               else
               {
                  m_LEDControl[iTmp].ActLEDState = LS_OFF;
                  m_LEDControl[iTmp].ActNumOfFlashes++;

                  if (m_LEDControl[iTmp].ActNumOfFlashes == m_LEDControl[iTmp].NumOfFlashes)
                  {
                     LedM_KillLEDJob (iTmp);

                     OS_SemaphoreGiveFromISR (TASK_ID_LED_M);
                  }
               }
            }
         }
         else
         {
            // Normales Blinken bis in alle Ewigkeit
         }
      }
   }
   
   // Am Ende Berechnung den berechneten LED-Zustand ausgeben
   if (m_LEDControl[LN_RED].ActLEDState == LS_ON)
      LEDRed_On();
   else
      LEDRed_Off();
   
   if (m_LEDControl[LN_GREEN].ActLEDState == LS_ON)
      LEDGreen_On();
   else
      LEDGreen_Off();

   if (m_LEDControl[LN_YELLOW].ActLEDState == LS_ON)
      LEDYellow_On();
   else
      LEDYellow_Off();
}
