//-----------------------------------------------------------------------------
//  MPL3115A2.c
/// @file
///
/// @brief Driver for the altitude sensor MPL3115A2
///
/// This is the driver for the altitude sensor MPL3115A2.
///
/// @copyright 2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "Events.h"

#include "PE_Types.h"
#include "PE_Error.h"
#include "PE_Const.h"
#include "IO_Map.h"

#include "MPL3115A2.h"
#include "MPL3115A2_Defs.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------
/// Size for the output buffer
#define MPL3115A2_MAX_NUM_OF_IN_DATA      0x06
/// Size for the input buffer
#define MPL3115A2_MAX_NUM_OF_OUT_DATA     0x02

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------
/// Flag: Data is transmitted via I2C1 interface
extern volatile bool m_I2C1_DataTransmitted;
/// Flag: Data is received via I2C1 interface
extern volatile bool m_I2C1_DataReceived;
/// Flag: An error is occur on the I2C1 interface
extern volatile bool m_I2C1_Error;

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Handle to the I2C interface for the MPL31315A2
LDD_TDeviceData * m_pMPL3115A2_I2C_Handle;
/// Output data buffer
uint8_t OutData[MPL3115A2_MAX_NUM_OF_OUT_DATA] = {0x2A, 0x01};
/// Input data buffer
uint8_t InData[MPL3115A2_MAX_NUM_OF_IN_DATA];

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function MPL3115A2_Init
///
/// @brief Init the driver
///
/// @param pI2C_Handle (@c [in] LDD_TDeviceData *) - I2C handle
///
/// This function inits the driver.
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_Init (LDD_TDeviceData * pI2C_Handle)
{   
   UBYTE CtrlReg4;
   UBYTE CtrlReg5;
   UBYTE PtDataCfgReg;
   UBYTE IdReg;

   m_pMPL3115A2_I2C_Handle = pI2C_Handle;

//   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;

   // Who I am Register abfragen
   OutData[0] = MPL3115A2_REG_WHO_AM_I; 

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &IdReg, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   // Interrupt Enable Register konfigurieren
   CtrlReg4 = SET_INT_EN_DRDY_MASK;

   OutData[0] = MPL3115A2_REG_CTRL_REG4;
   OutData[1] = CtrlReg4;

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;   

   // Interrupt Configuration Register konfigurieren
   CtrlReg5 = SET_INT_CFG_DRDY_MASK;

   OutData[0] = MPL3115A2_REG_CTRL_REG5;
   OutData[1] = CtrlReg5;

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   // PT_DATA_CFG Register
   PtDataCfgReg = SET_PDEFE_MASK;

   OutData[0] = MPL3115A2_REG_PT_DATA_CFG;
   OutData[1] = PtDataCfgReg;

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_SetOversampleRatio
///
/// @brief Sets the oversampling ratio
///
/// @param OversampleRatio (@c [in] UBYTE) - New value for the oversampling ratio
///
/// This function sets the oversampling ratio value in controlregister 1. The
/// minimum measure time is depended on this setting.
///
/// oversampling   measurements   measure time in ms
/// ------------------------------------------------
///       0             1                  6
///       1             2                 10
///       2             4                 18
///       3             8                 34
///       4            16                 66
///       5            32                130
///       6            64                258
///       7           128                512
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_SetOversampleRatio( UBYTE OversampleRatio )
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   // Oversamplingwert bernehmen
   CtrlReg1 &= 0xC7;
   CtrlReg1 |= OversampleRatio << 3;

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   OutData[0] = MPL3115A2_REG_CTRL_REG1;
   OutData[1] = CtrlReg1;

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;   
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_Active
///
/// @brief Switches the MPL3115A2 in active mode
///
/// This function switches the MPL3115A2 in active mode.
///
/// @see MPL3115A2_Standby()
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_Active (void)
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // Active-Flag setzen
   CtrlReg1 |= ACTIVE_SBYB_MASK;
   
   OutData[0] = MPL3115A2_REG_CTRL_REG1;
   OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE; 
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_Standby
///
/// @brief Switches the MPL3115A2 in standby mode
///
/// This function switches the MPL3115A2 in standby mode.
///
/// @see MPL3115A2_Active()
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_Standby (void)
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // Active-Flag zurcknehmen
   CtrlReg1 &= STANDBY_SBYB_MASK;
   
   OutData[0] = MPL3115A2_REG_CTRL_REG1;
   OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_SetAltimeterMode
///
/// @brief Switches the MPL3115A2 in altitude measure mode
///
/// This function switches the MPL3115A2 in altitude measure mode.
///
/// @see MPL3115A2_SetBarometerMode()
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_SetAltimeterMode (void)
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // "Hhenmessermodus" einschalten
   CtrlReg1 |= ALTI_ALT_MASK;
   
   OutData[0] = MPL3115A2_REG_CTRL_REG1;
   OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE; 
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_SetBarometerMode
///
/// @brief Switches the MPL3115A2 in preassure measure mode
///
/// This function switches the MPL3115A2 in preassure measure mode.
///
/// @see MPL3115A2_SetAltimeterMode()
///
/// @status  Draft
/// @date    2013-01-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-01-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_SetBarometerMode (void)
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // "Luftdruckmessermodus" einschalten
   CtrlReg1 &= BARO_ALT_MASK;
   
   OutData[0] = MPL3115A2_REG_CTRL_REG1;
   OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE; 
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_InitiateMeasurementAndWaitToFinish
///
/// @brief Starts a new measuring an waits until the measuring is finished
///
/// @param fWithReset (@c [in] UBYTE) - Reset the OST-Bit
///
/// This function starts a new measuring an waits until the measuring is finished.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_InitiateMeasurementAndWaitToFinish( UBYTE fWithReset )
{
   UBYTE CtrlReg1;
   UBYTE StatusReg;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   // Neue Messung anstossen ...
   OutData[0] = MPL3115A2_REG_CTRL_REG1;

   if( fWithReset == TRUE )
   {
      // Wenn Resetflag gesetzt ist OST-Bit erst zurcksetzen
      CtrlReg1 &= RESET_OST_MASK;

      OutData[1] = CtrlReg1;

      while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
      while (!m_I2C1_DataTransmitted);  
      m_I2C1_DataTransmitted = FALSE;
   }

   CtrlReg1 |= SET_OST_MASK;

   OutData[1] = CtrlReg1;

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   // Warten bis die neuen Messwerte da sind ...
   do 
   {
      StatusReg = MPL3115A2_ReadStatus();
   } while (!(StatusReg & STATUS_PDR_MASK));
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_InitiateMeasurement
///
/// @brief Starts a new measuring on the MPL3115A2
///
/// This function starts a new measuring on the MPL3115A2.
///
/// @see MPL3115A2_InitiateMeasurementAndWaitToFinish()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_InitiateMeasurement( void )
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   // Neue Messung anstossen ...
   OutData[0] = MPL3115A2_REG_CTRL_REG1;

   CtrlReg1 |= SET_OST_MASK;

   OutData[1] = CtrlReg1;

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_WaitUntilMeasurementComplete
///
/// @brief Waits until a measuring is complete
///
/// This function waits until a measuring is complete.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_WaitUntilMeasurementComplete (void)
{
   UBYTE StatusReg;

   do 
   {
      StatusReg = MPL3115A2_ReadStatus();
   } while (!(StatusReg & STATUS_PDR_MASK));
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_IsMeasurementComplete 
///
/// @brief Check if a measurement is complete
///
/// @return bool TRUE - Measurement complete, FALSE - Measuring in process
///
/// This function checks if a measurement is complete.
///
/// @status  Draft
/// @date    2013-08-17
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-08-17 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MPL3115A2_IsMeasurementComplete (void)
{
   UBYTE StatusReg;

   StatusReg = MPL3115A2_ReadStatus();
   
   if (!(StatusReg & STATUS_PDR_MASK))
   {
      return FALSE;
   }
   else
   {
      return TRUE;
   }
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_ReqAltitude
///
/// @brief Request the altitude measurement value
///
/// This function request the altitude measurement value.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_ReqAltitude (void)
{
   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_OUT_P;
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;   
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_ReadAltitudeEx
///
/// @brief Reads the altitude and the temperature value
///
/// @param pAltitude (@c [out] float *) - New altitude value
///
/// @param pTemperature (@c [out] float *) - New temperature value
///
/// @return bool TRUE - Measurement value are ok, False - Measuring failed
///
/// This function reads the altitude and the temperature value.
///
/// @see MPL3115A2_ReqAltitude()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MPL3115A2_ReadAltitudeEx (float * pAltitude, float * pTemperature)
{
   bool RetVal = TRUE;
   float Tmp;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   if (pAltitude != NULL && pTemperature != NULL)
   {
      while (I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &InData, MPL3115A2_P_T_VALUE_SIZE, LDD_I2C_SEND_STOP) != ERR_OK);
      while (!m_I2C1_DataReceived);
      m_I2C1_DataReceived = FALSE;  

      Tmp = InData[1] | (InData[0] << 8);
      *pAltitude = Tmp + ((float)((InData[2] & 0xF0) >> 4)/MPL3115A2_A_DIVIDER);

      *pTemperature = InData[3] + ((float)(InData[4] >> 4)/MPL3115A2_T_DIVIDER);
   }
   else
   {
      RetVal = FALSE;
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_ReadAltitude
///
/// @brief Reads the altitude value
///
/// @param pAltitude (@c [out] float *) - New altitude value
///
/// @return bool TRUE - Measurement value are ok, False - Measuring failed
///
/// This function reads the altitude value.
///
/// @see MPL3115A2_ReadAltitudeEx()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MPL3115A2_ReadAltitude (float * pAltitude)
{
   bool RetVal = TRUE;
   float Tmp;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   if( pAltitude != NULL )
   {
      while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &InData, MPL3115A2_P_T_VALUE_SIZE, LDD_I2C_SEND_STOP) != ERR_OK);
      while (!m_I2C1_DataReceived);
      m_I2C1_DataReceived = FALSE;  

      Tmp = InData[1] | (InData[0] << 8);
      *pAltitude = Tmp + ((float)((InData[2] & 0xF0) >> 4)/MPL3115A2_A_DIVIDER);
   }
   else
   {
      RetVal = FALSE;
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_ReqPressure
///
/// @brief Request the pressure measurement value
///
/// This function request the pressure measurement value.
///
/// @see MPL3115A2_ReqAltitude()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MPL3115A2_ReqPressure( void )
{
   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_OUT_P;
   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;   
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_ReadPressureExt
///
/// @brief Reads the pressure and the temperature value
///
/// @param pPreasure (@c [out] float *) - New pressure value
///
/// @param pTemperature (@c [out] float *) - New temperature value
///
/// @return bool TRUE - Measurement value are ok, False - Measuring failed
///
/// This function reads the pressure and the temperature value.
///
/// @see MPL3115A2_ReadPressure()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MPL3115A2_ReadPressureExt (float * pPreasure, float * pTemperature)
{
   bool RetVal = TRUE;
   float Tmp;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   if( pPreasure != NULL && pTemperature != NULL )
   {
      while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &InData, MPL3115A2_P_T_VALUE_SIZE, LDD_I2C_SEND_STOP) != ERR_OK);
      while (!m_I2C1_DataReceived);
      m_I2C1_DataReceived = FALSE;  

      Tmp = (InData[2] >> 6) | (InData[1] << 2) | (InData[0] << 10);
      *pPreasure = Tmp + ((float)((InData[2] & 0x30) >> 4)/MPL3115A2_P_DIVIDER);

      *pTemperature = InData[3] + ((float)(InData[4] >> 4)/MPL3115A2_T_DIVIDER);
   }
   else
   {
      RetVal = FALSE;
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function  MPL3115A2_ReadPressure
///
/// @brief Reads the pressure value
///
/// @param pPreasure (@c [out] float *) - New pressure value
///
/// @return bool TRUE - Measurement value are ok, False - Measuring failed
///
/// This function reads the pressure value.
///
/// @see MPL3115A2_ReadPressureExt()
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MPL3115A2_ReadPressure (float * pPreasure)
{
   bool RetVal = TRUE;
   float Tmp;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   if( pPreasure != NULL )
   {
      while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &InData, MPL3115A2_P_VALUE_SIZE, LDD_I2C_SEND_STOP) != ERR_OK);
      while (!m_I2C1_DataReceived);
      m_I2C1_DataReceived = FALSE;  

      Tmp = (InData[2] >> 6) | (InData[1] << 2) | (InData[0] << 10);
      *pPreasure = Tmp + ((float)((InData[2] & 0x30) >> 4)/MPL3115A2_P_DIVIDER);
   }
   else
   {
      RetVal = FALSE;
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function MPL3115A2_ReadStatus
///
/// @brief Reads the status register of the MPL3115A2
///
/// This function reads the status register of the MPL3115A2.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE MPL3115A2_ReadStatus( void )
{
   UBYTE StatusReg;

   I2C1_SelectSlaveDevice(m_pMPL3115A2_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, MPL3115A2_SLAVE_ADDRESS);

   OutData[0] = MPL3115A2_REG_DR_STATUS; 

   while(I2C1_MasterSendBlock(m_pMPL3115A2_I2C_Handle, &OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
 
   while(I2C1_MasterReceiveBlock(m_pMPL3115A2_I2C_Handle, &StatusReg, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   return StatusReg;
}
