//-----------------------------------------------------------------------------
//  CpuRegHAL.c
/// @file
///
/// @brief Abstractionlayer for the cpu registers
///
/// The abstractionlayer for the cpu registers is implementet in this modul.
///
/// @copyright 2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard include files
//-----------------------------------------------------------------------------
#include "stddef.h"

//-----------------------------------------------------------------------------
// Project specific include files
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "IO_Map.h"

//-----------------------------------------------------------------------------
// Privat definitions, macros and constants
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Private data types
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// External references
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global member for the modul
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Prototypes of private functions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Implementation
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function CRHal_GetPITCounterVal
///
/// @brief Read the tickcountvalue 1 register of the PIT
///
/// @return UDWORD Tickcountvalue 1 of the PIT
///
/// This function read the tickcountvalue 1 register of the PIT.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UDWORD CRHal_GetPITCounterVal ()
{
   return PIT_CVAL1;
}

//-----------------------------------------------------------------------------
//  Function CRHal_GetCPUDeviceID 
///
/// @brief Read the device id register of the cpu
///
/// @return UDWORD device id of the cpu
///
/// This function read the device id register of the cpu.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UDWORD CRHal_GetCPUDeviceID ()
{
   return SIM_SDID;
}

//-----------------------------------------------------------------------------
//  Function CRHal_GetCPUUniqueID 
///
/// @brief Read the unique id register of the cpu
///
/// @param pBuffer (@c [in] PUBYTE) - pointer to the memory for the unique id
///
/// This function read the unique id register of the cpu.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void CRHal_GetCPUUniqueID (PUBYTE pBuffer)
{
   UDWORD Tmp;
   
   if (pBuffer != NULL)
   {
      Tmp = SIM_UIDH;
      pBuffer[15] = (UBYTE)((Tmp >> 24) & 0xFF);
      pBuffer[14] = (UBYTE)((Tmp >> 16) & 0xFF);
      pBuffer[13] = (UBYTE)((Tmp >> 8) & 0xFF);
      pBuffer[12] = (UBYTE)(Tmp & 0xFF);

      Tmp = SIM_UIDMH;
      pBuffer[11] = (UBYTE)((Tmp >> 24) & 0xFF);
      pBuffer[10] = (UBYTE)((Tmp >> 16) & 0xFF);
      pBuffer[9]  = (UBYTE)((Tmp >> 8) & 0xFF);
      pBuffer[8]  = (UBYTE)(Tmp & 0xFF);

      Tmp = SIM_UIDML;
      pBuffer[7]  = (UBYTE)((Tmp >> 24) & 0xFF);
      pBuffer[6]  = (UBYTE)((Tmp >> 16) & 0xFF);
      pBuffer[5]  = (UBYTE)((Tmp >> 8) & 0xFF);
      pBuffer[4]  = (UBYTE)(Tmp & 0xFF);

      Tmp = SIM_UIDL;
      pBuffer[3]  = (UBYTE)((Tmp >> 24) & 0xFF);
      pBuffer[2]  = (UBYTE)((Tmp >> 16) & 0xFF);
      pBuffer[1]  = (UBYTE)((Tmp >> 8) & 0xFF);
      pBuffer[0]  = (UBYTE)(Tmp & 0xFF);      
   }
}

//-----------------------------------------------------------------------------
//  Function CRHal_EnableLowPowerModes 
///
/// @brief Enables the low power modes of the chip
///
/// This function enables the low power modes of the chip. The register can
/// write only once after any system reset.
///
/// @status  Draft
/// @date    2013-09-13
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-09-13 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void CRHal_EnableVeryLowPowerModes (void)
{
   SMC_PMPROT |= SMC_PMPROT_AVLP_MASK;
}
