//-----------------------------------------------------------------------------
//  DControlAPI.c
/// @file
///
/// @brief Protocolfunctions for DControlAPI
///
/// All function for DControlAPI are implemented in this module.
///
/// @copyright 2013 Sirona Dental Systems GmbH - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard include files
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "null.h"

//-----------------------------------------------------------------------------
// Project specific include files
//-----------------------------------------------------------------------------
#include "MessageDefs.h"
#include "DControlAPI.h"
#include "Tools.h"

#include "stdio.h"
#include "string.h"

//-----------------------------------------------------------------------------
// Privat definitions, macros and constants
//-----------------------------------------------------------------------------
/// Startvalue for the CRC calculation
#define CRC_START_VALUE                   0x00

// Offset values for API header
/// API header offset for identifier
#define OFFSET_IDENTIFIER                    0
/// API header offset for command id (hi byte)
#define OFFSET_CMD_ID_HI                     1
/// API header offset for command id (lo byte)
#define OFFSET_CMD_ID_LO                     2
/// API header offset for packet number (hi byte)
#define OFFSET_CUR_PACKET_NUM_HI             3
/// API header offset for packet number (lo byte)
#define OFFSET_CUR_PACKET_NUM_LO             4
/// API header offset for total number of packets (hi byte)
#define OFFSET_TOTAL_NUM_OF_PACKETS_HI       5
/// API header offset for total number of packets (lo byte)
#define OFFSET_TOTAL_NUM_OF_PACKETS_LO       6
/// API header offset for number of parameters in packet (hi byte)
#define OFFSET_NUM_OF_PARAMS_HI              7
/// API header offset for number of parameters in packet (lo byte)
#define OFFSET_NUM_OF_PARAMS_LO              8
/// API header offset for crc
#define OFFSET_HEADER_CRC                    9

// Offset values and size for Cmd CMD_ID_CONNECT
/// Cmd CMD_ID_CONNECT: Offset for device type
#define O_CMD_CONNECT_DEVICE_TYPE           10
/// Cmd CMD_ID_CONNECT: Offset for connection encryption
#define O_CMD_CONNECT_CON_ENCRYPTION        11
/// Cmd CMD_ID_CONNECT: Offset for maximal number of parameters (hi byte)
#define O_CMD_CONNECT_MAX_NUM_OF_PARAMS_HI  12
/// Cmd CMD_ID_CONNECT: Offset for maximal number of parameters (lo byte)
#define O_CMD_CONNECT_MAX_NUM_OF_PARAMS_LO  13
/// Cmd CMD_ID_CONNECT: Offset for crc
#define O_CMD_CONNECT_CRC                   14
/// Size of parameter for Cmd CMD_ID_CONNECT
#define SIZE_OF_CMD_CONNECT_DATA             5

// Offset values and size for Cmd CMD_ID_DEVICE_INFO 
/// Cmd CMD_ID_DEVICE_INFO: Offset for flash device type
#define O_CMD_DEV_INFO_FLASH_DEVICETYPE     10
/// Cmd CMD_ID_DEVICE_INFO: Offset for flash size
#define O_CMD_DEV_INFO_FLASH_SIZE           11
/// Cmd CMD_ID_DEVICE_INFO: Offset for cpu device id
#define O_CMD_DEV_INFO_CPU_DEVICEID         12
/// Cmd CMD_ID_DEVICE_INFO: Offset for cpu unique id
#define O_CMD_DEV_INFO_CPU_UNIQUEID         16
/// Cmd CMD_ID_DEVICE_INFO: Offset for software version
#define O_CMD_DEV_INFO_SW_VERSION           32
/// Cmd CMD_ID_DEVICE_INFO: Offset for software revision
#define O_CMD_DEV_INFO_SW_REVISION          33
/// Cmd CMD_ID_DEVICE_INFO: Offset for software subrevision
#define O_CMD_DEV_INFO_SW_SUBREVISION       34
/// Cmd CMD_ID_DEVICE_INFO: Offset for hardware version
#define O_CMD_DEV_INFO_HW_VERSION           35
/// Cmd CMD_ID_DEVICE_INFO: Offset for hardware revision
#define O_CMD_DEV_INFO_HW_REVISION          36
/// Cmd CMD_ID_DEVICE_INFO: Offset for firmware version
#define O_CMD_DEV_INFO_BT_FW_VERSION        37
/// Cmd CMD_ID_DEVICE_INFO: Offset for firmware revision
#define O_CMD_DEV_INFO_BT_FW_REVISION       38
/// Cmd CMD_ID_DEVICE_INFO: Offset for os type
#define O_CMD_DEV_INFO_OS_TYPE              39
/// Cmd CMD_ID_DEVICE_INFO: Offset for os version
#define O_CMD_DEV_INFO_OS_VERSION           40
/// Cmd CMD_ID_DEVICE_INFO: Offset for os revision
#define O_CMD_DEV_INFO_OS_REVISION          41
/// Cmd CMD_ID_DEVICE_INFO: Offset for os subrevision
#define O_CMD_DEV_INFO_OS_SUBREVISION       42
/// Cmd CMD_ID_DEVICE_INFO: Offset for number of files
#define O_CMD_DEV_INFO_NUM_OF_FILES         43
/// Cmd CMD_ID_DEVICE_INFO: Offset for crc
#define O_CMD_DEV_INFO_CRC                  45
/// Size of parameter for Cmd CMD_ID_DEVICE_INFO
#define SIZE_OF_CMD_DEVICE_INFO             36

// Offset values and size for Cmd CMD_ID_DEVICE_CONFIG
/// Cmd CMD_ID_DEVICE_CONFIG: Offset for measurement period normal mode (hi byte)
#define O_CMD_DEV_CONFIG_MPERIOD_NORMAL_HI  10
/// Cmd CMD_ID_DEVICE_CONFIG: Offset for measurement period normal mode (lo byte)
#define O_CMD_DEV_CONFIG_MPERIOD_NORMAL_LO  11
/// Cmd CMD_ID_DEVICE_CONFIG: Offset for measurement period logging mode (hi byte)
#define O_CMD_DEV_CONFIG_MPERIOD_LOGGING_HI 12
/// Cmd CMD_ID_DEVICE_CONFIG: Offset for measurement period logging mode (lo byte)
#define O_CMD_DEV_CONFIG_MPERIOD_LOGGING_LO 13
/// Cmd CMD_ID_DEVICE_CONFIG: Offset for crc
#define O_CMD_DEV_CONFIG_CRC                14
/// Size of parameter for Cmd CMD_ID_DEVICE_CONFIG
#define SIZE_OF_CMD_DEVICE_CONFIG            5

// Offset value and size for Cmd CMD_ID_FFS_FILE_HEADER
/// Cmd CMD_ID_FFS_FILE_HEADER: Offset for ffs file header number of data
#define O_CMD_FFS_FILE_HEADER_NUM_OF_DATA   10
/// Cmd CMD_ID_FFS_FILE_HEADER: Offset for crc
#define O_CMD_FFS_FILE_HEADER_CRC           14
/// Size of parameter for Cmd CMD_ID_FFS_FILE_HEADER
#define SIZE_OF_CMD_FFS_FILE_HEADER          5

// Offset values and size for Msg CMD_ID_MEASURE_MODE
/// Cmd CMD_ID_MEASURE_MODE: Offset for measure mode
#define O_CMD_MEASURE_MODE_MODE             10
/// Cmd CMD_ID_MEASURE_MODE: Offset for crc
#define O_CMD_MEASURE_MODE_CRC              11
/// Size of parameter for Cmd CMD_ID_MEASURE_MODE
#define SIZE_OF_CMD_MEASURE_MODE             2

// Offset value and size for Cmd CMD_ID_REQ_MEASURE_FILE
/// Cmd CMD_ID_REQ_MEASURE_FILE: Offset for request measure file
#define O_CMD_REQ_MEASURE_FILE              10
/// Cmd CMD_ID_REQ_MEASURE_FILE: Offset for crc
#define O_CMD_REQ_MEASURE_FILE_CRC          12
/// Size of parameter for Cmd CMD_ID_REQ_MEASURE_FILE
#define SIZE_OF_CMD_REQ_MEASURE_FILE         3

// Offset values and size fo Cmd CMD_ID_ACTUAL_MVALUES
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for measurements timestamp (hi byte)
#define O_CMD_ACT_MVALUES_TIMESTAMP_HI      10
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for measurements timestamp (lo byte)
#define O_CMD_ACT_MVALUES_TIMESTAMP_LO      14     
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual altitude in m
#define O_CMD_ACT_MVALUES_ACTUAL_ALTITUDE   18
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual pressure in pa
#define O_CMD_ACT_MVALUES_ACTUAL_PRESSURE   22
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual temperature in degree celcius
#define O_CMD_ACT_MVALUES_ACTUAL_TEMP       26
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual acceleration x direction in G
#define O_CMD_ACT_MVALUES_ACTUAL_ACC_X_DIR  30
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual acceleration y direction in G
#define O_CMD_ACT_MVALUES_ACTUAL_ACC_Y_DIR  34
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for actual acceleration z direction in G
#define O_CMD_ACT_MVALUES_ACTUAL_ACC_Z_DIR  38
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal altitude in m
#define O_CMD_ACT_MVALUES_MINIMAL_ALTITUDE  42
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal pressure in pa
#define O_CMD_ACT_MVALUES_MINIMAL_PRESSURE  46
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal temperature in degree celcius
#define O_CMD_ACT_MVALUES_MINIMAL_TEMP      50
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal acceleration x direction in G
#define O_CMD_ACT_MVALUES_MINIMAL_ACC_X_DIR 54
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal acceleration y direction in G
#define O_CMD_ACT_MVALUES_MINIMAL_ACC_Y_DIR 58
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for minimal acceleration z direction in G
#define O_CMD_ACT_MVALUES_MINIMAL_ACC_Z_DIR 62
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal altitude in m
#define O_CMD_ACT_MVALUES_MAXIMAL_ALTITUDE  66
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal pressure in pa
#define O_CMD_ACT_MVALUES_MAXIMAL_PRESSURE  70
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal temperature in degree celcius
#define O_CMD_ACT_MVALUES_MAXIMAL_TEMP      74
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal acceleration x direction in G
#define O_CMD_ACT_MVALUES_MAXIMAL_ACC_X_DIR 78
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal acceleration y direction in G
#define O_CMD_ACT_MVALUES_MAXIMAL_ACC_Y_DIR 82
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for maximal acceleration z direction in G
#define O_CMD_ACT_MVALUES_MAXIMAL_ACC_Z_DIR 86
/// Cmd CMD_ID_ACTUAL_MVALUES: Offset for crc
#define O_CMD_ACT_MVALUES_CRC               90
/// Size of parameter for Cmd CMD_ID_ACTUAL_MVALUES
#define SIZE_OF_CMD_ACTUAL_MVALUES          81

//-----------------------------------------------------------------------------
// Private data types
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// External references
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Global member for the modul
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Prototypes of private functions
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Implementation
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function BaroALtiAPI_GetCmdHeader
///
/// @brief Extract the command header from the raw data
///
/// @param pMsgRawData (@c [in] PUBYTE) - Raw data
///
/// @param pCmdHeader (@c [out] CMD_HEADER *) - Command header structure
///
/// @return UBYTE Return code
///
/// This function extracts the command header from the raw data.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroALtiAPI_GetCmdHeader( PUBYTE pMsgRawData, CMD_HEADER * pCmdHeader )
{
   UBYTE State = DC_API_OK;
   UBYTE CheckCRCValue = 0;

   if( pMsgRawData != NULL && pCmdHeader != NULL )
   {
      if( pMsgRawData[0] == '#' )
      {
         CheckCRCValue = CalculateCRC8(pMsgRawData,0,9,CheckCRCValue);

         if (CheckCRCValue != pMsgRawData[9])
         {
            State = DC_API_CRC_VALUE_FALSE;
         }
         else
         {
            pCmdHeader->Identifier = pMsgRawData[0];

            pCmdHeader->CommandID = pMsgRawData[1] << 8;
            pCmdHeader->CommandID |= pMsgRawData[2];

            pCmdHeader->ActPacketNumber = pMsgRawData[3] << 8;
            pCmdHeader->ActPacketNumber |= pMsgRawData[4];

            pCmdHeader->TotalPacketNumber = pMsgRawData[5] << 8;
            pCmdHeader->TotalPacketNumber |= pMsgRawData[6];

            pCmdHeader->SizeOfPacket = pMsgRawData[7] << 8;
            pCmdHeader->SizeOfPacket |= pMsgRawData[8];
         }
      }
      else
      {
         State = DC_API_UNKNOWN_IDENTIFIER;
      }
   }

   return State;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgAck 
///
/// @brief Set the raw data for the command ACK
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command ACK.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgAck (PUBYTE pMsgRawData)
{ 
   UBYTE RetVal = SIZE_OF_DC_API_HEADER;

   if (pMsgRawData == NULL)
      return 0;

   // Check the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_ACK >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_ACK & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = 0x00;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = 0x00;
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8 (pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgNak 
///
/// @brief Set the raw data for the command NAK
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @return UBYTE Return code
///
/// This function set the raw data for the command NAK.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgNak (PUBYTE pMsgRawData)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER;

   if (pMsgRawData == NULL)
      return 0;

   // Check the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_NAK >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_NAK & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = 0x00;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = 0x00;
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8 (pMsgRawData, 0, 9, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgPingAck 
///
/// @brief Set the raw data for the command PING_ACK
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @return UBYTE Return code
///
/// This function set the raw data for the command PING_ACK.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgPingAck (PUBYTE pMsgRawData)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER;

   if (pMsgRawData == NULL)
      return 0;

   // Check the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_PING_ACK >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_PING_ACK & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = 0x00;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = 0x00;
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8 (pMsgRawData, 0, 9, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgConnect 
///
/// @brief Set the raw data for the command CONNECT
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param DeviceType (@c [in] UBYTE) - Type of the device
///
/// @param ConEncryption (@c [in] UBYTE) - Type of encryption
///
/// @param MaxNumOfParams (@c [in] UWORD) - Maximum number of parameters per single command
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command CONNECT.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgConnect (PUBYTE pMsgRawData, UBYTE DeviceType, UBYTE ConEncryption, UWORD MaxNumOfParams)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_CONNECT_DATA;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_CONNECT >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_CONNECT & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_CONNECT_DATA >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_CONNECT_DATA & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, 9, CRC_START_VALUE);

   pMsgRawData[O_CMD_CONNECT_DEVICE_TYPE] = DeviceType;
   pMsgRawData[O_CMD_CONNECT_CON_ENCRYPTION] = ConEncryption;
   pMsgRawData[O_CMD_CONNECT_MAX_NUM_OF_PARAMS_HI] = (UBYTE)((MaxNumOfParams >> 8) & 0xFF);
   pMsgRawData[O_CMD_CONNECT_MAX_NUM_OF_PARAMS_LO] = (UBYTE)(MaxNumOfParams & 0xFF);
   pMsgRawData[O_CMD_CONNECT_CRC] = CalculateCRC8(pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_CONNECT_DATA-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgDeviceInfo
///
/// @brief Set the raw data for the command DEVICE_INFO
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param pDeviceInfo (@c [in] MSG_DATA_DEVICE_INFO *) - Device info data
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command DEVICE_INFO.
///
/// @status  Draft
/// @date    2013-05-30
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-30 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgDeviceInfo (PUBYTE pMsgRawData, MSG_DATA_DEVICE_INFO * pDeviceInfo)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_DEVICE_INFO;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_DEVICE_INFO >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_DEVICE_INFO & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_DEVICE_INFO >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_DEVICE_INFO & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   pMsgRawData[O_CMD_DEV_INFO_FLASH_DEVICETYPE]  = pDeviceInfo->FlashMemInfo.FlashDeviceType;
   pMsgRawData[O_CMD_DEV_INFO_FLASH_SIZE]        = pDeviceInfo->FlashMemInfo.FlashSize;
   pMsgRawData[O_CMD_DEV_INFO_CPU_DEVICEID + 0]  = (pDeviceInfo->CPU_DeviceID >> 24) & 0xFF;
   pMsgRawData[O_CMD_DEV_INFO_CPU_DEVICEID + 1]  = (pDeviceInfo->CPU_DeviceID >> 16) & 0xFF;
   pMsgRawData[O_CMD_DEV_INFO_CPU_DEVICEID + 2]  = (pDeviceInfo->CPU_DeviceID >> 8) & 0xFF;
   pMsgRawData[O_CMD_DEV_INFO_CPU_DEVICEID + 3]  = pDeviceInfo->CPU_DeviceID & 0xFF;
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 0]  = pDeviceInfo->CPU_UnigueID[0];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 1]  = pDeviceInfo->CPU_UnigueID[1];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 2]  = pDeviceInfo->CPU_UnigueID[2];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 3]  = pDeviceInfo->CPU_UnigueID[3];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 4]  = pDeviceInfo->CPU_UnigueID[4];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 5]  = pDeviceInfo->CPU_UnigueID[5];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 6]  = pDeviceInfo->CPU_UnigueID[6];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 7]  = pDeviceInfo->CPU_UnigueID[7];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 8]  = pDeviceInfo->CPU_UnigueID[8];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 9]  = pDeviceInfo->CPU_UnigueID[9];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 10] = pDeviceInfo->CPU_UnigueID[10];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 11] = pDeviceInfo->CPU_UnigueID[11];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 12] = pDeviceInfo->CPU_UnigueID[12];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 13] = pDeviceInfo->CPU_UnigueID[13];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 14] = pDeviceInfo->CPU_UnigueID[14];
   pMsgRawData[O_CMD_DEV_INFO_CPU_UNIQUEID + 15] = pDeviceInfo->CPU_UnigueID[15];
   pMsgRawData[O_CMD_DEV_INFO_SW_VERSION]        = pDeviceInfo->SWVersion;
   pMsgRawData[O_CMD_DEV_INFO_SW_REVISION]       = pDeviceInfo->SWRevision;
   pMsgRawData[O_CMD_DEV_INFO_SW_SUBREVISION]    = pDeviceInfo->SWSubRevision;
   pMsgRawData[O_CMD_DEV_INFO_HW_VERSION]        = pDeviceInfo->HWVersion;
   pMsgRawData[O_CMD_DEV_INFO_HW_REVISION]       = pDeviceInfo->HWRevision;
   pMsgRawData[O_CMD_DEV_INFO_BT_FW_VERSION]     = pDeviceInfo->BT_FWVersion;
   pMsgRawData[O_CMD_DEV_INFO_BT_FW_REVISION]    = pDeviceInfo->BT_FWRevision;
   pMsgRawData[O_CMD_DEV_INFO_OS_TYPE]           = pDeviceInfo->OSType;
   pMsgRawData[O_CMD_DEV_INFO_OS_VERSION]        = pDeviceInfo->OSVersion;
   pMsgRawData[O_CMD_DEV_INFO_OS_REVISION]       = pDeviceInfo->OSRevision;
   pMsgRawData[O_CMD_DEV_INFO_OS_SUBREVISION]    = pDeviceInfo->OSSubRevision;
   pMsgRawData[O_CMD_DEV_INFO_NUM_OF_FILES + 0]  = (pDeviceInfo->NumOfFiles >> 8) & 0xFF;
   pMsgRawData[O_CMD_DEV_INFO_NUM_OF_FILES + 1]  = pDeviceInfo->NumOfFiles & 0xFF;
   
   pMsgRawData[O_CMD_DEV_INFO_CRC] = CalculateCRC8(pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_DEVICE_INFO-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgActualMValues 
///
/// @brief Set the raw data for the command ACTUAL_MVALUES
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param pActualMValues (@c [in] MSG_DATA_ACTUAL_MVALUES *) - Actual measurement values
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command ACTUAL_MVALUES.
///
/// @status  Draft
/// @date    2013-06-10
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-10 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgActualMValues (PUBYTE pMsgRawData, MSG_DATA_ACTUAL_MVALUES * pActualMValues)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_ACTUAL_MVALUES;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_ACTUAL_MVALUES >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_ACTUAL_MVALUES & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_ACTUAL_MVALUES >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_ACTUAL_MVALUES & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   memcpy (&pMsgRawData[O_CMD_ACT_MVALUES_TIMESTAMP_HI], &pActualMValues->TimestampHi, sizeof(MSG_DATA_ACTUAL_MVALUES) );
 
   pMsgRawData[O_CMD_ACT_MVALUES_CRC] = CalculateCRC8 (pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_ACTUAL_MVALUES-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgMeasureMode 
///
/// @brief Set the raw data for the command MEASURE_MODE
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param pMeasureMode (@c [in] MSG_DATA_MEASURE_MODE *) - Measure mode
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command MEASURE_MODE.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgMeasureMode (PUBYTE pMsgRawData, MSG_DATA_MEASURE_MODE * pMeasureMode)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_MEASURE_MODE;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_MEASURE_MODE >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_MEASURE_MODE & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_MEASURE_MODE >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_MEASURE_MODE & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   pMsgRawData[O_CMD_MEASURE_MODE_MODE]  = pMeasureMode->MeasureMode;

   pMsgRawData[O_CMD_MEASURE_MODE_CRC] = CalculateCRC8 (pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_MEASURE_MODE-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetMsgDeviceConfig 
///
/// @brief Set the raw data for the command DEVICE_CONFIG
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param pDeviceConfig (@c [in] MSG_DATA_DEVICE_CONFIG *) - Device configuration data
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetMsgDeviceConfig (PUBYTE pMsgRawData, MSG_DATA_DEVICE_CONFIG * pDeviceConfig)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_DEVICE_CONFIG;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_DEVICE_CONFIG >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_DEVICE_CONFIG & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_DEVICE_CONFIG >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_DEVICE_CONFIG & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_NORMAL_HI]   = (pDeviceConfig->MeasurePeriodNormal >> 8) & 0xFF;
   pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_NORMAL_LO]   = pDeviceConfig->MeasurePeriodNormal & 0xFF;
   pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_LOGGING_HI]  = (pDeviceConfig->MeasurePeriodLogging >> 8) & 0xFF;
   pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_LOGGING_LO]  = pDeviceConfig->MeasurePeriodLogging & 0xFF;

   pMsgRawData[O_CMD_DEV_CONFIG_CRC] = CalculateCRC8 (pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_DEVICE_CONFIG-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetFFSFileComplete 
///
/// @brief Set the raw data for the command FFS_FILE_COMPLETE
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command FFS_FILE_COMPLETE.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetFFSFileComplete (PUBYTE pMsgRawData )
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_FFS_FILE_COMPLETE >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_FFS_FILE_COMPLETE & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = 0x00;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = 0x00;
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetFileHeader 
///
/// @brief Set the raw data for the command FILE_HEADER
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param NumOfData (@c [in] UDWORD) - Number of data in file in byte
///
/// @return UBYTE Return code
///
/// This function sets the raw data for the command FILE_HEADER.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_SetFileHeader (PUBYTE pMsgRawData, UDWORD NumOfData)
{
   UBYTE RetVal = SIZE_OF_DC_API_HEADER + SIZE_OF_CMD_FFS_FILE_HEADER;

   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_FFS_FILE_HEADER >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_FFS_FILE_HEADER & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = 0x00;
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = 0x01;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = 0x00;
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = 0x01;
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)((SIZE_OF_CMD_FFS_FILE_HEADER >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)(SIZE_OF_CMD_FFS_FILE_HEADER & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   pMsgRawData[O_CMD_FFS_FILE_HEADER_NUM_OF_DATA + 0]   = ((NumOfData >> 24) & 0xFF);
   pMsgRawData[O_CMD_FFS_FILE_HEADER_NUM_OF_DATA + 1]   = ((NumOfData >> 16) & 0xFF);
   pMsgRawData[O_CMD_FFS_FILE_HEADER_NUM_OF_DATA + 2]   = ((NumOfData >> 8) & 0xFF);
   pMsgRawData[O_CMD_FFS_FILE_HEADER_NUM_OF_DATA + 3]   = (NumOfData & 0xFF);

   pMsgRawData[O_CMD_FFS_FILE_HEADER_CRC] = CalculateCRC8 (pMsgRawData, SIZE_OF_DC_API_HEADER, SIZE_OF_CMD_FFS_FILE_HEADER-1, CRC_START_VALUE);

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_SetFileBlock 
///
/// @brief Set the raw data for the command FILE_BLOCK
///
/// @param pMsgRawData (@c [out] PUBYTE) - Raw data stream
///
/// @param pMeasureFileBlock (@c [in] PUBYTE) - Pointer to the file data block
///
/// @param SizeOfDataBlock (@c [in] UWORD) - Size of data block
///
/// @param iCurFileBlock (@c [in] UWORD) - Index actual file block
///
/// @param TotalNumOfBlocks (@c [in] UWORD) - Total number of file blocks
///
/// @return UWORD Return code
///
/// This function sets the raw data for the command FILE_BLOCK.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UWORD BaroAltiAPI_SetFileBlock (PUBYTE pMsgRawData, PUBYTE pMeasureFileBlock, UWORD SizeOfDataBlock, UWORD iCurFileBlock, UWORD TotalNumOfBlocks )
{
   UWORD RetVal;
   int i;
   
   if (SizeOfDataBlock > 0)
   {
      RetVal = SIZE_OF_DC_API_HEADER + SizeOfDataBlock + 1;
   }
   else
   {
      RetVal = SIZE_OF_DC_API_HEADER;
   }
   
   if (pMsgRawData == NULL)
      return 0;

   // fill the header
   pMsgRawData[OFFSET_IDENTIFIER] = '#';
   pMsgRawData[OFFSET_CMD_ID_HI] = (UBYTE)((CMD_ID_FFS_FILE >> 8) & 0xFF);
   pMsgRawData[OFFSET_CMD_ID_LO] = (UBYTE)(CMD_ID_FFS_FILE & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_HI] = (UBYTE)((iCurFileBlock >> 8) & 0xFF);
   pMsgRawData[OFFSET_CUR_PACKET_NUM_LO] = (UBYTE)(iCurFileBlock & 0xFF);
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_HI] = (UBYTE)((TotalNumOfBlocks >> 8) & 0xFF);
   pMsgRawData[OFFSET_TOTAL_NUM_OF_PACKETS_LO] = (UBYTE)(TotalNumOfBlocks & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_HI] = (UBYTE)(((SizeOfDataBlock+1) >> 8) & 0xFF);
   pMsgRawData[OFFSET_NUM_OF_PARAMS_LO] = (UBYTE)((SizeOfDataBlock+1) & 0xFF);
   pMsgRawData[OFFSET_HEADER_CRC] = CalculateCRC8( pMsgRawData, 0, SIZE_OF_DC_API_HEADER-1, CRC_START_VALUE);

   // fill the message data
   for (i = 0; i < SizeOfDataBlock; i++)
   {
      pMsgRawData[SIZE_OF_DC_API_HEADER + i] = pMeasureFileBlock[i];
   }
   
   pMsgRawData[SIZE_OF_DC_API_HEADER + SizeOfDataBlock] = CalculateCRC8 (pMsgRawData, SIZE_OF_DC_API_HEADER, SizeOfDataBlock, CRC_START_VALUE);

   return RetVal;   
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_GetMsgReqMeasureFileData 
///
/// @brief Extract the data for the command REQ_MEASURE_FILE_DATA from the raw data
///
/// @param pMsgRawData (@c [in] PUBYTE) - Raw data stream
///
/// @param pMeasureMode (@c [out] MSG_DATA_REQ_MEASURE_FILE *) - Structured command data
///
/// @return UBYTE Return code
///
/// This function extracts the data for the command REQ_MEASURE_FILE_DATA from
/// the raw data.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_GetMsgReqMeasureFileData (PUBYTE pMsgRawData, MSG_DATA_REQ_MEASURE_FILE * pMeasureMode)
{
   UBYTE State = DC_API_OK;
   UBYTE CheckCRCValue = 0;

   if( pMsgRawData != NULL && pMeasureMode != NULL )
   {
      CheckCRCValue = CalculateCRC8 (pMsgRawData, 0, SIZE_OF_CMD_REQ_MEASURE_FILE-1, CheckCRCValue);

      if (CheckCRCValue != pMsgRawData[O_CMD_REQ_MEASURE_FILE_CRC-SIZE_OF_DC_API_HEADER])
      {
         State = DC_API_CRC_VALUE_FALSE;
      }
      else
      {
         pMeasureMode->FileNumber =  pMsgRawData[O_CMD_REQ_MEASURE_FILE-SIZE_OF_DC_API_HEADER + 0] << 8;
         pMeasureMode->FileNumber |= pMsgRawData[O_CMD_REQ_MEASURE_FILE-SIZE_OF_DC_API_HEADER + 1];
      }
   }

   return State;

}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_GetMsgMeasureModeData 
///
/// @brief Extract the data for the command MEASURE_MODE_DATA from the raw data
///
/// @param pMsgRawData (@c [in] PUBYTE) - Raw data stream
///
/// @param pMeasureMode (@c [out] MSG_DATA_MEASURE_MODE *) - Structured command data
///
/// @return UBYTE Return code
///
/// This functions extracts the data for the command MEASURE_MODE_DATA from
/// the raw data.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_GetMsgMeasureModeData (PUBYTE pMsgRawData, MSG_DATA_MEASURE_MODE * pMeasureMode)
{
   UBYTE State = DC_API_OK;
   UBYTE CheckCRCValue = 0;

   if( pMsgRawData != NULL && pMeasureMode != NULL )
   {
      CheckCRCValue = CalculateCRC8 (pMsgRawData, 0, SIZE_OF_CMD_MEASURE_MODE-1, CheckCRCValue);

      if (CheckCRCValue != pMsgRawData[O_CMD_MEASURE_MODE_CRC-SIZE_OF_DC_API_HEADER])
      {
         State = DC_API_CRC_VALUE_FALSE;
      }
      else
      {
         pMeasureMode->MeasureMode = pMsgRawData[O_CMD_MEASURE_MODE_MODE-SIZE_OF_DC_API_HEADER];
      }
   }

   return State;
}

//-----------------------------------------------------------------------------
//  Function BaroAltiAPI_GetMsgDeviceConfigData 
///
/// @brief Extract the data for the command DEVICE_CONFIG_DATA from the raw data
///
/// @param pMsgRawData (@c [in] PUBYTE) - Raw data stream
///
/// @param pDeviceConfig (@c [out] MSG_DATA_DEVICE_CONFIG *) - Structured command data
///
/// @return UBYTE Return code
///
/// This function extracts the data for the command DEVICE_CONFIG_DATA from
/// the raw data.
///
/// @status  Draft
/// @date    2013-07-15
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-15 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE BaroAltiAPI_GetMsgDeviceConfigData (PUBYTE pMsgRawData, MSG_DATA_DEVICE_CONFIG * pDeviceConfig)
{
   UBYTE State = DC_API_OK;
   UBYTE CheckCRCValue = 0;

   if( pMsgRawData != NULL && pDeviceConfig != NULL )
   {
      CheckCRCValue = CalculateCRC8 (pMsgRawData,0,SIZE_OF_CMD_DEVICE_CONFIG-1,CheckCRCValue);

      if (CheckCRCValue != pMsgRawData[O_CMD_DEV_CONFIG_CRC-SIZE_OF_DC_API_HEADER])
      {
         State = DC_API_CRC_VALUE_FALSE;
      }
      else
      {
         pDeviceConfig->MeasurePeriodNormal = pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_NORMAL_HI-SIZE_OF_DC_API_HEADER] << 8;
         pDeviceConfig->MeasurePeriodNormal |= pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_NORMAL_LO-SIZE_OF_DC_API_HEADER];
         pDeviceConfig->MeasurePeriodLogging = pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_LOGGING_HI-SIZE_OF_DC_API_HEADER] << 8;
         pDeviceConfig->MeasurePeriodLogging |= pMsgRawData[O_CMD_DEV_CONFIG_MPERIOD_LOGGING_LO-SIZE_OF_DC_API_HEADER];
      }
   }

   return State;
}
