package de.jockels.forms;

import de.jockels.lib.StringTools;
import android.app.Activity;
import android.content.ContentValues;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.View;
import android.widget.CheckBox;
import android.widget.EditText;
import android.widget.Spinner;

public class DialogHelper<T extends Content> {
	public static final String TAG ="DialogHelper";
	
	public T current;
	ContentHelper<T> mContentHelper;
	Activity mRoot;
	Class<T> mMaker;
	int[] mIds;
	View[] mViews;
	int mAddIdx;
	boolean mLogging; // Error-Logging

	public DialogHelper(Activity aRoot, ContentHelper<T> aHelper, Class<T> aMaker) {
		mContentHelper = aHelper;
		mIds = new int[aHelper.getSize()];
		mViews = new View[aHelper.getSize()];
		mRoot = aRoot;
		mAddIdx = 0;
		mMaker = aMaker;
		mLogging = false;
		try {
			current = mMaker.newInstance();
		} catch (Exception e) {
			current = null;
		}
	}

	
	// TextWatcher / OnFocusChangeListener ---------------------------------------------------------
	private class Watcher implements TextWatcher {
		int id;
		Watcher(int aId) { id = aId; }

		public void afterTextChanged(Editable arg0) {
//			Log.v(TAG, "afterTextChanged "+id);
			setErrorMessage(id);
		}
		public void beforeTextChanged(CharSequence arg0, int arg1, int arg2, int arg3) { }
		public void onTextChanged(CharSequence arg0, int arg1, int arg2, int arg3) { }
	}
	
	
	// Form-Beschreibung -----------------------------------------------------------------------------
	public void setConnection(int idx, int id, View v) {
		mIds[idx] = id;
		mViews[idx] = v;
		if (v instanceof EditText)
			((EditText)v).addTextChangedListener(new Watcher(id));
	}

	public DialogHelper<T> addConnection(int id, View v) {
		setConnection(mAddIdx, id, v);
		mAddIdx++;
		return this;
	}
	
	public DialogHelper<T> addConnection(int id) {
		addConnection(id, mRoot.findViewById(id));
		return this;
	}
	
	
	// Interaktion mit Views -----------------------------------------------------------------------
	public void writeToView(T aContent) {
		Log.v(TAG, "writeToView");
		boolean before = mLogging;
		mLogging = false;
		for (int i=0; i<aContent.s.length; i++) writeToView(i, aContent);
		mLogging = before;
		Log.v(TAG, "writeToView end");
	}
	
	public void writeToView(int i, T aContent) {
		switch (mContentHelper.getType(i)) {
		case STRING:
			((EditText)mViews[i]).setText(aContent.s[i]);
			break;
		case INT:
			((Spinner)mViews[i]).setSelection(aContent.getInt(i));
			break;
		case LONG:
			((EditText)mViews[i]).setText(""+aContent.getLong(i));
			break;
		case BOOLEAN:
			((CheckBox)mViews[i]).setChecked(aContent.l[i] > 0);
			break;
		}
	}

	public void readFromView() {
		for (int i=0; i<mContentHelper.getSize(); i++) readFromView(i);
	}

	public void readFromView(int i) {
		switch (mContentHelper.getType(i)) {
		case STRING:
			current.s[i] = ((EditText)mViews[i]).getText().toString();
			break;
		case INT:
			current.l[i] = ((Spinner)mViews[i]).getSelectedItemPosition();
			break;
		case LONG:
			current.l[i] = StringTools.tryParseLong(((EditText)mViews[i]).getText().toString(), 0);
		case BOOLEAN:
			current.l[i] = ((CheckBox)mViews[i]).isChecked() ? 1 : 0;
			break;
		}
	}
	
	// Validierung ---------------------------------------------------------------------------------------
	public void setErrorMessage(int idx) {
		if (!mLogging) return;
		readFromView(idx);
		int error = current.error(idx);
		if (error==0) {
			// ist gltig, setError(null) aber nur unter bestimmten Bedingungen
			if (!mViews[idx].hasFocus()) 
				switch (mContentHelper.getType(idx)) {
				case STRING: case LONG: ((EditText)mViews[idx]).setError(null); break;
				case BOOLEAN:
				case INT:
				}
		} else {
			// ungltig, also setError
			switch (mContentHelper.getType(idx)) {
			case STRING: case LONG: ((EditText)mViews[idx]).setError(mRoot.getText(error)); break;
			case INT:
			case BOOLEAN:
			}
		}
	}
	
	public void setErrorMessages() { 
		Log.v(TAG, "setErrorMessages");
		if (!mLogging) return;
		// muss anders als bei isValid alle durchlaufen
		for (int i=0; i<mContentHelper.getSize(); i++) setErrorMessage(i);
	}
	
	public void setErrorLogging(boolean aLogging) { 
		Log.v(TAG, "setErrorLogging "+aLogging);
		mLogging = aLogging; 
		if (mLogging) setErrorMessages();
	}

	public boolean getErrorLogging() { return mLogging; }
	
	
	// Vereinfachungen mit ContentHelper ------------------------------------------------------
	public int error() { return current.error(); }
	
	public ContentValues writeToContentValues() { return mContentHelper.writeToContentValues(current); }
	
	public boolean isEqual(T c) {
		readFromView();
		return mContentHelper.isEqual(c, current);
	}
}
