// ctpuzzle.cpp : Definiert den Einsprungpunkt fr die Konsolenanwendung.
//

#include "stdafx.h"

// #define ZAEHLEN

typedef unsigned __int64 int64;
typedef unsigned long int32;
typedef unsigned short int16;
typedef unsigned char byte;

// Oft gebrauchte Verschiebe-Operationen als Shifts ausgedrckt
#define SHIFT_PLUSX(x)   (((x)&(int64)033333333333333333333)<<1)
#define SHIFT_PLUS2X(x)  (((x)&(int64)011111111111111111111)<<2)
#define SHIFT_MINUSX(x)  (((x)&(int64)066666666666666666666)>>1)
#define SHIFT_MINUS2X(x) (((x)&(int64)044444444444444444444)>>2)
#define SHIFT_PLUSY(x)   (((x)&(int64)007770777077707770777)<<3)
#define SHIFT_PLUS2Y(x)  (((x)&(int64)000770077007700770077)<<6)
#define SHIFT_MINUSY(x)  (((x)&(int64)077707770777077707770)>>3)
#define SHIFT_MINUS2Y(x) (((x)&(int64)077007700770077007700)>>6)
#define SHIFT_PLUSZ(x)   (((x)&(int64)000007777777777777777)<<12)
#define SHIFT_PLUS2Z(x)  (((x)&(int64)000000000777777777777)<<24)
#define SHIFT_MINUSZ(x)  (((x)&(int64)077777777777777770000)>>12)
#define SHIFT_MINUS2Z(x) (((x)&(int64)077777777777700000000)>>24)

// Ab diesen Grenzwerten beginnt vorab ausgerechnetes "Endspiel":
// Alle Kombinationen von Steinen, die komplett oberhalb von 
// Bit ENDSPIEL_START liegen.
#define HAUPTSPIEL_MASKE   ((int64) 0x00000007fffffff)
#define ENDSPIEL_START 31
#define ENDSPIEL_RESTMASKE ((int64) 0x000000f80000000)
#define EXTENDER_SIZE 1500000


//
// -------------------------
//      Datenstrukturen
// -------------------------
//

#define N_VARIANTEN 128
#define SUCH_EBENEN 5


// Ein Puzzle-Teil in allen Varianten
typedef struct {
    int64 basisform;
    int64 relVarianten[5][12][N_VARIANTEN];
    short schluessel[5][12][20];
    int64 alleVarianten[577];
    int64 variantenEndspiel[400];
} Stein;


// Eine Lsung im Endspiel 
typedef struct {
    // Belegung Bits 36-59 bilden den Index fr das Array 'endspiele'.
    // Sie brauchen deshalb hier nicht abgelegt zu werden.
    int16 muster;	// High 6 Bit: Belegung (30-35), Low 10 Bit: Verbrauch
    int16 nloesungen;	// Anzahl verschiedener Lsungen zu dieser Belegung
} EndspielLoesung;


// Verketteter Datenblock fr Endspiel-Speicherung
typedef struct extender {
    EndspielLoesung loesungen[7];
    struct extender *next;
} EndspielExtender;


// XYZ-Koordinate fr das Rotieren/Verschieben in der Initialisierungsphase
typedef struct {
    int x;
    int y;
    int z;
    bool valid;
} XYZ;


//
// -------------------------
//        Variablen
// -------------------------
//

// Programmstart
time_t start;


// Folge von 1-Bit-Masken (war mal zentrale Steuerung 
// fr die Suchreihenfolge, ist jetzt fast berflssig)
int64 reihenfolge[60];


// Steine in Grundform
Stein basisSteine[12];

// Kopie von 'basisSteine', 
// aber hier werden jeweils die mit den vorab gesetzten Steinen 
// kollidierenden Varianten herausgelscht.
Stein steine[12];


// Steuer-Struktur, die fr eine 12-Bit-Ebenenbelegung sagt,
// welche Position als nchste besetzt werden soll ('Hotspot').
// Auerdem Info auf Basis der Hotspot-Nachbarn, welche 
// Varianten-Teilmenge passen kann.
// Indizes [0] = Hotspot, [1] = Start-Schlssel, [2] = End-Schlssel

// Hotspot fr Ebenen 0 und 1
short hotspotSchluessel[0x1000][4];

// Hotspot fr Ebene 2, vermeidet Positionen, 
// die schon im Endspiel abgelegt sind.
short hotspotSchluesselE2[0x1000][4];


// (Hash-)Array als Einstieg in den Endspiel-Speicher.
// Hashcode = Bits 36-59
EndspielExtender *endspiele[0x1000000];

// Vorrat an Blcken fr verkettete Endspiel-Daten
EndspielExtender extender[EXTENDER_SIZE];

// Belegungs-Zhler fr 'extender'
int32 nExtender = 0;


// Eine recht brauchbare statische Suchreihenfolge
int BIT_REIHE_INV_RINGE[] = {
    0 , 1, 2, 5, 8,11,10, 9, 6, 3, 4, 7,
    12,13,14,17,20,23,22,21,18,15,16,19,
    24,25,26,29,32,35,34,33,30,27,28,31,
    36,37,38,41,44,47,46,45,42,39,40,43,
    48,49,50,53,56,59,58,57,54,51,52,55
};


/** Bitmuster der Nachbarn in einer Ebene. */
int NACHBARN[] = {
    00012, 00025, 00042,
    00121, 00252, 00424,
    01210, 02520, 04240,
    02100, 05200, 02400
};

// Einige Statistik-Zhler, wenn #define ZAEHLEN

// Anzahl ins Puzzle gesetzter Steine
int32 nSuchvorgaenge = 0;

// Anzahl an Versuchen, einen Stein zu setzen
int64 nVersuche = 0;

// Anzahl an Versuchen, eine Konfiguration 
// als Endspiel zu besttigen oder zu verwerfen
int32 nEndspiele = 0;

// !!! DIE WICHTIGSTE VARIABLE !!!
int32 nLoesungen;

//
// -------------------------
//        Funktionen
// -------------------------
//


    /**
     * Liefert True, wenn die Koordinaten im vorgegebenen 3*4*5-Quader liegen.
     * @param koords Array der Koordinaten, jeweils als {x, y, z}
     * @return True, wenn alle im 3*4*5-Quader liegen
     */
    bool isGueltig(XYZ koords[]) {
	XYZ *p;
        for (p=koords; p->valid; p++) {
            if (!((p->x >= 0) && (p->x < 3) &&
                  (p->y >= 0) && (p->y < 4) &&
                  (p->z >= 0) && (p->z < 5))) {
                return false;
            }
        }
        return true;
    }


    inline XYZ newXYZ(int x, int y, int z) {
	XYZ pt;
	pt.x = x;
	pt.y = y;
	pt.z = z;
	pt.valid = true;
	return pt;
    }


    /**
     * Verschiebt die Punkte in den positiven Oktanten, auf jeder Achse
     * beginnend mit 0 als kleinstem Wert.
     */
    void verschiebeNachNull(XYZ koords[]) {
        int xMin = 1000;
        int yMin = 1000;
        int zMin = 1000;
	XYZ *kp;

        for (kp=koords; kp->valid; kp++) {
            if (kp->x < xMin) xMin = kp->x;
            if (kp->y < yMin) yMin = kp->y;
            if (kp->z < zMin) zMin = kp->z;
        }
        for (kp=koords; kp->valid; kp++) {
            kp->x -= xMin;
            kp->y -= yMin;
            kp->z -= zMin;
        }
    }

    /**
     * Verschiebt den Stein um DX, DY, DZ.
     */
    void verschiebe(XYZ koords[], XYZ ziel[], int dx, int dy, int dz) {
	XYZ *kp, *zp;
        for (kp=koords, zp=ziel; kp->valid; kp++, zp++) {
            ziel->x = kp->x + dx;
            ziel->y = kp->y + dy;
            ziel->z = kp->z + dz;
        }
	zp->valid = false;
    }

    /**
     * Verschiebt den Stein um DX, DY, DZ.
     */
    int64 verschiebeBits(int64 bits, int dx, int dy, int dz) {
        int shift = dx + 3 * dy + 12 * dz;
        return bits << shift;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der X-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int xAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->x > maxwert) maxwert = kp->x;
        }
        return maxwert + 1;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der Y-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int yAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->y > maxwert) maxwert = kp->y;
        }
        return maxwert + 1;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der Z-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int zAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->z > maxwert) maxwert = kp->z;
        }
        return maxwert + 1;
    }

    /**
     * Wandelt von Koordinaten-Array in Bitvektor um.
     */
    int64 getBits(XYZ koords[]) {
        int64 result = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
	    result += ((int64) 1) <<
		(kp->x + 3*kp->y + 12*kp->z);
        }
        return result;
    }

    /**
     * Wandelt von Bitvektor in Koordinaten-Array um.
     */
    void getKoords(int64 bits, XYZ ziel[]) {
        XYZ *zp = ziel;
	int64 maske = 1;
	int x, y, z;
        for (z=0; z<5; z++) {
            for (y=0; y<4; y++) {
                for (x=0; x<3; x++) {
                    if ((bits & maske) != 0) {
			*zp++ = newXYZ(x, y, z);
                    }
                    maske = maske << 1;
                }
            }
        }
	zp->valid = false;
    }

    /**
     * Rotiert einen Punkt um den Quader-Mittelpunkt.
     * @param punkt originaler Punkt
     * @param orientierung Version von 0 bis 23
     * @return rotierter Punkt.
     */
    XYZ rotiereXYZ(XYZ punkt, int orientierung) {
        int x = punkt.x;
        int y = punkt.y;
        int z = punkt.z;

        switch (orientierung) {
            case  0: return newXYZ(x,   y,   z);
            case  1: return newXYZ(2-x, 3-y, z);
            case  2: return newXYZ(2-x, y,   4-z);
            case  3: return newXYZ(x,   3-y, 4-z);
            case  4: return newXYZ(y,   z,   x);
            case  5: return newXYZ(2-y, 3-z, x);
            case  6: return newXYZ(2-y, z,   4-x);
            case  7: return newXYZ(y,   3-z, 4-x);
            case  8: return newXYZ(z,   x,   y);
            case  9: return newXYZ(2-z, 3-x, y);
            case 10: return newXYZ(2-z, x,   4-y);
            case 11: return newXYZ(z,   3-x, 4-y);
            case 12: return newXYZ(2-x, 3-z, 4-y);
            case 13: return newXYZ(x,   z,   4-y);
            case 14: return newXYZ(x,   3-z, y);
            case 15: return newXYZ(2-x, z,   y);
            case 16: return newXYZ(2-z, 3-y, 4-x);
            case 17: return newXYZ(z,   y,   4-x);
            case 18: return newXYZ(z,   3-y, x);
            case 19: return newXYZ(2-z, y,   x);
            case 20: return newXYZ(2-y, 3-x, 4-z);
            case 21: return newXYZ(y,   x,   4-z);
            case 22: return newXYZ(y,   3-x, z);
            case 23: return newXYZ(2-y, x,   z);
            default: return punkt;
        }
    }


    /**
     * Stein in Form eines XYZ-Arrays drehen.
     */
    void rotiere(XYZ koords[], XYZ ziel[], int orientierung) {
	XYZ *kp, *zp;
        for (kp=koords, zp=ziel; kp->valid; kp++, zp++) {
            *zp = rotiereXYZ(*kp, orientierung);
        }
	zp->valid = false;
    }

    /**
     * Stein in Form eines Bitmusters drehen.
     */
    int64 rotiereBits(int64 bits, int orientierung) {
	XYZ koords[7], rotiert[7];
	getKoords(bits, koords);
	rotiere(koords, rotiert, orientierung);
	return getBits(rotiert);
    }


    /**
     * Liefert zu einer Stein-Lage den Start-Schritt gem eingestellter
     * Schritt-Reihenfolge.
     */
    int getStartPos(int64 bits) {
        int i;
        for (i=0; i<60; i++) {
            if ((bits & reihenfolge[i]) != 0) {
                return i;
            }
        }
        return 63;
    }


    /**
     * Gibt es irgendwo ein isoliertes Einer-Loch?
     */
    inline bool isGesund(int64 belegung) {
	int64 freie = (~belegung) & (int64) 077777777777777777777;
        int64 muster = 
	    belegung |
            ((freie & (int64) 033333333333333333333) <<  1) |
            ((freie & (int64) 066666666666666666666) >>  1) |
            ((freie & (int64) 007770777077707770777) <<  3) |
            ((freie & (int64) 077707770777077707770) >>  3) |
            ((freie & (int64) 000007777777777777777) << 12) |
            ((freie & (int64) 077777777777777770000) >> 12);
        return (muster  == (int64) 077777777777777777777);
    }


    /**
     * Gibt es irgendwo ein isoliertes Einer- oder Zweier-Loch?
     */
    inline bool isGesund2(int64 bel) {
	int64 frei = (~bel) & (int64) 077777777777777777777;
        int64 kreuz;

	kreuz = 
	    bel |
	    SHIFT_PLUSY(frei) | 
	    SHIFT_MINUSY(frei) |
	    SHIFT_PLUSZ(frei) |
	    SHIFT_MINUSZ(frei);
	if ((SHIFT_PLUSX(frei) | kreuz | SHIFT_MINUSX(frei))
	    != 077777777777777777777) {
	    return false;
	}
	if ((SHIFT_PLUSX(frei) | kreuz | SHIFT_MINUSX(kreuz) | SHIFT_MINUS2X(frei))
	    != 077777777777777777777) {
	    return false;
	}

	kreuz =
	    bel | 
	    SHIFT_PLUSX(frei) |
	    SHIFT_MINUSX(frei) |
	    SHIFT_PLUSZ(frei) |
	    SHIFT_MINUSZ(frei);
	if ((SHIFT_PLUSY(frei) | kreuz | SHIFT_MINUSY(kreuz) | SHIFT_MINUS2Y(frei))
	    != 077777777777777777777) {
	    return false;
	}

	kreuz = 
	    bel | 
	    SHIFT_PLUSX(frei) |
	    SHIFT_MINUSX(frei) |
	    SHIFT_PLUSY(frei) |
	    SHIFT_MINUSY(frei);
	if ((SHIFT_PLUSZ(frei) | kreuz | SHIFT_MINUSZ(kreuz) | SHIFT_MINUS2Z(frei))
	    != 077777777777777777777) {
	    return false;
	}

        return true;
    }

    /**
     * Fllt 'alleVarianten'.
     */
    void berechneAlleVarianten(Stein *stein) {
        XYZ koords[7];
	getKoords(stein->basisform, koords);

        // Alle rotierten Versionen erzeugen
        int64 alleRotiertenVersionen[24];
	int nRotierteVersionen = 0;
        for (int orientierung=0; orientierung<24; orientierung++) {
            XYZ koordsNeu[7];
	    rotiere(koords, koordsNeu, orientierung);
            verschiebeNachNull(koordsNeu);
	    int64 rotiert = getBits(koordsNeu);
            if (isGueltig(koordsNeu)) {
		bool istNeu = true;
		for (int i=0; i<nRotierteVersionen; i++) {
		    if (alleRotiertenVersionen[i] == rotiert) {
			istNeu = false;
			break;
		    }
		}
		if (istNeu) {
		    alleRotiertenVersionen[nRotierteVersionen++] = rotiert;
		}
            }
        }

	int nAlle = 0;

        for (int i=0; i<nRotierteVersionen; i++) {
            int64 basisVersion = alleRotiertenVersionen[i];
	    XYZ koordsVersion[7];
            getKoords(basisVersion, koordsVersion);
            int xSchritte = 4 - xAusdehnung(koordsVersion);
            int ySchritte = 5 - yAusdehnung(koordsVersion);
            int zSchritte = 6 - zAusdehnung(koordsVersion);
            for (int z=0; z<zSchritte; z++) {
                for (int y=0; y<ySchritte; y++) {
                    for (int x=0; x<xSchritte; x++) {
                        int64 verschoben = 
			    verschiebeBits(basisVersion, x, y, z);
			stein->alleVarianten[nAlle++] = verschoben;
                    }
                }
            }
        }
    }


    bool gesundheitTesten(int verbrauchsMaske, int64 belegung) {
        int64 belegbar = 0;
        for (int i=0, sMaske=1; i<10; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
		int64 variante;
		int64 *varp;
                for (varp=steine[i].alleVarianten; (variante=*varp)!=0; varp++) {
                    if ((belegung & variante) == 0) {
			belegbar |= variante;
                    }
                }
            }
        }
        return ((belegbar | belegung) == (int64) 077777777777777777777);
    }

    /**
     * Fllt 'alleVarianten', behlt dabei von den 
     * je 4 symmetrischen Varianten die, die am weitesten 
     * "vorne" liegt.
     */
    void berechneSymmetrieVarianten(Stein *stein) {
        XYZ koords[7];
	getKoords(stein->basisform, koords);

        // Alle rotierten Versionen erzeugen
        int64 alleRotiertenVersionen[24];
	int nRotierteVersionen = 0;
        for (int orientierung=0; orientierung<24; orientierung+=4) {
            XYZ koordsNeu[7];
	    rotiere(koords, koordsNeu, orientierung);
            verschiebeNachNull(koordsNeu);
	    int64 rotiert = getBits(koordsNeu);
            if (isGueltig(koordsNeu)) {
		bool istNeu = true;
		for (int i=0; i<nRotierteVersionen; i++) {
		    if (alleRotiertenVersionen[i] == rotiert) {
			istNeu = false;
			break;
		    }
		}
		if (istNeu) {
		    alleRotiertenVersionen[nRotierteVersionen++] = rotiert;
		}
            }
        }

	int nAlle = 0;
        for (int i=0; i<nRotierteVersionen; i++) {
            int64 basisVersion = alleRotiertenVersionen[i];
	    XYZ koordsVersion[7];
            getKoords(basisVersion, koordsVersion);
            int xSchritte = 4 - xAusdehnung(koordsVersion);
            int ySchritte = 5 - yAusdehnung(koordsVersion);
            int zSchritte = 6 - zAusdehnung(koordsVersion);
            for (int z=0; z<zSchritte; z++) {
                for (int y=0; y<ySchritte; y++) {
                    for (int x=0; x<xSchritte; x++) {
                        int64 verschoben1 = 
			    verschiebeBits(basisVersion, x, y, z);
                        int startbit1 = getStartPos(verschoben1);
                        int64 verschoben2 = rotiereBits(verschoben1, 1);
                        int startbit2 = getStartPos(verschoben2);
                        int64 verschoben3 = rotiereBits(verschoben1, 2);
                        int startbit3 = getStartPos(verschoben3);
                        int64 verschoben4 = rotiereBits(verschoben1, 3);
                        int startbit4 = getStartPos(verschoben4);


			int64 kandidat = verschoben1;
                        int kleinstePos = startbit1;
                        if (startbit2 < kleinstePos) {
                            kandidat = verschoben2;
                            kleinstePos = startbit2;
                        }
                        if (startbit3 < kleinstePos) {
                            kandidat = verschoben3;
                            kleinstePos = startbit3;
                        }
                        if (startbit4 < kleinstePos) {
                            kandidat = verschoben4;
                            kleinstePos = startbit4;
                        }

			stein->alleVarianten[nAlle++] = kandidat;
                    }
                }
            }
        }
    }

    int64 listeA[145];
    int64 listeB[145];
    int64 listeC[145];
    int64 listeD[145];
    int64 listeE[145];
    int64 listeF[145];
    int64 listeG[145];
    int64 listeI[145];
    int64 listeJ[145];
    int64 listeK[145];
    int64 listeM[145];


    /**
     * Baut aus 'alleVarianten' die verschiedenen sortierten Arrays.
     *
     * 'variantenEndspiel' enthlt alle, die fr das Endspiel passen.
     *
     * 'relVarianten[ebene][pos_in_ebene] enthlt die, die bei der Suche
     * an der Stelle [ebene][pos_in_ebene] in Frage kommen. 
     * Die Reihenfolge im Array ist so organisiert, dass fr alle mglichen
     * Belegungen der 4 Nachbarpltze ein zusammenhngender Ausschnitt
     * des Arrays die passenden Varianten ergibt.
     */
    void sortiereVarianten(Stein *stein) {
	int nEndspiel = 0;
	for (int64 *p=stein->alleVarianten; *p!=0; p++) {
	    int64 variante = *p;
	    if ((variante & HAUPTSPIEL_MASKE) == 0) {
		stein->variantenEndspiel[nEndspiel++] = variante;
	    }
	}
	stein->variantenEndspiel[nEndspiel] = 0;

	int maxIndex = 0;

	for (int ebene=0; ebene<SUCH_EBENEN; ebene++) {
	    int64 nullMaske = 0xffffffffffff >> (12 * (4 - ebene));
	    for (int relpos=0; relpos<12; relpos++) {
		int64 nonNullMaske = ((int64) 1) << (12 * ebene + relpos);
		int nA=0, nB=0, nC=0, nD=0, nE=0, nF=0, nG=0, nI=0, nJ=0, nK=0, nM=0;

		for (int64 *p2=stein->alleVarianten; *p2!=0; p2++) {
		    int64 variante = *p2;
		    if (((variante & nonNullMaske) != 0) &&
			((variante & nullMaske) == 0)) {
			int64 relVariante = variante >> (12 * ebene);
			int32 schnitt = relVariante & 0xfff;
			int32 pattern = (schnitt & NACHBARN[relpos]) << (11 - relpos);
			switch (pattern) {
			case 0x0000:
			    listeA[nA++] = relVariante;
			    break;
			case 0x0100:
			    listeB[nB++] = relVariante;
			    break;
			case 0x0400:
			    listeC[nC++] = relVariante;
			    break;
			case 0x0500:
			    listeD[nD++] = relVariante;
			    break;
			case 0x1000:
			    listeE[nE++] = relVariante;
			    break;
			case 0x1100:
			    listeF[nF++] = relVariante;
			    break;
			case 0x1400:
			    listeG[nG++] = relVariante;
			    break;
			case 0x4000:
			    listeI[nI++] = relVariante;
			    break;
			case 0x4100:
			    listeJ[nJ++] = relVariante;
			    break;
			case 0x4400:
			    listeK[nK++] = relVariante;
			    break;
			case 0x5000:
			    listeM[nM++] = relVariante;
			    break;
			case 0x1500:
			case 0x4500:
			case 0x5100:
			case 0x5400:
			case 0x5500:
			    // Diese Patterns brauchen nicht indiziert zu werden!
			    // Wegen Hotspot-Reihenfolge werden sie nie an dieser 
			    // Startposition auftreten.
			    break;
			default:
			    printf("Logikfehler bei Stein %07x%08x, Ebene %d, Pos %d:\n",
				   (int32) (stein->basisform >> 32),
				   (int32) stein->basisform,
				   ebene, relpos);
			    printf("   Variante %07x%08x ergibt %04x\n",
				   (int32) (relVariante >> 32),
				   (int32) relVariante,
				   pattern);
			    exit(1);
			}
		    }
		}
		int64 *pl = stein->relVarianten[ebene][relpos];
		int index = 0;
		short *ps = stein->schluessel[ebene][relpos];
		*ps++ = index;

		memcpy(&pl[index], listeD, nD * sizeof(int64));
		index += nD;
		*ps++ = index;

		memcpy(&pl[index], listeB, nB * sizeof(int64));
		index += nB;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeC, nC * sizeof(int64));
		index += nC;
		*ps++ = index;

		memcpy(&pl[index], listeG, nG * sizeof(int64));
		index += nG;
		*ps++ = index;

		memcpy(&pl[index], listeE, nE * sizeof(int64));
		index += nE;
		*ps++ = index;
		
		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeF, nF * sizeof(int64));
		index += nF;
		*ps++ = index;

		memcpy(&pl[index], listeB, nB * sizeof(int64));
		index += nB;
		*ps++ = index;

		memcpy(&pl[index], listeJ, nJ * sizeof(int64));
		index += nJ;
		*ps++ = index;

		memcpy(&pl[index], listeI, nI * sizeof(int64));
		index += nI;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeC, nC * sizeof(int64));
		index += nC;
		*ps++ = index;

		memcpy(&pl[index], listeK, nK * sizeof(int64));
		index += nK;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeE, nE * sizeof(int64));
		index += nE;
		*ps++ = index;

		memcpy(&pl[index], listeI, nI * sizeof(int64));
		index += nI;
		*ps++ = index;

		memcpy(&pl[index], listeM, nM * sizeof(int64));
		index += nM;
		*ps++ = index;

		if (index > maxIndex) maxIndex = index;
	    }
	}
    }


    /**
     * Bits im Wort zhlen ohne die eigentlich naheliegende Schleife.
     */
    int zaehleShortBits(int zahl) {
	zahl = zahl - ((zahl >> 1) & 0x5555);
	zahl = (zahl & 0x3333) + ((zahl >> 2) & 0x3333);
	zahl = (zahl & 0x0f0f) + ((zahl >> 4) & 0x0f0f);
	return (zahl & 0x00ff) + ((zahl >> 8) & 0x00ff);
    }


    /**
     * Stein initialisieren.
     */
    void initStein(Stein *stein, int64 basisform, bool symmetrieFilter) {
	memset(stein, 0, sizeof(stein));
	stein->basisform = basisform;
	if (symmetrieFilter) {
	    berechneSymmetrieVarianten(stein);
	}
	else {
            berechneAlleVarianten(stein);
	}
	sortiereVarianten(stein);
    }


    /**
     * Kopiert den Stein von QUELLE nach ZIEL, lscht dabei aber 
     * alle Varianten, die mit BELEGT kollidieren.
     * Das im Betrieb irrelevante Feld 'variantenEndspiel' bleibt leer.
     */
    void steinFiltern(Stein *ziel, Stein *quelle, int64 belegt)
    {
	ziel->basisform = quelle->basisform;
	ziel->variantenEndspiel[0] = 0;

	int nAlle = 0;
	for (int64 *pa=quelle->alleVarianten; *pa!=0; pa++) {
	    int64 variante = *pa;
	    if ((variante & belegt) == 0) {
		ziel->alleVarianten[nAlle++] = variante;
	    }
	}
	ziel->alleVarianten[nAlle] = 0;

	for (int e=0; e<SUCH_EBENEN; e++) {
	    int64 relBelegt = belegt >> (12 * e);
	    for (int pos=0; pos<12; pos++) {
		int iq=0;
		int iz=0;
		for (int is=0; is<=18; is++) {
		    int schl=quelle->schluessel[e][pos][is];
		    while (iq < schl) {
			int64 variante = quelle->relVarianten[e][pos][iq++];
			if ((variante & relBelegt) == 0) {
			    ziel->relVarianten[e][pos][iz++] = variante;
			}
		    }
		    ziel->schluessel[e][pos][is] = iz;
		}
	    }
	}
    }

    /**
     * Die 12 Steine des Puzzles in individueller Reihenfolge
     */
    void initSteine(void) {
	initStein(&basisSteine[0],  020013, false);
	initStein(&basisSteine[1],  010033, false);
	initStein(&basisSteine[2],    0472, false);
	initStein(&basisSteine[3],    0323, false);
	initStein(&basisSteine[4],    0326, false);
	initStein(&basisSteine[5],   01113, false);
	initStein(&basisSteine[6],   02322, false);
	initStein(&basisSteine[7],    0722, false);
	initStein(&basisSteine[8],    0133, false);
	initStein(&basisSteine[9],    0136, false);
	initStein(&basisSteine[10],  02323, true);
	initStein(&basisSteine[11],   0272, false);
    }


    /**
     * Initialisierung von 'reihenfolge'.
     */
    void initReihenfolge(void) {
	for (int i=0; i<60; i++) {
	    reihenfolge[i] = ((int64) 1) << BIT_REIHE_INV_RINGE[i];
	}
    }


    /**
     * Hotspot-Datenstruktur initialisieren...
     */
    void initHotspot(short hotspotSchluessel[4096][4], int nBits) {

	// Fr alle 4096 Belegungen einer 3*4-Ebene...
        for (int pat=0; pat<0x1000; pat++) {
            int frei = 0xfff - pat;
	    int hotspot = -1;
            int nHotspot = 999999;

	    // Welche Lcke hat die wenigsten Nachbarn?
	    // Das ist der Hotspot dieser Belegung
            for (int p=0, maske=1; p<nBits; p++, maske<<=1) {
                if ((frei & maske) != 0) {
		    int n = zaehleShortBits(frei & NACHBARN[p]);
                    if (n < nHotspot) {
                        nHotspot = n;
                        hotspot = p;
                    }
                }
            }
            hotspotSchluessel[pat][0] = hotspot;

	    //
	    // Schlssel ablegen, damit die Suche die "richtige" Teilmenge
	    // der 'relVarianten' findet.
	    // Das implementierte Verfahren liefert der Suche eine vorgefilterte
	    // Teilmenge, die mit den 4 Nachbarn garantiert kollisionsfrei ist.
	    //

	    // Konfiguration der maximal 4 Nachbarn um den Hotspot:
	    // Es gibt davon 16 Stck, benannt A (0000) bis P (1111).
	    // Einige entfallen, weil ein Hotspot nie im Leeren steht.
	    // (Der Hotspot steht durch Shiften an Bitposition 0x800)
	    int32 konfig = ((~pat) & NACHBARN[hotspot]) << (11 - hotspot);

	    // Welche Steine-Varianten passen?
	    // Variantengruppen heien je nach Belegung der Nachbarn
	    // auch A bis P.
	    // Varianten A passen immer,
	    // Varianten B bei Konfiguration B, D, F, J
	    // usw.
	    // Stein trgt in 'relVarianten' eine Variantenliste 
	    // laut String DBACGEAFBJIACKAEIM.
	    // Welcher Ausschnitt (Start- und End-Index) aus DBACGEAFBJIACKAEIM ?
	    switch (konfig) {
	    case 0x0000:    // Konfig A braucht A
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 3;
		break;
	    case 0x0100:    // B braucht AB
		hotspotSchluessel[pat][1] = 1;
		hotspotSchluessel[pat][2] = 3;
		break;
	    case 0x0400:    // C braucht AC
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 4;
		break;
	    case 0x0500:    // D braucht ABCD
		hotspotSchluessel[pat][1] = 0;
		hotspotSchluessel[pat][2] = 4;
		break;
	    case 0x1000:    // E braucht AE
		hotspotSchluessel[pat][1] = 5;
		hotspotSchluessel[pat][2] = 7;
		break;
	    case 0x1100:    // F braucht ABEF
		hotspotSchluessel[pat][1] = 5;
		hotspotSchluessel[pat][2] = 9;
		break;
	    case 0x1400:    // G braucht ACEG
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 5;
		break;
	    case 0x4000:    // I braucht AI
		hotspotSchluessel[pat][1] = 10;
		hotspotSchluessel[pat][2] = 12;
		break;
	    case 0x4100:    // J braucht ABIJ
		hotspotSchluessel[pat][1] = 8;
		hotspotSchluessel[pat][2] = 12;
		break;
	    case 0x4400:    // K braucht ACIK
		hotspotSchluessel[pat][1] = 10;
		hotspotSchluessel[pat][2] = 14;
		break;
	    case 0x5000:    // M braucht AEIM
		hotspotSchluessel[pat][1] = 14;
		hotspotSchluessel[pat][2] = 18;
		break;

	    default:
		printf("Logikfehler: Bild %03x, Pos %d ergibt %04x\n",
		       pat, hotspot, konfig);
		exit(1);
	    }
        }
    }

    /**
     * Nullen der Endspiel-Daten.
     */
    void initEndspiele(void) {
	memset(endspiele, 0, sizeof(endspiele));
	memset(extender, 0, sizeof(extender));
    }


    /**
     * Notiert das Komplement einer vorwrts gefundenen 
     * Steine-Konfiguration als Endspiel-Lsung.
     * @param verbrauch Verbrauch aus Vorwrts-Suche
     * @param belegung Belegung aus Vorwrts-Suche
     */
    void merkeEndspiel(int16 verbrauch, int64 belegung) {
	int32 index = ((~belegung) >> 36) & 0xffffff;
	int16 muster = 
	    ((~verbrauch) & 0x3ff) + 
	    (((~belegung) & ENDSPIEL_RESTMASKE) >> 20);
	int i;
	EndspielExtender **quelle = &endspiele[index];
	EndspielExtender *ex = *quelle;
	// Bis zum letzten Extender in der Kette springen
	for ( ; ex!=NULL; ex=ex->next) {
	    for (int i=0; i<7; i++) {
		if (ex->loesungen[i].muster == muster) {
		    ex->loesungen[i].nloesungen++;
		    return;
		}
		if (ex->loesungen[i].nloesungen == 0) {
		    ex->loesungen[i].muster = muster;
		    ex->loesungen[i].nloesungen = 1;
		    return;
		}
	    }
	    quelle = &ex->next;
	}

	// Neuen Extender anhngen
	ex = &extender[nExtender++];
	*quelle = ex;
	ex->loesungen[0].muster = muster;
	ex->loesungen[0].nloesungen = 1;

	if (nExtender >= EXTENDER_SIZE) {
	    printf("Groesse des EndspielExtenders reicht nicht.\n");
	}
    }


    /**
     * Rekursives Fllen des Endspiel-Speichers. Geht ber die Steine
     * und nicht ber die Positionen. Auf jeder Aufruf-Ebene wird ein
     * Stein in jeder erlaubten Lage gesetzt oder auch gar nicht gesetzt.
     * 
     * @param steinNum Nummer des Steins
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchenEndspielInvers(int steinNum,
                              int16 verbrauchsMaske,
                              int64 belegung) {
        if (steinNum >= 10) {
	    merkeEndspiel(verbrauchsMaske, belegung);
	    return;
        }

        int nextNum = steinNum + 1;
        int nextVm = verbrauchsMaske | (1<<steinNum);
        int64 *varp = basisSteine[steinNum].variantenEndspiel;
	int64 steinForm;
        while ((steinForm = *varp++) != 0) {
            if ((steinForm & belegung) == 0) {
                int64 belNeu = belegung | steinForm;
		if (isGesund2(belNeu)) {
		    suchenEndspielInvers(nextNum, nextVm, belNeu);
		}
            }
        }
        suchenEndspielInvers(nextNum, verbrauchsMaske, belegung);
    }


    /**
     * Vor-Initialisieren der Endspielvarianten .
     */
    void initMoeglichkeitenMitInverserSuche(void) {
	initEndspiele();
        suchenEndspielInvers(0, 0, 0);

	printf("Circa %d Endspiele vorberechnet.\n", nExtender);
    }


    /**
     * Sucht eine gegebene Konfiguration im Endspiel-Speicher.
     * @param verbrauchsMaske Verbrauch bisher
     * @param belegung Belegung, 24 Bit geshiftet, wie fr Ebene2.
     */
    inline void suchenE(int verbrauchsMaske, int64 belegung) {

#ifdef ZAEHLEN
	nEndspiele++;
#endif

	int32 index = (int32)(belegung >> 12) & 0xffffff;
	int16 muster = 
	    ((belegung << 4) & (ENDSPIEL_RESTMASKE >> 20)) | 
	    (verbrauchsMaske & 0x3ff);

        EndspielExtender *exp = endspiele[index];
	for ( ; exp!=NULL; exp=exp->next) {
	    for (int i=0; i<7; i++) {
		if (exp->loesungen[i].muster == muster) {
		    nLoesungen += exp->loesungen[i].nloesungen;
		    return;
		}
		if (exp->loesungen[i].nloesungen == 0) {
		    return;
		}
	    }
	}
    }	

    /**
     * Rekursives Suchen auf Ebene 2.
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen,
     *                 um 24 Bit abwrts geshiftet
     */
    void suchen2(int verbrauchsMaske, int64 belegung) {
	if (!isGesund(belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluesselE2[schnitt][0];
	if (relpos < 0) {
	    suchenE(verbrauchsMaske, belegung);
	    return;
	}

	int sStart = hotspotSchluesselE2[schnitt][1];
	int sEnde = hotspotSchluesselE2[schnitt][2];
	int i;
	int sMaske;

        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=0, sMaske=1; i<10; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[2][relpos];
		int start = steine[i].schluessel[2][relpos][sStart];
		int ende = steine[i].schluessel[2][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) 
		{
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {
#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen2(vmNeu, belNeu);
                    }
                }
            }
        }
    }

    /**
     * Rekursives Suchen auf Ebene 1.
     * 
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen,
     *                 um 12 Bit abwrts geshiftet.
     */
    void suchen1(int verbrauchsMaske, int64 belegung) {
	if (!isGesund(belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluessel[schnitt][0];
	if (relpos < 0) {
	    suchen2(verbrauchsMaske, (belegung >> 12) | (int64) 0xfff000000000000);
	    return;
	}

	int sStart = hotspotSchluessel[schnitt][1];
	int sEnde = hotspotSchluessel[schnitt][2];
	int i;
	int sMaske;

        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=0, sMaske=1; i<10; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[1][relpos];
		int start = steine[i].schluessel[1][relpos][sStart];
		int ende = steine[i].schluessel[1][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) 
		{
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {
#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen1(vmNeu, belNeu);
                    }
                }
            }
        }
    }

    /**
     * Rekursives Suchen auf Ebene 0.
     * 
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchen0(int verbrauchsMaske, int64 belegung) {
	if (!isGesund(belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluessel[schnitt][0];
	if (relpos < 0) {
	    suchen1(verbrauchsMaske, (belegung >> 12) | (int64) 0xfff000000000000);
	    return;
	}

	int sStart = hotspotSchluessel[schnitt][1];
	int sEnde = hotspotSchluessel[schnitt][2];
	int i;
	int sMaske;


        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=0, sMaske=1; i<10; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[0][relpos];
		int start = steine[i].schluessel[0][relpos][sStart];
		int ende = steine[i].schluessel[0][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) {
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {

#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen0(vmNeu, belNeu);
                    }
                }
            }
        }
    }


    /**
     * Ablauf der Suche. Steine 10 und 11 werden vorab in allen Varianten 
     * gesetzt. Fr alle anderen Steine gilt das ebenenweise Fllen 
     * mit dynamischer Bitreihenfolge (Hotspot).
     */
    void suchablauf() {
	start = time(NULL);
	initReihenfolge();
	initHotspot(hotspotSchluessel, 12);
	initHotspot(hotspotSchluesselE2, ENDSPIEL_START - 24);
	initSteine();
	for (int j=0; j<12; j++) {
	    steine[j] = basisSteine[j];
	}
	initMoeglichkeitenMitInverserSuche();

        // Variante: Stein 0 vorab platzieren
        int64 *ps0;
	int64 stein0;
	int i0;

	for (ps0 = basisSteine[10].alleVarianten, i0 = 1; 
	     (stein0=*ps0) != 0; 
	     ps0++, i0++) 
	{
	    int64 *ps1;
	    int64 stein1;
	    int i1;

	    for (ps1 = basisSteine[11].alleVarianten, i1 = 1; 
	         (stein1=*ps1) != 0; 
	         ps1++, i1++) 
	    {
		if ((stein1 & stein0) == 0) {
		    int64 belegung = stein0 | stein1;

		    // Varianten ausfiltern, die mit den schon gesetzten
		    // Steinen 10 und 11 kollidieren.
		    for (int j=0; j<10; j++) {
			steinFiltern(&steine[j], &basisSteine[j], belegung);
		    }

		    suchen0(0xc00, belegung);
		}
	    }

	    time_t schritt = time(NULL);
#ifdef ZAEHLEN
	    printf ("%2d von 56: %4d sec bei %6d Lsg. nach %5d/%5d/%5d Mio. Schritten...\n",
		    i0, schritt - start, nLoesungen, 
		    nSuchvorgaenge / 1000000,
		    (int) (nVersuche / 1000000),
		    nEndspiele / 1000000);
#else
	    printf ("%2d von 56: %4d sec bei %6d Lsg.\n",
		    i0, schritt - start, nLoesungen);
#endif

	}

	time_t ende = time(NULL);
	printf("%d Loesungen nach %d Sekunden.\n",
		nLoesungen, ende - start);
    }


int main(int argc, char* argv[])
{
    suchablauf();
    return 0;
}

