// ctpuzzle.cpp : Definiert den Einsprungpunkt fr die Konsolenanwendung.
//

#include "stdafx.h"

// #define ZAEHLEN

typedef unsigned __int64 int64;
typedef unsigned long int32;
typedef unsigned short int16;
typedef unsigned char byte;


#define SHIFT_PLUSX(x)   (((x)&(int64)033333333333333333333)<<1)
#define SHIFT_PLUS2X(x)  (((x)&(int64)011111111111111111111)<<2)
#define SHIFT_MINUSX(x)  (((x)&(int64)066666666666666666666)>>1)
#define SHIFT_MINUS2X(x) (((x)&(int64)044444444444444444444)>>2)
#define SHIFT_PLUSY(x)   (((x)&(int64)007770777077707770777)<<3)
#define SHIFT_PLUS2Y(x)  (((x)&(int64)000770077007700770077)<<6)
#define SHIFT_MINUSY(x)  (((x)&(int64)077707770777077707770)>>3)
#define SHIFT_MINUS2Y(x) (((x)&(int64)077007700770077007700)>>6)
#define SHIFT_PLUSZ(x)   (((x)&(int64)000007777777777777777)<<12)
#define SHIFT_PLUS2Z(x)  (((x)&(int64)000000000777777777777)<<24)
#define SHIFT_MINUSZ(x)  (((x)&(int64)077777777777777770000)>>12)
#define SHIFT_MINUS2Z(x) (((x)&(int64)077777777777700000000)>>24)


//
// -------------------------
//      Datenstrukturen
// -------------------------
//


#define N_VARIANTEN 128
#define SUCH_EBENEN 5

typedef struct {
    int64 basisform;
    int64 relVarianten[5][12][N_VARIANTEN];
    short schluessel[5][12][20];
    int64 alleVarianten[577];
    int32 variantenBis24[145];
} Stein;

typedef struct extender {
    int16 verbrauch[30];
    struct extender *next;
} EndspielExtender;

typedef struct {
    int64 belegung;
    int16 verbrauch;
    int nLoesungen;
} Aufgabe;

typedef union {
    int16 verbrauch[4];
    EndspielExtender *extension[2];
} Endspiel;


typedef struct {
    int x;
    int y;
    int z;
    bool valid;
} XYZ;


//
// -------------------------
//        Variablen
// -------------------------
//

time_t start;

int64 reihenfolge[60];

Stein steine[12];
Stein basisSteine[12];

// [0] = Hotspot, [1] = Start-Schlssel, [2] = End-Schlssel
short hotspotSchluessel[0x1000][4];

Endspiel endspiele[0x1000000];

EndspielExtender extender[32000];

int32 nExtender = 0;

int BIT_REIHE_INV_RINGE[] = {
    0 , 1, 2, 5, 8,11,10, 9, 6, 3, 4, 7,
    12,13,14,17,20,23,22,21,18,15,16,19,
    24,25,26,29,32,35,34,33,30,27,28,31,
    36,37,38,41,44,47,46,45,42,39,40,43,
    48,49,50,53,56,59,58,57,54,51,52,55
};

/** Bitmuster der Nachbarn in einer Ebene. */
int NACHBARN[] = {
    00012, 00025, 00042,
    00121, 00252, 00424,
    01210, 02520, 04240,
    02100, 05200, 02400
};

/** Bitmuster der Nachbarn in einer Ebene. */
int DIAG_NACHBARN[] = {
    00020, 00050, 00020,
    00202, 00505, 00202,
    02020, 05050, 02020,
    00200, 00500, 00200
};


int N_BITS[] = {
    0, 1, 1, 2, 1, 2, 2, 3, 1, 2, 2, 3, 2, 3, 3, 4
};

int SW_REICHWEITE[0x1000]; 

Aufgabe aufgaben[3000];
int nAufgaben;
int naechsteAufgabe = 0;
int nGeloest = 0;

int threads[] = {0, 1, 2, 3};

Stein threadSteine[4][12];

int32 threadLoesungen[4];

HANDLE hAufgabeMutex;

int32 nSuchvorgaenge = 0;
int64 nVersuche = 0;
int32 nEndspiele = 0;

int32 nLoesungen;

//
// -------------------------
//        Funktionen
// -------------------------
//


    /**
     * Liefert True, wenn die Koordinaten im vorgegebenen 3*4*5-Quader liegen.
     * @param koords Array der Koordinaten, jeweils als {x, y, z}
     * @return True, wenn alle im 3*4*5-Quader liegen
     */
    bool isGueltig(XYZ koords[]) {
	XYZ *p;
        for (p=koords; p->valid; p++) {
            if (!((p->x >= 0) && (p->x < 3) &&
                  (p->y >= 0) && (p->y < 4) &&
                  (p->z >= 0) && (p->z < 5))) {
                return false;
            }
        }
        return true;
    }

    inline XYZ newXYZ(int x, int y, int z) {
	XYZ pt;
	pt.x = x;
	pt.y = y;
	pt.z = z;
	pt.valid = true;
	return pt;
    }


    /**
     * Verschiebt die Punkte in den positiven Oktanten, auf jeder Achse
     * beginnend mit 0 als kleinstem Wert.
     */
    void verschiebeNachNull(XYZ koords[]) {
        int xMin = 1000;
        int yMin = 1000;
        int zMin = 1000;
	XYZ *kp;

        for (kp=koords; kp->valid; kp++) {
            if (kp->x < xMin) xMin = kp->x;
            if (kp->y < yMin) yMin = kp->y;
            if (kp->z < zMin) zMin = kp->z;
        }
        for (kp=koords; kp->valid; kp++) {
            kp->x -= xMin;
            kp->y -= yMin;
            kp->z -= zMin;
        }
    }

    /**
     * Verschiebt den Stein um DX, DY, DZ.
     */
    void verschiebe(XYZ koords[], XYZ ziel[], int dx, int dy, int dz) {
	XYZ *kp, *zp;
        for (kp=koords, zp=ziel; kp->valid; kp++, zp++) {
            ziel->x = kp->x + dx;
            ziel->y = kp->y + dy;
            ziel->z = kp->z + dz;
        }
	zp->valid = false;
    }

    /**
     * Verschiebt den Stein um DX, DY, DZ.
     */
    int64 verschiebeBits(int64 bits, int dx, int dy, int dz) {
        int shift = dx + 3 * dy + 12 * dz;
        return bits << shift;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der X-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int xAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->x > maxwert) maxwert = kp->x;
        }
        return maxwert + 1;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der Y-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int yAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->y > maxwert) maxwert = kp->y;
        }
        return maxwert + 1;
    }

    /**
     * Bestimmt die Ausdehnung des Steins in Richtung der Z-Achse.
     * Es wird vorausgesetzt, dass der Stein an Null-Position liegt.
     * @param koords Stein als Koordinaten-Array
     * @return Ausdehnung in der angegebenen Achse
     */
    int zAusdehnung(XYZ koords[]) {
        int maxwert = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
            if (kp->z > maxwert) maxwert = kp->z;
        }
        return maxwert + 1;
    }

    /**
     * Wandelt von Koordinaten-Array in Bitvektor um.
     */
    int64 getBits(XYZ koords[]) {
        int64 result = 0;
	XYZ *kp;
        for (kp=koords; kp->valid; kp++) {
	    result += ((int64) 1) <<
		(kp->x + 3*kp->y + 12*kp->z);
        }
        return result;
    }

    /**
     * Wandelt von Bitvektor in Koordinaten-Array um.
     */
    void getKoords(int64 bits, XYZ ziel[]) {
        XYZ *zp = ziel;
	int64 maske = 1;
	int x, y, z;
        for (z=0; z<5; z++) {
            for (y=0; y<4; y++) {
                for (x=0; x<3; x++) {
                    if ((bits & maske) != 0) {
			*zp++ = newXYZ(x, y, z);
                    }
                    maske = maske << 1;
                }
            }
        }
	zp->valid = false;
    }

    /**
     * Rotiert einen Punkt um den Quader-Mittelpunkt.
     * @param punkt originaler Punkt
     * @param orientierung Version von 0 bis 23
     * @return rotierter Punkt.
     */
    XYZ rotiereXYZ(XYZ punkt, int orientierung) {
        int x = punkt.x;
        int y = punkt.y;
        int z = punkt.z;

        switch (orientierung) {
            case  0: return newXYZ(x,   y,   z);
            case  1: return newXYZ(2-x, 3-y, z);
            case  2: return newXYZ(2-x, y,   4-z);
            case  3: return newXYZ(x,   3-y, 4-z);
            case  4: return newXYZ(y,   z,   x);
            case  5: return newXYZ(2-y, 3-z, x);
            case  6: return newXYZ(2-y, z,   4-x);
            case  7: return newXYZ(y,   3-z, 4-x);
            case  8: return newXYZ(z,   x,   y);
            case  9: return newXYZ(2-z, 3-x, y);
            case 10: return newXYZ(2-z, x,   4-y);
            case 11: return newXYZ(z,   3-x, 4-y);
            case 12: return newXYZ(2-x, 3-z, 4-y);
            case 13: return newXYZ(x,   z,   4-y);
            case 14: return newXYZ(x,   3-z, y);
            case 15: return newXYZ(2-x, z,   y);
            case 16: return newXYZ(2-z, 3-y, 4-x);
            case 17: return newXYZ(z,   y,   4-x);
            case 18: return newXYZ(z,   3-y, x);
            case 19: return newXYZ(2-z, y,   x);
            case 20: return newXYZ(2-y, 3-x, 4-z);
            case 21: return newXYZ(y,   x,   4-z);
            case 22: return newXYZ(y,   3-x, z);
            case 23: return newXYZ(2-y, x,   z);
            default: return punkt;
        }
    }

    void rotiere(XYZ koords[], XYZ ziel[], int orientierung) {
	XYZ *kp, *zp;
        for (kp=koords, zp=ziel; kp->valid; kp++, zp++) {
            *zp = rotiereXYZ(*kp, orientierung);
        }
	zp->valid = false;
    }

    int64 rotiereBits(int64 bits, int orientierung) {
	XYZ koords[7], rotiert[7];
	getKoords(bits, koords);
	rotiere(koords, rotiert, orientierung);
	return getBits(rotiert);
    }

    /**
     * Liefert zu einer Stein-Lage den Start-Schritt gem eingestellter
     * Schritt-Reihenfolge der Instanz.
     */
    int getStartPos(int64 bits) {
        int i;
        for (i=0; i<60; i++) {
            if ((bits & reihenfolge[i]) != 0) {
                return i;
            }
        }
        return 63;
    }

    /**
     * Gibt es irgendwo ein isoliertes Einer-Loch?
     */
    inline bool isGesund(int64 belegung) {
	int64 freie = (~belegung) & (int64) 077777777777777777777;
        int64 muster = 
	    belegung |
            ((freie & (int64) 033333333333333333333) <<  1) |
            ((freie & (int64) 066666666666666666666) >>  1) |
            ((freie & (int64) 007770777077707770777) <<  3) |
            ((freie & (int64) 077707770777077707770) >>  3) |
            ((freie & (int64) 000007777777777777777) << 12) |
            ((freie & (int64) 077777777777777770000) >> 12);
        return (muster  == (int64) 077777777777777777777);
    }

    /**
     * Gibt es irgendwo ein isoliertes Einer- oder Zweier-Loch?
     */
    inline bool isGesund2(int64 bel) {
	int64 frei = (~bel) & (int64) 077777777777777777777;
        int64 kreuz;

	kreuz = 
	    bel |
	    SHIFT_PLUSY(frei) | 
	    SHIFT_MINUSY(frei) |
	    SHIFT_PLUSZ(frei) |
	    SHIFT_MINUSZ(frei);
	if ((SHIFT_PLUSX(frei) | kreuz | SHIFT_MINUSX(frei))
	    != 077777777777777777777) {
	    return false;
	}
	if ((SHIFT_PLUSX(frei) | kreuz | SHIFT_MINUSX(kreuz) | SHIFT_MINUS2X(frei))
	    != 077777777777777777777) {
	    return false;
	}

	kreuz =
	    bel | 
	    SHIFT_PLUSX(frei) |
	    SHIFT_MINUSX(frei) |
	    SHIFT_PLUSZ(frei) |
	    SHIFT_MINUSZ(frei);
	if ((SHIFT_PLUSY(frei) | kreuz | SHIFT_MINUSY(kreuz) | SHIFT_MINUS2Y(frei))
	    != 077777777777777777777) {
	    return false;
	}

	kreuz = 
	    bel | 
	    SHIFT_PLUSX(frei) |
	    SHIFT_MINUSX(frei) |
	    SHIFT_PLUSY(frei) |
	    SHIFT_MINUSY(frei);
	if ((SHIFT_PLUSZ(frei) | kreuz | SHIFT_MINUSZ(kreuz) | SHIFT_MINUS2Z(frei))
	    != 077777777777777777777) {
	    return false;
	}

        return true;
    }

    inline int schwarzUeberschuss(int64 muster) {
	muster = muster ^ (int64) 0x555aaa555aaa555;

	muster = muster - 
	         ((muster >>  1) & (int64) 0x555555555555555);
	muster = (muster         & (int64) 0x333333333333333) +
		 ((muster >>  2) & (int64) 0x333333333333333);
	muster = (muster         & (int64) 0xf0f0f0f0f0f0f0f) +
		 ((muster >>  4) & (int64) 0xf0f0f0f0f0f0f0f);
	muster = (muster         & (int64) 0x0ff00ff00ff00ff) +
		 ((muster >>  8) & (int64) 0x0ff00ff00ff00ff);
	muster = (muster         & (int64) 0x000ffff0000ffff) +
		 ((muster >> 16) & (int64) 0x000ffff0000ffff);
	return (int32) muster + (int32) (muster >> 32) - 30;
    }

    /**
     * Bestimmt den berschuss an belegten schwarzen gegenber weien Feldern
     * und testet, ob er mit den verfgbaren Steinen noch kompensiert werden kann.
     * Jede 3*4-Schicht ist mit einem Schachbrettmuster versehen, zwischen
     * den Schichten werden jeweils Schwarz und Wei vertauscht.
     */
    inline bool swTesten(int verbrauch, int64 belegung) {
        int ueberschuss = schwarzUeberschuss(belegung);
        int reichweite = SW_REICHWEITE[verbrauch];
        return (abs(ueberschuss) <= reichweite);
    }
 
    void berechneAlleVarianten(Stein *stein) {
        XYZ koords[7];
	getKoords(stein->basisform, koords);

        // Alle rotierten Versionen erzeugen
        int64 alleRotiertenVersionen[24];
	int nRotierteVersionen = 0;
        for (int orientierung=0; orientierung<24; orientierung++) {
            XYZ koordsNeu[7];
	    rotiere(koords, koordsNeu, orientierung);
            verschiebeNachNull(koordsNeu);
	    int64 rotiert = getBits(koordsNeu);
            if (isGueltig(koordsNeu)) {
		bool istNeu = true;
		for (int i=0; i<nRotierteVersionen; i++) {
		    if (alleRotiertenVersionen[i] == rotiert) {
			istNeu = false;
			break;
		    }
		}
		if (istNeu) {
		    alleRotiertenVersionen[nRotierteVersionen++] = rotiert;
		}
            }
        }

	int nAlle = 0;

        for (int i=0; i<nRotierteVersionen; i++) {
            int64 basisVersion = alleRotiertenVersionen[i];
	    XYZ koordsVersion[7];
            getKoords(basisVersion, koordsVersion);
            int xSchritte = 4 - xAusdehnung(koordsVersion);
            int ySchritte = 5 - yAusdehnung(koordsVersion);
            int zSchritte = 6 - zAusdehnung(koordsVersion);
            for (int z=0; z<zSchritte; z++) {
                for (int y=0; y<ySchritte; y++) {
                    for (int x=0; x<xSchritte; x++) {
                        int64 verschoben = 
			    verschiebeBits(basisVersion, x, y, z);
			stein->alleVarianten[nAlle++] = verschoben;
                    }
                }
            }
        }
    }


    bool gesundheitTesten(int verbrauchsMaske, int64 belegung) {
        int64 belegbar = 0;
        for (int i=1, sMaske=2; i<12; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
		int64 variante;
		int64 *varp;
                for (varp=steine[i].alleVarianten; (variante=*varp)!=0; varp++) {
                    if ((belegung & variante) == 0) {
			belegbar |= variante;
                    }
                }
            }
        }
        return ((belegbar | belegung) == (int64) 077777777777777777777);
    }

    // Ist noch irgendwo eine gerade Dreier-Reihe frei?
    // Wird fr fast alle Steine gebraucht.
    inline bool reiheTesten(int vm, int64 bel) {
	// Kein zutreffender Stein mehr?
	// Also Test nicht anwendbar, True zurck.
	if ((vm & 0x9fd) == 0x9fd) {
	    return true;
	}

	int64 frei = (~bel) & (int64) 077777777777777777777;
	int64 mx = SHIFT_MINUSX(frei);
	int64 px = SHIFT_PLUSX(frei);
	int64 my = SHIFT_MINUSY(frei);
	int64 py = SHIFT_PLUSY(frei);
	int64 mz = SHIFT_MINUSZ(frei);
	int64 pz = SHIFT_PLUSZ(frei);
	if (((mx & frei & px) != 0) ||
	    ((my & frei & py) != 0) ||
	    ((mz & frei & pz) != 0)) {
	    if (((vm & 0x061) == 0x061) ||
		((my & frei & py & SHIFT_PLUSY(py)) != 0) ||
		((mz & frei & pz & SHIFT_PLUSZ(pz)) != 0)) {
		return true;
	    }
	}
	return false;
    }

    void berechneSymmetrieVarianten(Stein *stein) {
        XYZ koords[7];
	getKoords(stein->basisform, koords);

        // Alle rotierten Versionen erzeugen
        int64 alleRotiertenVersionen[24];
	int nRotierteVersionen = 0;
        for (int orientierung=0; orientierung<24; orientierung+=4) {
            XYZ koordsNeu[7];
	    rotiere(koords, koordsNeu, orientierung);
            verschiebeNachNull(koordsNeu);
	    int64 rotiert = getBits(koordsNeu);
            if (isGueltig(koordsNeu)) {
		bool istNeu = true;
		for (int i=0; i<nRotierteVersionen; i++) {
		    if (alleRotiertenVersionen[i] == rotiert) {
			istNeu = false;
			break;
		    }
		}
		if (istNeu) {
		    alleRotiertenVersionen[nRotierteVersionen++] = rotiert;
		}
            }
        }

	int nAlle = 0;
        for (int i=0; i<nRotierteVersionen; i++) {
            int64 basisVersion = alleRotiertenVersionen[i];
	    XYZ koordsVersion[7];
            getKoords(basisVersion, koordsVersion);
            int xSchritte = 4 - xAusdehnung(koordsVersion);
            int ySchritte = 5 - yAusdehnung(koordsVersion);
            int zSchritte = 6 - zAusdehnung(koordsVersion);
            for (int z=0; z<zSchritte; z++) {
                for (int y=0; y<ySchritte; y++) {
                    for (int x=0; x<xSchritte; x++) {
                        int64 verschoben1 = 
			    verschiebeBits(basisVersion, x, y, z);
                        int startbit1 = getStartPos(verschoben1);
                        int64 verschoben2 = rotiereBits(verschoben1, 1);
                        int startbit2 = getStartPos(verschoben2);
                        int64 verschoben3 = rotiereBits(verschoben1, 2);
                        int startbit3 = getStartPos(verschoben3);
                        int64 verschoben4 = rotiereBits(verschoben1, 3);
                        int startbit4 = getStartPos(verschoben4);


			int64 kandidat = verschoben1;
                        int kleinstePos = startbit1;
                        if (startbit2 < kleinstePos) {
                            kandidat = verschoben2;
                            kleinstePos = startbit2;
                        }
                        if (startbit3 < kleinstePos) {
                            kandidat = verschoben3;
                            kleinstePos = startbit3;
                        }
                        if (startbit4 < kleinstePos) {
                            kandidat = verschoben4;
                            kleinstePos = startbit4;
                        }

			stein->alleVarianten[nAlle++] = kandidat;
                    }
                }
            }
        }
    }

    int64 listeA[145];
    int64 listeB[145];
    int64 listeC[145];
    int64 listeD[145];
    int64 listeE[145];
    int64 listeF[145];
    int64 listeG[145];
    int64 listeI[145];
    int64 listeJ[145];
    int64 listeK[145];
    int64 listeM[145];

    void sortiereVarianten(Stein *stein) {
	int nBis24 = 0;
	for (int64 *p=stein->alleVarianten; *p!=0; p++) {
	    int64 variante = *p;
	    if ((variante & (int64) 0x000000fffffffff) == 0) {
		stein->variantenBis24[nBis24++] = (int32) (variante >> 36);
	    }
	}
	stein->variantenBis24[nBis24] = 0;

	int maxIndex = 0;

	for (int ebene=0; ebene<SUCH_EBENEN; ebene++) {
	    int64 nullMaske = 0xffffffffffff >> (12 * (4 - ebene));
	    for (int relpos=0; relpos<12; relpos++) {
		int64 nonNullMaske = ((int64) 1) << (12 * ebene + relpos);
		int nA=0, nB=0, nC=0, nD=0, nE=0, nF=0, nG=0, nI=0, nJ=0, nK=0, nM=0;

		for (int64 *p2=stein->alleVarianten; *p2!=0; p2++) {
		    int64 variante = *p2;
		    if (((variante & nonNullMaske) != 0) &&
			((variante & nullMaske) == 0)) {
			int64 relVariante = variante >> (12 * ebene);
			int32 schnitt = relVariante & 0xfff;
			int32 pattern = (schnitt & NACHBARN[relpos]) << (11 - relpos);
			switch (pattern) {
			case 0x0000:
			    listeA[nA++] = relVariante;
			    break;
			case 0x0100:
			    listeB[nB++] = relVariante;
			    break;
			case 0x0400:
			    listeC[nC++] = relVariante;
			    break;
			case 0x0500:
			    listeD[nD++] = relVariante;
			    break;
			case 0x1000:
			    listeE[nE++] = relVariante;
			    break;
			case 0x1100:
			    listeF[nF++] = relVariante;
			    break;
			case 0x1400:
			    listeG[nG++] = relVariante;
			    break;
			case 0x4000:
			    listeI[nI++] = relVariante;
			    break;
			case 0x4100:
			    listeJ[nJ++] = relVariante;
			    break;
			case 0x4400:
			    listeK[nK++] = relVariante;
			    break;
			case 0x5000:
			    listeM[nM++] = relVariante;
			    break;
			case 0x1500:
			case 0x4500:
			case 0x5100:
			case 0x5400:
			case 0x5500:
			    // Diese Patterns brauchen nicht indiziert zu werden!
			    // Wegen Hotspot-Reihenfolge werden sie nie an dieser 
			    // Startposition auftreten.
			    break;
			default:
			    printf("Logikfehler bei Stein %07x%08x, Ebene %d, Pos %d:\n",
				   (int32) (stein->basisform >> 32),
				   (int32) stein->basisform,
				   ebene, relpos);
			    printf("   Variante %07x%08x ergibt %04x\n",
				   (int32) (relVariante >> 32),
				   (int32) relVariante,
				   pattern);
			    exit(1);
			}
		    }
		}
		int64 *pl = stein->relVarianten[ebene][relpos];
		int index = 0;
		short *ps = stein->schluessel[ebene][relpos];
		*ps++ = index;

		memcpy(&pl[index], listeD, nD * sizeof(int64));
		index += nD;
		*ps++ = index;

		memcpy(&pl[index], listeB, nB * sizeof(int64));
		index += nB;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeC, nC * sizeof(int64));
		index += nC;
		*ps++ = index;

		memcpy(&pl[index], listeG, nG * sizeof(int64));
		index += nG;
		*ps++ = index;

		memcpy(&pl[index], listeE, nE * sizeof(int64));
		index += nE;
		*ps++ = index;
		
		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeF, nF * sizeof(int64));
		index += nF;
		*ps++ = index;

		memcpy(&pl[index], listeB, nB * sizeof(int64));
		index += nB;
		*ps++ = index;

		memcpy(&pl[index], listeJ, nJ * sizeof(int64));
		index += nJ;
		*ps++ = index;

		memcpy(&pl[index], listeI, nI * sizeof(int64));
		index += nI;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeC, nC * sizeof(int64));
		index += nC;
		*ps++ = index;

		memcpy(&pl[index], listeK, nK * sizeof(int64));
		index += nK;
		*ps++ = index;

		memcpy(&pl[index], listeA, nA * sizeof(int64));
		index += nA;
		*ps++ = index;

		memcpy(&pl[index], listeE, nE * sizeof(int64));
		index += nE;
		*ps++ = index;

		memcpy(&pl[index], listeI, nI * sizeof(int64));
		index += nI;
		*ps++ = index;

		memcpy(&pl[index], listeM, nM * sizeof(int64));
		index += nM;
		*ps++ = index;

		if (index > maxIndex) maxIndex = index;
	    }
	}
    }

    int zaehleShortBits(int zahl) {
	zahl = zahl - ((zahl >> 1) & 0x5555);
	zahl = (zahl & 0x3333) + ((zahl >> 2) & 0x3333);
	zahl = (zahl & 0x0f0f) + ((zahl >> 4) & 0x0f0f);
	return (zahl & 0x00ff) + ((zahl >> 8) & 0x00ff);
    }

    void initStein(Stein *stein, int64 basisform, bool symmetrieFilter) {
	memset(stein, 0, sizeof(stein));
	stein->basisform = basisform;
	if (symmetrieFilter) {
	    berechneSymmetrieVarianten(stein);
	}
	else {
            berechneAlleVarianten(stein);
	}
	sortiereVarianten(stein);
    }

    /**
     * Kopiert den Stein von QUELLE nach ZIEL, lscht dabei aber 
     * alle Varianten, die mit BELEGT kollidieren.
     * Das im Betrieb irrelevante Feld 'variantenBis24' bleibt leer.
     */
    void steinFiltern(Stein *ziel, Stein *quelle, int64 belegt)
    {
	ziel->basisform = quelle->basisform;
	ziel->variantenBis24[0] = 0;

	int nAlle = 0;
	for (int64 *pa=quelle->alleVarianten; *pa!=0; pa++) {
	    int64 variante = *pa;
	    if ((variante & belegt) == 0) {
		ziel->alleVarianten[nAlle++] = variante;
	    }
	}
	ziel->alleVarianten[nAlle] = 0;

	for (int e=0; e<SUCH_EBENEN; e++) {
	    int64 relBelegt = belegt >> (12 * e);
	    for (int pos=0; pos<12; pos++) {
		int iq=0;
		int iz=0;
		for (int is=0; is<=18; is++) {
		    int schl=quelle->schluessel[e][pos][is];
		    while (iq < schl) {
			int64 variante = quelle->relVarianten[e][pos][iq++];
			if ((variante & relBelegt) == 0) {
			    ziel->relVarianten[e][pos][iz++] = variante;
			}
		    }
		    ziel->schluessel[e][pos][is] = iz;
		}
	    }
	}
    }

    void initSteine(void) {
	initStein(&basisSteine[0],   02323, true);
	initStein(&basisSteine[1],  010033, false);
	initStein(&basisSteine[2],    0472, false);
	initStein(&basisSteine[3],    0323, false);
	initStein(&basisSteine[4],    0326, false);
	initStein(&basisSteine[5],   01113, false);
	initStein(&basisSteine[6],   02322, false);
	initStein(&basisSteine[7],    0722, false);
	initStein(&basisSteine[8],    0133, false);
	initStein(&basisSteine[9],    0136, false);
	initStein(&basisSteine[10], 020013, false);
	initStein(&basisSteine[11],   0272, false);
    }

    void initReihenfolge(void) {
	for (int i=0; i<60; i++) {
	    reihenfolge[i] = ((int64) 1) << BIT_REIHE_INV_RINGE[i];
	}

	for (int v=0; v<0x1000; v++) {
            int verfuegbar = 0xfff - v;
            // Steine 1 bis 9 haben +/- 1:
            int rw = zaehleShortBits(verfuegbar & 0x3fe);
            if ((verfuegbar & 0x800) != 0) rw += 3;
            if ((verfuegbar & 0x001) != 0) rw += 2;
            SW_REICHWEITE[v] = rw;
        }	

        int zaehler2[12];
	memset(zaehler2, 0, sizeof(zaehler2));
        for (int pat=0; pat<0xfff; pat++) {
            int frei = 0xfff - pat;
	    int hotspot = -1;
            int nHotspot = 999999;
            for (int p=0, maske=1; p<12; p++, maske<<=1) {
                if ((frei & maske) != 0) {
		    int n = 256 * zaehleShortBits(frei & NACHBARN[p])
			     // - 16 * zaehleShortBits(frei & DIAG_NACHBARN[p])
			     // - getStartPos((int64) p)
			     ;
                    if (n < nHotspot) {
                        nHotspot = n;
                        hotspot = p;
                    }
                }
            }
            hotspotSchluessel[pat][0] = hotspot;

	    // Konfiguration der maximal 4 Nachbarn um den Hotspot
	    // 16 Stck, benannt A (0000) bis P (1111).
	    // Einige entfallen, weil ein Hotspot nie im Leeren steht.
	    int32 konfig = ((~pat) & NACHBARN[hotspot]) << (11 - hotspot);

	    // Welche Steine-Varianten passen?
	    // Variantengruppen heien je nach Belegung der Nachbarn
	    // auch A bis P.
	    // Varianten A passen immer,
	    // Varianten B bei Konfiguration B, D, F, J
	    // usw.
	    // Stein trgt Variantenliste laut String DBACGEAFBJIACKAEIM.
	    // Welcher Ausschnitt aus DBACGEAFBJIACKAEIM ?
	    switch (konfig) {
	    case 0x0000:    // Konfig A braucht A
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 3;
		break;
	    case 0x0100:    // B braucht AB
		hotspotSchluessel[pat][1] = 1;
		hotspotSchluessel[pat][2] = 3;
		break;
	    case 0x0400:    // C braucht AC
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 4;
		break;
	    case 0x0500:    // D braucht ABCD
		hotspotSchluessel[pat][1] = 0;
		hotspotSchluessel[pat][2] = 4;
		break;
	    case 0x1000:    // E braucht AE
		hotspotSchluessel[pat][1] = 5;
		hotspotSchluessel[pat][2] = 7;
		break;
	    case 0x1100:    // F braucht ABEF
		hotspotSchluessel[pat][1] = 5;
		hotspotSchluessel[pat][2] = 9;
		break;
	    case 0x1400:    // G braucht ACEG
		hotspotSchluessel[pat][1] = 2;
		hotspotSchluessel[pat][2] = 5;
		break;
	    case 0x4000:    // I braucht AI
		hotspotSchluessel[pat][1] = 10;
		hotspotSchluessel[pat][2] = 12;
		break;
	    case 0x4100:    // J braucht ABIJ
		hotspotSchluessel[pat][1] = 8;
		hotspotSchluessel[pat][2] = 12;
		break;
	    case 0x4400:    // K braucht ACIK
		hotspotSchluessel[pat][1] = 10;
		hotspotSchluessel[pat][2] = 14;
		break;
	    case 0x5000:    // M braucht AEIM
		hotspotSchluessel[pat][1] = 14;
		hotspotSchluessel[pat][2] = 18;
		break;

	    default:
		printf("Logikfehler: Bild %03x, Pos %d ergibt %04x\n",
		       pat, hotspot, konfig);
		exit(1);
	    }
        }
	hotspotSchluessel[0xfff][0] = -1;
	hotspotSchluessel[0xfff][1] = 0;
	hotspotSchluessel[0xfff][2] = 0;
    }

    void initEndspiele(void) {
	memset(endspiele, 0, sizeof(endspiele));
	memset(extender, 0, sizeof(extender));
    }

    /**
     * Notiert das Komplement einer vorwrts gefundenen 
     * Steine-Konfiguration als Endspiel-Lsung.
     * @param verbrauch Verbrauch aus Vorwrts-Suche
     * @param belegung Belegung aus Vorwrts-Suche
     */
    void merkeEndspiel(int verbrauch, int belegung) {
	belegung = 0xffffff - belegung;
	verbrauch = 0xfff - verbrauch;
	int i;
	Endspiel *pe = &endspiele[belegung];
	// Kennzeichen, dass Extender benutzt wird
	if (pe->verbrauch[0] > 0xfff) {
	    EndspielExtender *ex = pe->extension[1];
	    // Bis zum letzten Extender in der Kette springen
	    while (ex->next != NULL) {
		ex = ex->next;
	    }
	    // Suchen nach freiem Platz
	    for (i=0; i<30; i++) {
		if (ex->verbrauch[i] == 0) {
		    ex->verbrauch[i] = verbrauch;
		    return;
		}
	    }
	    // Neuen Extender anhngen
	    ex->next = extender + nExtender++;
	    ex = ex->next;
	    ex->verbrauch[0] = verbrauch;
	}
	else {
	    // Ist im Original-Eintrag noch ein freier Platz?
	    for (i=0; i<4; i++) {
		if (pe->verbrauch[i] == 0) {
		    pe->verbrauch[i] = verbrauch;
		    return;
		}
	    }
	    // Extender anlegen
	    EndspielExtender *exNeu = extender + nExtender++;
	    for (i=0; i<4; i++) {
		exNeu->verbrauch[i] = pe->verbrauch[i];
	    }
	    exNeu->verbrauch[4] = verbrauch;
	    pe->verbrauch[0] = 0x4000;
	    pe->extension[1] = exNeu;
	}
    }

    /**
     * Rekursives Testen auf Lsbarkeit (fr Initialisierungen).
     * Es wird vorausgesetzt, dass die Startposition unbelegt ist.
     * @param startpos zu belegende Position
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchenEndspielInvers(int steinNum,
                              int16 verbrauchsMaske,
                              int32 belegung) {
        if (steinNum >= 12) {
            return;
        }

#ifdef ZAEHLEN
        nSuchvorgaenge++;
#endif

        int nextNum = steinNum + 1;
        int nextVm = verbrauchsMaske + (1<<steinNum);
        int32 *varp = basisSteine[steinNum].variantenBis24;
	int32 steinForm;
        while ((steinForm = *varp++) != 0) {
            if ((steinForm & belegung) == 0) {
                int32 belNeu = belegung | steinForm;
                merkeEndspiel(nextVm, belNeu);
                suchenEndspielInvers(nextNum, nextVm, belNeu);
            }
        }
        if (belegung < 0xffffff) {
            suchenEndspielInvers(nextNum, verbrauchsMaske, belegung);
        }
    }

    /**
     * Vor-Initialisieren der Endspielvarianten ab Position 36.
     * Hier durch Vorwrts-Suchen bis Position 23.
     */
    void initMoeglichkeitenMitInverserSuche(void) {
	initEndspiele();
        merkeEndspiel(0, 0);
        suchenEndspielInvers(1, 0, 0);

	printf("Endspiele vorberechnet.\n");
    }



    inline void suchenET(int verbrauchsMaske, int64 belegung) {

#ifdef ZAEHLEN
	nEndspiele++;
#endif

	int nutzVm = verbrauchsMaske & 0xfff;
	int threadNum = verbrauchsMaske >> 12;
	int nl = 0;
    	int32 index = (int32)belegung & 0xffffff;
        Endspiel *esp = &endspiele[index];
        int16 *vp = esp->verbrauch;
        if (*vp != 0) {
	    if (*vp > 0xfff) {
		EndspielExtender *pee = esp->extension[1];
                do {
		    for (int exi=0; exi<30; exi++) {
			if (pee->verbrauch[exi] == nutzVm) {
			    nl++;
			}
		    }
		} while ((pee = pee->next) != NULL);
            }
            else {
		if (vp[0] == nutzVm) nl++;
                if (vp[1] == nutzVm) nl++;
                if (vp[2] == nutzVm) nl++;
                if (vp[3] == nutzVm) nl++;
            }
        }
	threadLoesungen[threadNum] += nl;
    }	


    /**
     * Rekursives Suchen in Ebene 2
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchen2(int verbrauchsMaske, int64 belegung, Stein *steine) {
	if (!isGesund(belegung)) {
	    return;
	}
	if (!reiheTesten(verbrauchsMaske, belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluessel[schnitt][0];
	if (relpos < 0) {
	    suchenET(verbrauchsMaske, belegung >> 12);
	    return;
	}

	int sStart = hotspotSchluessel[schnitt][1];
	int sEnde = hotspotSchluessel[schnitt][2];
	int i;
	int sMaske;

        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=1, sMaske=2; i<11; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[2][relpos];
		int start = steine[i].schluessel[2][relpos][sStart];
		int ende = steine[i].schluessel[2][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) 
		{
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {
#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen2(vmNeu, belNeu, steine);
                    }
                }
            }
        }
    }

    /**
     * Rekursives Suchen in Ebene 1.
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchen1(int verbrauchsMaske, int64 belegung, Stein *steine) {
	if (!isGesund2(belegung)) {
	    return;
	}
	if (!reiheTesten(verbrauchsMaske, belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluessel[schnitt][0];
	if (relpos < 0) {
	    suchen2(verbrauchsMaske, 
		    (belegung >> 12) | (int64) 0xfff000000000000,
		    steine);
	    return;
	}

	int sStart = hotspotSchluessel[schnitt][1];
	int sEnde = hotspotSchluessel[schnitt][2];
	int i;
	int sMaske;

        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=1, sMaske=2; i<11; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[1][relpos];
		int start = steine[i].schluessel[1][relpos][sStart];
		int ende = steine[i].schluessel[1][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) 
		{
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {
#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen1(vmNeu, belNeu, steine);
                    }
                }
            }
        }
    }

    /**
     * Rekursives Suchen in Ebene 0.
     * @param verbrauchsMaske Bitmaske der bislang verbrauchten Steine
     * @param belegung Bitmaske der bisher belegten Positionen
     */
    void suchen0(int verbrauchsMaske, int64 belegung, Stein *steine) {
	if (!isGesund2(belegung)) {
	    return;
	}
	if (!reiheTesten(verbrauchsMaske, belegung)) {
	    return;
	}
	int schnitt = ((int32) belegung) & 0xfff;
	int relpos = hotspotSchluessel[schnitt][0];
	if (relpos < 0) {
	    suchen1(verbrauchsMaske, 
		    (belegung >> 12) | (int64) 0xfff000000000000,
		    steine);
	    return;
	}

	int sStart = hotspotSchluessel[schnitt][1];
	int sEnde = hotspotSchluessel[schnitt][2];
	int i;
	int sMaske;


        // ACHTUNG: Stein 0 muss immer vorab gesetzt sein!
        for (i=1, sMaske=2; i<11; i++, sMaske=sMaske<<1) {
            if ((verbrauchsMaske & sMaske) == 0) {
                int vmNeu = verbrauchsMaske | sMaske;

		int64 *varp = steine[i].relVarianten[0][relpos];
		int start = steine[i].schluessel[0][relpos][sStart];
		int ende = steine[i].schluessel[0][relpos][sEnde];

#ifdef ZAEHLEN
		nVersuche += (ende - start);
#endif

                for (int j=start; j<ende; j++) {
		    int64 variante = varp[j];
                    if ((belegung & variante) == 0) {

#ifdef ZAEHLEN
                        nSuchvorgaenge++;
#endif
                        int64 belNeu = belegung | variante;

			suchen0(vmNeu, belNeu, steine);
                    }
                }
            }
        }
    }




    void suchablauf() {
	start = time(NULL);
	initReihenfolge();
	initSteine();
	initMoeglichkeitenMitInverserSuche();

        // Variante: Stein 0 vorab platzieren
        int64 *ps0;
	int64 stein0;
	int i0;

	for (ps0 = basisSteine[0].alleVarianten, i0 = 1; 
	     (stein0=*ps0) != 0; 
	     ps0++, i0++) 
	{
	    int64 *ps1;
	    int64 stein1;
	    int i1;

	    for (ps1 = basisSteine[11].alleVarianten, i1 = 0; 
	         (stein1=*ps1) != 0; 
	         ps1++, i1++) 
	    {
		if ((stein1 & stein0) == 0) {
		    int64 belegung = stein0 | stein1;

		    // Nur die nicht kollidierenden Varianten
		    for (int j=1; j<11; j++) {
			steinFiltern(&steine[j], &basisSteine[j], belegung);
			// steine[j] = basisSteine[j];
		    }

		    suchen0(0x801, belegung, steine);
		}
	    }

	    time_t schritt = time(NULL);
#ifdef ZAEHLEN
	    printf ("%2d (%3d*): %4d sec bei %6d Lsg. nach %5d/%5d/%5d Mio. Schritten...\n",
		    i0, i1, schritt - start, nLoesungen, 
		    nSuchvorgaenge / 1000000,
		    (int) (nVersuche / 1000000),
		    nEndspiele / 1000000);
#else
	    printf ("%2d: %4d sec bei %6d Lsg.\n",
		    i0, schritt - start, nLoesungen);
#endif

	}

	time_t ende = time(NULL);
	printf("%d Loesungen nach %d Sekunden.\n",
		nLoesungen, ende - start);
    }


    Aufgabe *gibAufgabe(void) {
	Aufgabe *result;
	WaitForSingleObject(hAufgabeMutex, INFINITE);
	result = &aufgaben[naechsteAufgabe];
	if (result->belegung == 0) {
	    result = NULL;
	}
	else {
	    naechsteAufgabe++;
	}
	ReleaseMutex(hAufgabeMutex);
	return result;
    }


    void nimmLoesungen(int anzahl) {
	WaitForSingleObject(hAufgabeMutex, INFINITE);
	nLoesungen += anzahl;
	nGeloest++;
	if ((nGeloest % 32) == 0) {
	    printf("%4d von %4d Teilaufgaben nach %3d sec: %6d Loesungen.\n",
		    nGeloest, 
		    nAufgaben, 
		    time(NULL) - start,
		    nLoesungen);
	}
	if (nGeloest == nAufgaben) {
	    printf("Insgesamt %d Loesungen nach %d Sekunden.\n", 
		    nLoesungen, time(NULL) - start);
	    // exit(0);
	}
	ReleaseMutex(hAufgabeMutex);
    }




    void threadArbeiter(void *numPtr) {
	int nummer = *(int *)numPtr;

	Aufgabe *aufgabe;

	while ((aufgabe = gibAufgabe()) != NULL) {
	    int64 belegung = aufgabe->belegung;
	    int verbrauch = aufgabe->verbrauch + (nummer << 12);

	    for (int j=1; j<11; j++) {
		steinFiltern(&threadSteine[nummer][j], &basisSteine[j], belegung);
	    }

	    threadLoesungen[nummer] = 0;
	    suchen0(verbrauch, belegung, threadSteine[nummer]);
	    nimmLoesungen(threadLoesungen[nummer]);
	}
    }

    void threadablauf(int nThreads) {

	printf("Ablauf mit %d Threads.\n", nThreads);
	start = time(NULL);
	initReihenfolge();
	initSteine();
	initMoeglichkeitenMitInverserSuche();

        nAufgaben = 0;

        int64 *ps0;
	int64 stein0;
	for (ps0=basisSteine[0].alleVarianten; (stein0=*ps0)!=0; ps0++) {
	    int64 *ps1;
	    int64 stein1;
	    for (ps1=basisSteine[11].alleVarianten; (stein1=*ps1)!=0; ps1++) {
		if ((stein1 & stein0) == 0) {
		    aufgaben[nAufgaben].belegung = stein0 | stein1;
		    aufgaben[nAufgaben].verbrauch = 0x801;

		    nAufgaben++;
		}
	    }
	}

	for (int i=1; i<nThreads; i++) {
	    _beginthread(threadArbeiter, 0, (void *) &threads[i]);
	}
	threadArbeiter((void *) &threads[0]);

	while (nGeloest < nAufgaben) {
	    Sleep(200);
	}

	time_t ende = time(NULL);
	printf("Main-Ende nach %d Loesungen nach %d Sekunden.\n",
		nLoesungen, ende - start);
    }



int main(int argc, char* argv[])
{
    int nThreads = 1;
    if (argc > 1) {
	nThreads = atoi(argv[1]);
	if (nThreads > 4) nThreads = 4;
	if (nThreads < 1) nThreads = 1;
    }
    threadablauf(nThreads);
    return 0;
}

