// Copyright (C) 2003 by Michael Pichler.
// Partially based on code Copyright (C) 2003 by Harald Bgeholz.
// See copympi.txt for further information.
//
// created: mpichler, 20030423
// changed: mpichler, 20030424



#include "point.h"
#include "part.h"
#include "tuple.h"

#include <stdio.h>


// hard limit for n-tuples: n * MAX_PARTPOS^n, however
// in reality *much* fewer tuples satisfy the "set next unset bit" constraint
#define MAX_TUPLEPOS 60000
// no need to store per instance, all static methods operate on these static data
static Bitvector position_[MAX_TUPLEPOS];
static int count_[MAX_TUPLEPOS];
static int numpos_ = 0;  // used array length
static int totalcount_ = 0;  // statistics


/*
 * never need an instance, just compute number of solutions for this pair
 */
int Tuple::solveTuple (const Part* a, const Part* b, const Part* c, const Part* d)
{
  numpos_ = 0;  totalcount_ = 0;
  // "inner" permutation of ordered Parts
  addPositions (a, b, c, d);  addPositions (a, b, d, c);
  addPositions (a, c, b, d);  addPositions (a, c, d, b);
  addPositions (a, d, b, c);  addPositions (a, d, c, b);
  addPositions (b, a, c, d);  addPositions (b, a, d, c);
  addPositions (b, c, a, d);  addPositions (b, c, d, a);
  addPositions (b, d, a, c);  addPositions (b, d, c, a);
  addPositions (c, a, b, d);  addPositions (c, a, d, b);
  addPositions (c, b, a, d);  addPositions (c, b, d, a);
  addPositions (c, d, a, b);  addPositions (c, d, b, a);
  addPositions (d, a, b, c);  addPositions (d, a, c, b);
  addPositions (d, b, a, c);  addPositions (d, b, c, a);
  addPositions (d, c, a, b);  addPositions (d, c, b, a);

  // estimated work reduce: numpos_ calls instead of totalcount_
  //# fprintf (stderr, "\n{%d/%d=%.3f}", totalcount_, numpos_, (double)totalcount_/numpos_);

  int solvecount = 0;
  // start at level 4 with the bits already set by the first 4 parts
  // caller must have set permut[0..3] to part indices (inner permutation does not matter)
  for (int i = 0;  i < numpos_;  ++i)
    solvecount += count_[i] * Part::solve (position_[i], 4, 0, BIT_ONE);

  return solvecount;
}


/*
 * fill position_ and count_ array, update numpos
 */
void Tuple::addPositions (const Part* a, const Part* b, const Part* c, const Part* d)
{
  // all positions from part a that have smallest bit 0 set
  const int aend = a->start_[1];
  for (int apos = a->start_[0];  apos < aend;  ++apos)
  {
    const Bitvector& bva = a->bvpos_[apos];
    // the smallest unset bit in bva (bit 0 always set)
    int sba = 1;
    while (bva.test (sba))
      ++sba;
    // all positions from b that have sba set
    const int bend = b->start_[sba+1];
    for (int bpos = b->start_[sba];  bpos < bend;  ++bpos)
    {
      const Bitvector& bvb = b->bvpos_[bpos];
      if (bva.test (bvb))  // overlap - try next b
        continue;
      Bitvector bvab (bva, bvb);  // bitwise or
      // the smallest unset bit in bvab (bit sba always set)
      int sbab = sba + 1;
      while (bvab.test (sbab))
        ++sbab;
      // all positions from c that have sbab set
      const int cend = c->start_[sbab+1];
      for (int cpos = c->start_[sbab];  cpos < cend;  ++cpos)
      {
        const Bitvector& bvc = c->bvpos_[cpos];
        if (bvab.test (bvc))  // overlap - try next c
          continue;
        Bitvector bvabc (bvab, bvc);  // bitwise or
        // the smallest unset bit in bvabc (bit sbab always set)
        int sbabc = sbab + 1;
        while (bvabc.test (sbabc))
          ++sbabc;
        // all positions from d that have sbabc set
        const int dend = d->start_[sbabc+1];
        for (int dpos = d->start_[sbabc];  dpos < dend;  ++dpos)
        {
          const Bitvector& bvd = d->bvpos_[dpos];
          if (bvabc.test (bvd))  // overlap - try next d
            continue;

          // REMIND: sbabc+1 would be a good start point for solve method
          // to find the smallest unset bit (may wish to store it)
          addPosition (Bitvector (bvabc, bvd));  // bitwise or

        } // d
      } // c
    } // b
  } // a
}

/*
 * add a Bitvector or increase counter
 */
void Tuple::addPosition (Bitvector bvall)
{
  // did we have it before?
  for (int i = 0;  i < numpos_;  ++i)
  {
    if (position_[i] == bvall)  // not new: increment count
    {
      ++(count_[i]);  ++totalcount_;
      return;  // done
    }
  }
  // really new:
  if (numpos_ == MAX_TUPLEPOS)
  {
    fprintf (stderr, "internal error: Pair0 limit %d exceeded\n", numpos_);
    exit (1);
  }
  position_[numpos_] = bvall;  // copy
  count_[numpos_] = 1;  ++totalcount_;
  ++numpos_;
}
