(in-package cl-user)
(defvar *dimensions* nil)
(defvar *solutions* 0)
(defvar *sols* nil)
(defvar *global-symmetries* nil)
(defvar *all-orientations* nil)

(defun scalar-product (a b)
  (apply #'+ (mapcar #'* a b)))

(defun vec-sub (a b)
  (mapcar #'- a b))

(defun vec-add (a b)
  (mapcar #'+ a b))

(defun cross-product (a b)
  (let ((ax (first a))
	(ay (second a))
	(az (third a))
	(bx (first b))
	(by (second b))
	(bz (third b)))
    (list (- (* ay bz)(* az by))
	  (- (* az bx)(* ax bz))
	  (- (* ax by)(* ay bx)))))

(defun transpose (r)
  (let ((a (first r))	
	(b (second r))	
	(c (third r)))
    (mapcar #'list a b c)))

(defun mat-mul (a bt)
  (let* ((b (transpose bt))
	 (a1 (first a))
	 (a2 (second a))
	 (a3 (third a))
	 (b1 (first b))
	 (b2 (second b))
	 (b3 (third b)))
    (list
     (list (scalar-product a1 b1)
	   (scalar-product a1 b2)
	   (scalar-product a1 b3))
     (list (scalar-product a2 b1)
	   (scalar-product a2 b2)
	   (scalar-product a2 b3))
     (list (scalar-product a3 b1)
	   (scalar-product a3 b2)
	   (scalar-product a3 b3)))))

(defun mat-apply (a b)
  (let ((a1 (first a))
	(a2 (second a))
	(a3 (third a)))
    (list (scalar-product a1 b)
	  (scalar-product a2 b)
	  (scalar-product a3 b))))

(defun equal-form (a b)
  (dolist (x a)
    (setf b (remove x b :test #'equal)))
  (if (null b) t nil))

(defun schwerpunkt (lst)
  (let ((x 0)
	(y 0)
	(z 0)
	(n 0))
    (dolist (e lst)
      (incf x (first e))
      (incf y (second e))
      (incf z (third e))
      (incf n))
    (list (/ x n)(/ y n)(/ z n))))

(defun all-orientations ()
  (if *all-orientations*
      *all-orientations*
  (let ((result nil)
	(units '((1 0 0)
		 (0 1 0)
		 (-1 0 0) 
		 (0 -1 0)
		 (0 0 1)
		 (0 0 -1))))
    (dolist (x units)
      (dolist (y units)
	(when (zerop (scalar-product x y))
	  (let ((z (cross-product x y)))
	    (push (list x y z) result)))))
    (setq *all-orientations* result)
    result)))

(defclass o-hashtable ()
  ((ht :initform (make-hash-table :test 'equal) :accessor ht)
   (symtable :initform 
	     (let ((result (make-hash-table :test 'equal)))
	       (dolist (p (all-orientations) result)
		 (setf (gethash p result) p)))
	     :accessor symtable)))

(defmethod set-oh (key (o o-hashtable) value)
  (setf (gethash (gethash key (symtable o)) (ht o)) value))

(defmethod add-oh (key (o o-hashtable) v)
  (let ((old (gethash (gethash key (symtable o)) (ht o))))
    (setf (gethash (gethash key (symtable o)) (ht o)) (cons v old))))

(defmethod get-oh (key (o o-hashtable))
  (gethash (gethash key (symtable o)) (ht o)))

(defmethod set-sym ((o o-hashtable) o1  o2)
  (setf (gethash o2 (symtable o)) o1))
    
(defclass klotz ()
  ((id :initarg :id :accessor id)
   (form :initform nil :initarg :form :accessor form)
   (pos :initform nil :initarg :pos :accessor pos)
   (orientation :initform nil :accessor orientation)
   (orientations :initform nil :initarg :orientations :accessor orientations)
   (domain :initform (make-instance 'o-hashtable) :initarg :domain :accessor domain)
   (constraint :allocation :class)))

(defmethod proof((origin Klotz)(constrainer klotz) or d)
  (let ((f1 (form-as-assigned constrainer)))
    (if (null f1) (format t "ERROR: constrainer not assigned"))
    (dolist (v d)
      (let ((f2 (rotated-moved origin or v)))
	(dolist (e1 f1)
	  (dolist (e2 f2)
	    (when (equal e1 e2)
	      (format t "ERROR: domain leads to forbidden cube ~a~%" e1))))))))
#|
(defmethod remove-if-equal-form ((k klotz) form)
  (print-domain k)
  (format t "removing form ~a~%" form)
  (let ((oldo (orientations k))
	(newo nil))
  (do ((o (pop oldo)(pop oldo)))((null o))
    (let ((oldd (get-oh o (domain k)))
	  (newd nil))
      (format t "original domain ~A~%" oldd)
      (do ((v (pop oldd)(pop oldd)))((null v))
	(format t "or ~a vec ~a is ~a~%" o v (rotated-moved k o v))
	(unless (equal-form form (rotated-moved k o v))
	  (format t "is kept~%")
	  (push v newd)))
      (set-oh o (domain k) newd)
      (format t "remaining domain ~a~%" newd)
      (when newd
	(push o newo))))
  (setf (orientations k) newo)))

(defmethod remove-global-symmetries ((k klotz))
  (print-domain k)
  (let ((newd (make-instance 'o-hashtable))
	(newo nil)
	(oldo (copy-list (orientations k))))
    (do ((o (pop oldo)(pop oldo)))((null o))
      (push o newo)
      (let ((oldd (copy-list (get-oh o (domain k)))))
	(do ((v (pop oldd)(pop oldd)))
	    ((null v))
	  (add-oh o newd v)
	  (dolist (g *global-symmetries*)
	    (remove-if-equal-form k (global-rotated-moved k g o v))
	    (print-domain k))))
      (format t "~A~%" newo)
      (format t "~A ~%" (get-oh (car newo) newd))
      (setf (orientations k) newo)
      (setf (domain k) newd)))
  (setf (orientations k)
    (remove-if #'(lambda(x)(null (get-oh x (domain k)))) (orientations k)))
  (print-domain k))
|#

(defmethod remove-global-symmetries ((k klotz))
  
  
  (let ((allforms nil)
	(allov nil))
    (dolist (o (orientations k))
      (dolist (v (get-oh o (domain k)))
	
	(dolist (g *global-symmetries*)
	  (let ((f (global-rotated-moved k g o v)))
	    (if (find-if #'(lambda(x)(equal-form x (copy-list f))) allforms)
		(progn ())
	      (progn
		(push f allforms)
		(pushnew  (list o v) allov :test 'equal)
		))))))
    
    (let ((onew nil)
	  (dnew (make-instance 'o-hashtable)))
      (dolist (ov allov)
	(let ((o (first ov))
	      (v (second ov)))
	  (pushnew o onew :test 'equal)
	  (add-oh o dnew v)))
      (setf (orientations k) onew)
      (setf (domain k) dnew))
    ))
	 

(defmethod clone-with-reduced-domain ((origin klotz)(constrainer klotz))
  (let* ((orientations nil)
	 (domain (make-instance 'o-hashtable))
	 (v (pos constrainer))
	 (o (orientation constrainer))
	 (inv-o (transpose o)))
    ;;(format t "clone ~d constrained by ~d~%" (id origin)(id constrainer))
    (dolist (or (orientations origin))
      ;;(format t "orientation ~a ~%" or)
      (let* ((d (get-oh or (domain origin)))
	     
	     (constraints (mapcar #'(lambda(x)(vec-add v (mat-apply o x)))
	      (get-constraint constrainer origin (mat-mul inv-o or)))))
	;;(format t "constraints~a~%" constraints)
	;;(format t "domain ~a~%" d)
	(dolist (c constraints)
	  (setf d (remove c d :test #'equal)))
	(when d
	  (set-oh or domain d)
	  ;;(proof origin constrainer or d)
	  ;;(format t "-> ~a~%" d)
	  (push or orientations))))
    (if orientations
	(make-instance 'klotz
	  :id (id origin)
	  :form (form origin)
	  :orientations orientations
	  :domain domain)
      nil)))
    
(defmethod constraint ((k klotz))
  (if (slot-boundp k 'constraint)
      (slot-value k 'constraint)
    (setf (slot-value k 'constraint) 
	(make-hash-table :test 'equal))))

(defmethod set-constraint ((k1 klotz) (k2 klotz) o value)
  (let ((key (list (id k1)(id k2) o))
	(c (constraint k1)))
    (setf (gethash key c) value)))

(defmethod get-constraint ((k1 klotz) (k2 klotz) o)
  (let ((key (list (id k1) (id k2) o)))
    (gethash key (constraint k1))))

(defmethod rotated ((k klotz) o)
  (mapcar #'(lambda(x)(mat-apply o x))(form k)))

(defmethod rotated-moved ((k klotz) o v)
  (let ((f (rotated k o)))
    (mapcar #'(lambda(x)(vec-add x v)) f)))

(defmethod global-rotated-moved ((k klotz) g o v)
  (let ((f (rotated-moved k o v))
	(mid (mapcar #'(lambda(x)(/ x 2)) *dimensions*)))
    (mapcar #'(lambda(x)(vec-add (mat-apply g (vec-sub x mid)) mid))
	     f)))
    
(defmethod  domain-size ((k klotz))
  (let ((sum 0))
    (dolist (o (orientations k) sum)
      (incf sum (list-length (get-oh o (domain k)))))))

(defmethod form-as-assigned ((k klotz))
  (let ((v (pos k))
	(o (orientation k)))
    (rotated-moved k o v)))

(defmethod assign ((k klotz) o v)
  (setf (pos k) v)
  (setf (orientation k) o))

(defmethod print-domain ((k klotz))
  (format t "Domain of Klotz ~d:~%" (id k))
  (dolist (o (orientations k))
    (format t "    Orientation ~a ->" o)
    (format t " ~a~%" (get-oh o (domain k)))))

(defun sort-by-domain-size (ks)
  (sort ks #'(lambda(x y)(< (domain-size x)(domain-size y)))))

(defun print-solution (klotzlist)
  (let ((sol nil))
    (dolist (k klotzlist)
      (let ((f (form-as-assigned k)))
	(push (list  (id k)(pos k)(orientation k) f) sol)
	(format t "~d:~a-~a-~a~%" (id k)(pos k)(orientation k) f)))
    ;;(push sol *sols*)
    (format t "~%")))

(defun initial-list ()
  (let ((lst nil))
    (push 
     (make-instance 'klotz 
       :id 1
       :form '((0 0 0)(1 0 0)(0 0 1)(0 0 2)(1 0 0))) lst)
    (push 
     (make-instance 'klotz 
       :id 2
       :form '((0 0 0)(0 1 0)(0 -1 0)(1 1 0)(-1 -1 0))) lst)
    (push
     (make-instance 'klotz 
       :id 3
       :form '((0 0 0)(0 1 0)(0 2 0)(0 3 0)(-1 0 0))) lst)
    (push
     (make-instance 'klotz 
       :id 4
       :form '((0 0 0)(0 1 0)(0 2 0)(1 2 0)(0 3 0))) lst)
    (push
     (make-instance 'klotz
       :id 5
       :form '((0 0 0)(0 1 0)(0 0 1)(0 1 1)(1 0 0))) lst)
    (push
     (make-instance 'klotz
       :id 6
       :form '((0 0 0)(1 0 0)(-1 0 0)(0 -1 0)(0 -2 0))) lst)
    (push
     (make-instance 'klotz
       :id 7
       :form '((0 0 0)(0 1 0)(1 0 0)(1 1 0)(1 2 0))) lst)
    (push
     (make-instance 'klotz
       :id 8
       :form '((0 0 0)(1 0 0)(1 1 0)(0 -1 0)(-1 -1 0))) lst)
    (push
     (make-instance 'klotz
       :id 9
       :form '((0 0 0)(0 1 0)(-1 0 0)(-1 0 1))) lst)
    (push
     (make-instance 'klotz
       :id 10
       :form '((0 0 0)(0 1 0)(1 0 0)(2 0 0)(1 -1 0))) lst)
    (push
     (make-instance 'klotz
       :id 11
       :form '((0 0 0)(1 0 0)(0 1 0)(-1 0 0)(0 -1 0))) lst)
    (push
     (make-instance 'klotz
       :id 12
       :form '((0 0 0)(1 0 0)(0 1 0)(0 2 0)(0 3 0)(1 3 0))) lst)
    lst))
#|
(defun initial-list ()
  (let ((lst nil))
    (push 
     (make-instance 'klotz 
		    :id 1
		    :form '((0 0 0)(1 0 0)(0 1 0))) lst)
    (push 
     (make-instance 'klotz 
		    :id 2
		    :form '((0 0 0)(1 0 0)(0 1 0))) lst)
    (push 
     (make-instance 'klotz 
		    :id 3
		    :form '((0 0 0)(0 1 0))) lst)))

(defun initial-list ()
  (let ((lst nil))
    (push 
     (make-instance 'klotz 
		    :id 1
		    :form '((0 0 0)(1 0 0))) lst)
    (push 
     (make-instance 'klotz 
		    :id 2
		    :form '((0 0 0)(1 0 0))) lst)
    ))
|#
(defvar *Klotzlist* (initial-list))

(defun in-range (v)
  (let* ((d *dimensions*)
	 (i (first v))
	 (j (second v))
	 (k (third v))
	 (x (first d))
	 (y (second d))
	 (z (third d)))
    (if (<= 0 i x)
	(if (<= 0 j y)
	    (if (<= 0 k z)
		t)
	  nil)
      nil)))
	


(defmethod find-symmetry ((k klotz))
  (let* ((result nil)
	 (form (form k))
	 (forms (list form))
	 (sp (schwerpunkt form))
	 (centered (mapcar #'(lambda(x)(mapcar #'- x sp)) form)))
    (dolist (r (all-orientations) (setf (orientations k) result))
      (let ((rotated (mapcar #'(lambda(x)(mat-apply r x)) centered))
	    (good t))
	(dolist (f forms
		  (if good 
		      (progn
			(push rotated forms)
			(push r result))))
	  (if (equal-form f rotated)
	      (progn
		(setq good nil))))))))

(defun find-symmetries ()
  (dolist (klotz *klotzlist*)
    (find-symmetry klotz)
    (format t "independent orientations of Klotz ~d: ~a~%" (id klotz)(orientations klotz))
    ))

(defun find-constraints ()
  (dolist (k1 *klotzlist*)
    (dolist (k2 *klotzlist*) 
      ;; to do: inverse relation could be deduced at the same time 
      (unless (eq k1 k2)
	(dolist (o (all-orientations))
	  (let ((constraint-vector nil))
	    ;;(format t "Testing ~d with ~d orientation ~o~%" (id k1) (id k2) o)
	    (dolist (v1 (form k1) 
		      (set-constraint k1 k2 o constraint-vector))
	      (dolist (v2 (rotated k2 o))
		(pushnew (vec-sub v1 v2) constraint-vector :test #'equal)
		;;(format t "v1 ~d  v2 ~d vector ~a~%" v1 v2 constraint-vector)
	))))))))

(defun fill-domains ()
  (let ((x (first *dimensions*))
	(y (second *dimensions*))
	(z (third *dimensions*)))
  
  (dolist (klotz *klotzlist*)
    (dolist (o (orientations klotz))
      (let ((result nil))
	(do ((i 0 (incf i)))((> i x))
	  (do ((j 0 (incf j)))((> j y))
	    (do ((k 0 (incf k)))((> k z))
	      (let ((good t)
		    (f (rotated-moved klotz o (list i j k))))
		(dolist (c f)
		  (if (not (in-range c))
		      (setq good nil))) ;; to do break out of loop if bad
		(if good 
		    (push (list i j k) result))))))
	(set-oh o (domain klotz) result))
      (format t "Klotz ~d Orientation ~a~%domain ~a~%" (id klotz) o (get-oh o (domain klotz)))))))

(defun set-dimensions (x y z)
  (setq *dimensions* (list (1- x) (1- y) (1- z)))
  (setq *global-symmetries* nil)
  (dolist (o (all-orientations) *global-symmetries*)
    (let ((test (mapcar #'abs (mat-apply o *dimensions*))))
      (format t "~a~%" test)
      (when (equal *dimensions* test)
	(push o *global-symmetries*)))))
	  
(defun forward-checking-solver (unassigned assigned)
  
  (let ((currentvar (pop unassigned)))
    (push currentvar assigned)
    
    
    (dolist (o (orientations currentvar))
      
      
      (dolist (v (get-oh o (domain currentvar)))
	(block try-this-one
	  (let ((futurelist nil))
	    (assign currentvar o v)
	    
	    (when (null unassigned)
	      (incf *solutions*)
	      (format t "~d solutions~%" *solutions*) 
	      (print-solution assigned)
	      
	      (format t "~%~%")
	      (return-from forward-checking-solver))
	  
	    (dolist (futurevar unassigned)
	      (let ((clone (clone-with-reduced-domain futurevar currentvar)))
		(if (null clone) (return-from try-this-one))
		(push clone futurelist)))
	  (forward-checking-solver (sort-by-domain-size futurelist) assigned)))))))

(defun test ()
  (dribble "test.log")
  (set-dimensions 3 4 5)
  (setq *sols* nil)
  (setq *klotzlist* (initial-list))
  (setq *solutions* 0)
  (find-symmetries)
  (find-constraints)
  (fill-domains)
  (remove-global-symmetries (car *klotzlist*))
  (print-domain (car *klotzlist*))
  (forward-checking-solver (sort-by-domain-size *klotzlist*) nil)
  ;;(format t "~a" *sols*)
  (dribble)
  )
(eval-when (:execute)
(test))

