{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit main;

/////////////////////////////////////////////////////////////
//  Hauptprogramm
/////////////////////////////////////////////////////////////

interface

procedure run;

implementation

uses
    SysUtils,
    classes,
    windows,
    stderror,
    zTimers,
    ctpDefs,
    ctpGlobalData,
    ctpStdParts,
    ctpPuzzleParts,
    ctpPuzzle,
    ctpUtil,
    solver,
    solvethrd;


/////////////////////////////////////////////////////////////
//  Konstanten und globale Variablen
/////////////////////////////////////////////////////////////

const
    // Anzahl Lsungen, nach denen eine Zeile ausgegeben wird
    LINE_INTERVAL     = 512;
    // Anzahl Lsungen, nach denen eine ganze Lsung ausgegeben wird
    // Muss ein Vielfaches von LINE_INTERVAL sein
    SOLUTION_INTERVAL = 4 * LINE_INTERVAL;

var
    // Timer fr globale Zeitmessung
    Timer:            TZSimpleTimer;

    // Stream fr Binr-Ausgabe der Lsungen
    OutFileName:      string = '';
    OutStream:        TStream = nil;

    // Anzahl Lsungen
    SolutionCount:    integer = 0;

    // Locks fr OnSolution
    SolutionLock:     TRTLCriticalSection;
    SolutionLockFlag: integer; // Notlsung, s. TryAcquireSolutionLock

    // Max. Anzahl Threads aus Aufrufparametern
    ThreadCount:      integer = 2;

    // Lsungsroutine der TPuzzles
    PuzzleMode:       TPuzzleMode = pmAssembler;

    // Part-Klasse und Anzeigetext
    PuzzlePartClass:  TPuzzlePartClass = TPuzzlePartAdv;
    PuzzlePartClassText: string = '1';

    // Zeilenweise Bildschirmausgaben ein/aus
    PrintLines:       boolean = true;
    // "Grafische" Lsungsausgabe ein/aus
    PrintSolutions:   boolean = true;

    // Ctrl+C gedrckt
    CtrlCPressed:     LongBool = false;




/////////////////////////////////////////////////////////////
// Ein-Ausgabe-Routinen fr vor und nach dem Lsen
/////////////////////////////////////////////////////////////


function O(str: string): string;
begin
    result := copy(str, 1, maxint);
    CharToOem(PChar(result), PChar(result));
end;

function ProgName: string;
begin
    result := AnsiLowerCase(ChangeFileExt(ExtractFileName(ParamStr(0)), ''));
end;

procedure Logo;
begin
    writeln(stderr);
    writeln(stderr, ProgName, O(' (c) Andreas Zapf 2003'));
    writeln(stderr);
end;

procedure Syntax;
const ident = '   ';
begin
    writeln(stderr, O('Syntax:'));
    writeln(stderr, ident, ProgName,
          O(' [-ma|-mp]'),
{$IFDEF DEBUG}
          O(' [-p0..-p2'),
{$ENDIF}
          O(' [-s|-S]'),
          O(' [-t0..t8]'),
          O(' [-?]'),
          O(' [-f:datei]'));
    writeln(stderr);

    writeln(stderr, ident, O('-ma       Assembler-Lsungsroutine benutzen'));
    writeln(stderr, ident, O('-mp       Pascal-Lsungsroutine benutzen'));
    writeln(stderr, ident, O('          Default: -ma'));
    writeln(stderr);
{$IFDEF DEBUG}
    writeln(stderr, ident, O('-p0..p2   Spezifitt des Suchbaums, ' +
                                        'p0: am wenigsten spezifisch'));
    writeln(stderr, ident, O('          Default: -p1'));
    writeln(stderr);
{$ENDIF}
    writeln(stderr, ident, O('-s        Weniger Zwischenausgaben'));
    writeln(stderr, ident, O('-S        Keine Zwischenausgaben'));
    writeln(stderr);

    writeln(stderr, ident, O('-t0..t8   Anzahl Lsungs-Threads, ' +
                                        't0: kein Multithreading'));
    writeln(stderr, ident, O('          Default: -t2'));
    writeln(stderr);

    writeln(stderr, ident, O('-?        Diesen Text anzeigen'));
    writeln(stderr);

    writeln(stderr, ident, O('-f:datei  Ausgabedatei fr die Lsungen ' +
                                       '(binr, nur fr Testzwecke)'));
    writeln(stderr, ident, O('          ACHTUNG: Datei wird ohne Nachfrage ' +
                                        'berschrieben.'));
    writeln(stderr);

    Halt(128);
end;

procedure ParseParams;

    procedure UnknownParam(idx: integer);
    begin
       writeln(stderr, O('Unbekannte Option: '), ParamStr(idx));
       writeln(stderr);
       Syntax;
    end;

    procedure SetPartClass(const Param: string; idx: integer);
    begin
        if (Length(param) <> 2) then UnknownParam(idx);
        case param[2] of
            '0': PuzzlePartClass := TPuzzlePartStd;
            '1': PuzzlePartClass := TPuzzlePartAdv;
            '2': PuzzlePartClass := TPuzzlePartNextBits;
            else UnknownParam(idx);
        end;
        PuzzlePartClassText := param[2];
    end;

    procedure SetPrintOptions(const Param: string; idx: integer);
    begin
        if Param = 's' then begin
            PrintLines :=     true;
            PrintSolutions := false;
        end
        else if Param = 'S' then begin
            PrintLines :=     false;
            PrintSolutions := false;
        end
        else begin
            UnknownParam(idx);
        end;
    end;

    procedure SetThreadCount(const Param: string; idx: integer);
    begin
        if (Length(param) = 2) and (param[2] in ['0'..'8']) then
            ThreadCount := ord(param[2]) - ord('0')
        else UnknownParam(idx);
    end;

    procedure SetOutFileName(const Param: string; idx: integer);
    begin
        if (Length(param) > 2) and (param[2] = ':') then
            OutFileName := Copy(Param, 3, maxint)
        else UnknownParam(idx);
    end;

    procedure ParseParam(const Param: string; idx: integer);
    begin
        if Length(Param) = 0 then UnknownParam(idx);
        
        if        param    = 'ma' then PuzzleMode := pmAssembler
        else if   param    = 'mp' then PuzzleMode := pmPascal
        else if   param[1] = 'p'  then SetPartClass(Param, idx)
        else if   param    = 's'  then SetPrintOptions(Param, idx)
        else if   param    = 'S'  then SetPrintOptions(Param, idx)
        else if   param[1] = 't'  then SetThreadCount(Param, idx)
        else if   param    = '?'  then Syntax
        else if   param[1] = 'f'  then SetOutFileName(Param, idx)
        else UnknownParam(idx);
    end;

var
    i: integer;
    param: string;
begin
    for i := 1 to ParamCount do
    begin
        param := ParamStr(i);
        if param[1] in ['-', '/'] then
        begin
            delete(param, 1, 1);
            ParseParam(param, i);
        end
        else UnknownParam(i);
    end;
end;

procedure ShowSettings;

    function PuzzleModeString(pm: TPuzzleMode): string;
    begin
        case pm of
            pmAssembler: result := O('Assembler (x86-Code)');
            pmPascal:    result := O('Pascal');
            else         result := Format(O('Unbekannt (%d)'), [ord(pm)]);
        end;
    end;

    function MultiThreadString(cnt: integer): string;
    begin
        if cnt = 0 then  result := O('Keines')
        else             result := O('Bis zu ' + IntToStr(cnt) + ' Thread(s)');
    end;

    function OutFileString(const filename: string): string;
    begin
        if length(filename) = 0 then
            result := O('Keine')
        else result := ExpandFileName(filename);
    end;

begin
    writeln(Format(O('Lsungsroutine: %s'), [PuzzleModeString(PuzzleMode)]));
{$IFDEF DEBUG}
    writeln(Format(O('Suchbaum-Typ:   %s'), [PuzzlePartClassText]));
{$ENDIF}
    writeln(Format(O('Multithreading: %s'), [MultiThreadString(ThreadCount)]));
    writeln(Format(O('Ausgabedatei:   %s'), [OutFileString(OutFileName)]));
    writeln;
end;

procedure FinalizeAndShowResults(PuzzlesDone: TPuzzleQueue);
    function PuzzleString(Puzzle: TPuzzle): string;
    begin
        if Puzzle.InitialPart >= 0 then
            result := Format(O('Puzzle %3d'), [Puzzle.InitialPart])
        else result := O('Puzzle (ges.)');
    end;
var
    Puzzle:      TPuzzle;
    TimeElapsed: LongWord;
    seconds:     LongWord;
    wasStopped:  boolean;
begin
    if CtrlCPressed then
    begin
        writeln;
        writeln(O('Programm abgebrochen, Ergebnisse unvollstndig.'));
    end;

    if not assigned(PuzzlesDone) then exit;

    wasStopped := false;

    writeln;
    writeln(O('Ergebnisse:'));

    Puzzle := PuzzlesDone.pop;
    while assigned(Puzzle) do
    begin
        write(Format(O('%-14.14s: %6d Lsungen'),
            [PuzzleString(Puzzle), Puzzle.Solutions]));
        if Puzzle.WasStopped then begin
            writeln(O(' (abgebrochen)'));
            wasStopped := true;
        end
        else writeln;
        FreeAndNil(Puzzle);
        Puzzle := PuzzlesDone.Pop;
    end;
    writeln;

    write(Format(O('%-14.14s: %6d Lsungen'),
        [O('Gesamt'), SolutionCount]));
    if WasStopped then writeln(O(' (abgebrochen)'))
    else writeln;

    TimeElapsed := Timer.Elapsed;
    seconds := (TimeElapsed + 500) div 1000;
    if seconds = 0 then inc(seconds);

    writeln(Format(O('%-14.14s: %4d s (%d Lsungen/s)'),
       [O('Lsungszeit'), seconds, SolutionCount div integer(seconds)]));
    writeln;
end;



/////////////////////////////////////////////////////////////
// Lock-Funktionen fr OnSolution
/////////////////////////////////////////////////////////////

// Zugriff sperren, vor allem fr die Bildschirmausgaben
procedure AcquireSolutionLock;
begin
    if IsMultiThread then
    begin
        EnterCriticalSection(SolutionLock);
        SolutionLockFlag := 1;
    end;
end;

// Zugriff freigeben
procedure ReleaseSolutionLock;
begin
    if IsMultiThread then
    begin
        SolutionLockFlag := 0;
        LeaveCriticalSection(SolutionLock);
    end;
end;

// (Fast) nichtblockierend Zugriff versuchen
function TryAcquireSolutionLock: boolean;
begin
    // TryEnterCriticalSection gibt unter Win98 immer
    // FALSE. Deshalb hier eine Notlsung, die
    // die meisten Wait-Operationen vermeidet
    if not IsMultiThread then
        result := true
    else begin
        result := SolutionLockFlag = 0;
        // Nur hier knnte ein anderer Thread dazwischenfunken
        if result then
        begin
            EnterCriticalSection(SolutionLock);
            SolutionLockFlag := 1;
        end;
    end;
end;

/////////////////////////////////////////////////////////////
// Wenn ein TPuzzle eine Lsung gefunden hat...
/////////////////////////////////////////////////////////////
procedure OnSolution(Puzzle: TPuzzle; var Cancel: boolean);
var
    SolCount:    integer;

    procedure StoreSolution;
    var
        i: integer;
    begin
        AcquireSolutionLock;
        try
            for i := 0 to High(Puzzle.PartLocations) do
            begin
                OutStream.WriteBuffer(Puzzle.PartLocations[i],
                    sizeof(Puzzle.PartLocations[0]));
            end;
        finally
            ReleaseSolutionLock;
        end;
    end;

    procedure PrintSolution(line, solution: boolean);
    var
        TimeElapsed: LongWord;
        SecondsGes: integer;

        function ThreadString: string;
        begin
            if GetCurrentThreadID = MainThreadID
            then result := '[main]'
            else result := Format('%.8x', [GetCurrentThreadID]);
        end;

        function InitialPartString: string;
        begin
            if Puzzle.InitialPart >= 0
            then result := IntToStr(Puzzle.InitialPart)
            else result := '0';
        end;

    const LineFmt = 'Thread: %-8s  Puzzle:%2s (%d Lsg.)  Gesamt: %d Lsg. in %ds (%d/s)';
    begin
        // Zugunsten der Performance ggf. auf Bildschirmausgabe
        // verzichten
        if not TryAcquireSolutionLock then exit;
        try
            if line then begin
                // Zeiten messen
                TimeElapsed := Timer.Elapsed;
                secondsGes := (TimeElapsed + 500) div 1000;
                if secondsGes = 0 then inc(secondsGes);

                // Ausgabe
                writeln(Format(LineFmt,
                    [ThreadString, InitialPartString, Puzzle.Solutions,
                    SolCount, secondsGes, SolCount div secondsGes]));
            end;

            if solution then begin
                write(PartsToAscii(Puzzle.PartLocations));
            end;
        finally
            ReleaseSolutionLock;
        end;
    end;

begin
    InterlockedIncrement(SolutionCount);

    // SolutionCount lokal merken, damit sich die Anzahl whrend
    // der Ausgaben nicht verndert.
    SolCount := SolutionCount;

    if CtrlCPressed then begin
        Cancel := true;
        exit;
    end;

    if (PrintLines and (SolCount mod LINE_INTERVAL = 0)) then begin
        PrintSolution(
            true,
            PrintSolutions and (SolCount mod SOLUTION_INTERVAL = 0));
    end;

    if assigned(OutStream) then StoreSolution;

end;

/////////////////////////////////////////////////////////////
// Haupt-Routine
/////////////////////////////////////////////////////////////

procedure run;
var
    Puzzles, PuzzlesDone: TPuzzleQueue;
    iPart: integer;
    i: integer;
begin

    // Aufrufparameter auswerten...
    ParseParams;
    // ...und Ergebnis anzeigen
    ShowSettings;

    // Die c't-Teile erzeugen
    ctpStdParts.MakeStdParts;

    // Alle Rotationen erzeugen...
    MakeRotations;
    // ...und daraus den Suchbaum fr das Puzzle
    MakePuzzleParts(PuzzlePartClass);

    if CtrlCPressed then
    begin
        FinalizeAndShowResults(nil);
        Halt(1);
    end;

    // Ausgabedatei
    if Length(OutFileName) > 0 then
    begin
        OutStream := TFileStream.Create(
            OutFileName,fmCreate or fmShareExclusive);
    end;

    //... und los geht's
    Puzzles     := TPuzzleQueue.Create;
    PuzzlesDone := TPuzzleQueue.Create;
    try

        for iPart := 0 to PuzzleParts.Count-1 do begin
            Puzzles.Push(
                TPuzzle.Create(PuzzleMode, PuzzleParts, iPart, OnSolution));
        end;

        Timer.Start;

        if ThreadCount < 1 then
            Solver.solve(Puzzles, PuzzlesDone)
        else begin
            for i := 1 to ThreadCount do StartSolveThread(Puzzles, PuzzlesDone);
            while WaitForSolveThreads(2000) > 0 do;
        end;

        Timer.Stop;

        FinalizeAndShowResults(PuzzlesDone);

    finally
        if assigned(OutStream) then FreeAndNil(OutStream);
        FreeAndNil(PuzzlesDone);
        FreeAndNil(Puzzles);
    end;
end;

function CtrlCHandler(dwCtrlType: dword): LongBool; stdcall;
begin
    result := false;
    if dwCtrlType in [CTRL_C_EVENT] {, CTRL_BREAK_EVENT] } then
    begin
        InterlockedExchange(Integer(CtrlCPressed), Integer(true));
        result := true;
    end;
end;



initialization

    InitializeCriticalSection(SolutionLock);
    SetConsoleCtrlHandler(@CtrlCHandler, true);

finalization

    DeleteCriticalSection(SolutionLock);

end.
