{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpTransform;

/////////////////////////////////////////////////////////////
//  Transformations-Tabellen und -Funktionen fr TIdxPuzzle
//  und TIdxCube 
/////////////////////////////////////////////////////////////

interface

uses ctpDefs, ctpBase;

type

// Transformationstabelle fr TIdxCube -> TIdxCube
PTransformationCube = ^TTransformationCube;
TTransformationCube = array[TIdxCube] of TIdxCube;

// Transformationstabelle fr TIdxPuzzle -> TIdxPuzzle
PTransformationPuzzle = ^TTransformationPuzzle;
TTransformationPuzzle = array[TIdxPuzzle] of TIdxPuzzle;

// Arrays aus Transformationstabellen fr Mehrfach-Translationen
// und -Rotationen

// Pro Achse ein Array mit Mehrfach-Rotationen.
// Die Kombination von Rotationen um eine Achse ist kommutativ,
// nicht aber die Kombination von Rotationen um verschiedene Achsen
TRotationsCube =
    array[TAxis, 0..ROTATIONS_MAX-1]
    of TTransformationCube;

// Arrays fr Mehrfach- und kombinierte Translationen.
// Die Kombination von Translationen ist kommutativ
TTranslationsCube =
    array[0..SIZE_MAX-1, 0..SIZE_MAX-1, 0..SIZE_MAX-1]
    of TTransformationCube;

TTranslationsPuzzle =
    array[0..SIZE_X-1, 0..SIZE_Y-1, 0..SIZE_Z-1]
    of TTransformationPuzzle;

var
// eine Transformationstabelle fr TIdxCube -> TIdxPuzzle
// (um ein Wrfel-Teil in ein Puzzle-Teil zu wandeln);
// an ungltigen Positionen steht -1.
TransCubeToPuzzle: array[TIdxCube] of integer;

// Viele Transformationstabellen fr Rotationen und Translationen.
// Wahrscheinlich werden die meisten nicht gebraucht, viele sind
// auch redundant...

RotationsCube:      TRotationsCube;
TranslationsCube:   TTranslationsCube;
TranslationsPuzzle: TTranslationsPuzzle;

// Eine Transformationstabelle fr die Standard-Rotationen im Wrfel
// Standard-Rotationen erzeugen die 24 mglichen Orientierungen
// eines Teils im Raum
StdRotationsCube: array[TStdRotationCube] of TTransformationCube;


// Anzahl Rotationsschritte -> Index einer Rotationstabelle
// fr Wrfel-Teile
function IdxRotationsCube(i: integer): cardinal;
// Anzahl Translationsschritte -> Index einer Translationstabelle
// fr Wrfel-Teile
function IdxTranslationsCube(i: integer): cardinal;

// Anzahl Translationsschritte -> Index einer Translationstabelle
// fr Puzzle-Teile
function IdxTranslationsPuzzleX(i: integer): cardinal;
function IdxTranslationsPuzzleY(i: integer): cardinal;
function IdxTranslationsPuzzleZ(i: integer): cardinal;


// Init-Funktion
procedure Init;



implementation

uses SysUtils;

// Verkettung von 2 Transformationstabellen
// ACHTUNG: tgt muss von src1 und src2 verschieden sein!
procedure CombineTransformationsCube(
    const src1, src2: TTransformationCube;
    var tgt: TTransformationCube);
var
    idx: TIdxCube;
begin
    if (@src1 = @tgt) or (@src2 = @tgt) then
        raise Exception.Create('Cannot combine transformations in place');
    for idx := 0 to COUNT_IDX_CUBE-1 do
        tgt[idx] := src2[src1[idx]];
end;

procedure CombineTransformationsPuzzle(
    const src1, src2: TTransformationPuzzle;
    var tgt: TTransformationPuzzle);
var
    idx: TIdxPuzzle;
begin
    if (@src1 = @tgt) or (@src2 = @tgt) then
        raise Exception.Create('Cannot combine transformations in place');
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
        tgt[idx] := src2[src1[idx]];
end;


function IdxTransformArray(i: integer; size: cardinal): cardinal;
var
    tmp: integer;
begin
    tmp := i mod integer(size);
    while tmp < 0 do inc(tmp, size);
    result := cardinal(tmp);
end;

function IdxRotationsCube(i: integer): cardinal;
begin
    result := IdxTransformArray(i,ROTATIONS_MAX);
end;

function IdxTranslationsCube(i: integer): cardinal;
begin
    result := IdxTransformArray(i,SIZE_MAX);
end;

function IdxTranslationsPuzzleX(i: integer): cardinal;
begin
    result := IdxTransformArray(i,SIZE_X);
end;

function IdxTranslationsPuzzleY(i: integer): cardinal;
begin
    result := IdxTransformArray(i,SIZE_Y);
end;

function IdxTranslationsPuzzleZ(i: integer): cardinal;
begin
    result := IdxTransformArray(i,SIZE_Z);
end;

// Init-Funktionen
procedure InitCubeToPuzzle;
var
    idx: TIdxCube;
begin
    for idx := 0 to COUNT_IDX_CUBE-1 do
        TransCubeToPuzzle[idx] := IdxPuzzle(CellCube(idx));
end;

procedure InitRotationsCube;
var
    n: cardinal;
    c: TCell;
    axis: TAxis;
    idx: TIdxCube;
    Trans: PTransformationCube;
begin
    for axis := Low(TAxis) to High(TAxis) do
    begin
        // 0. Array (Identitt) initialisieren
        Trans := @(RotationsCube[axis,0]);
        for idx := 0 to COUNT_IDX_CUBE-1 do Trans[idx] := idx;
    end;
    
    // 1. Array ber Rot... initialisieren
    Trans := @(RotationsCube[xAxis,1]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        RotCubeX(c);
        Trans[idx] := IdxCube(c);
    end;

    Trans := @(RotationsCube[yAxis,1]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        RotCubeY(c);
        Trans[idx] := IdxCube(c);
    end;

    Trans := @(RotationsCube[zAxis,1]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        RotCubeZ(c);
        Trans[idx] := IdxCube(c);
    end;

    // Fr jede Achse 2. und 3. Array ber Combine... initialisieren
    for axis := Low(TAxis) to High(TAxis) do
        for n := 2 to ROTATIONS_MAX-1 do
    begin
        CombineTransformationsCube(
            RotationsCube[axis,n-1],
            RotationsCube[axis,1],
            RotationsCube[axis,n]);
    end;
end;

procedure InitTranslationsCube;
var
    m, n, x, y, z: cardinal;
    c: TCell;
    idx: TIdxCube;
    Trans: PTransformationCube;
begin
    // 0. Array (Identitt) initialisieren
    Trans := @(TranslationsCube[0,0,0]);
    for idx := 0 to COUNT_IDX_CUBE-1 do Trans[idx] := idx;

    // Fr jede Achse 1. Array ber Trans... initialisieren
    Trans := @(TranslationsCube[1,0,0]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        TransCubeX(c);
        Trans[idx] := IdxCube(c);
    end;

    Trans := @(TranslationsCube[0,1,0]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        TransCubeY(c);
        Trans[idx] := IdxCube(c);
    end;

    Trans := @(TranslationsCube[0,0,1]);
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        c := CellCube(idx);
        TransCubeZ(c);
        Trans[idx] := IdxCube(c);
    end;

    // Fr jede Achse restliche Arrays ber Combine... initialisieren
    for n := 2 to SIZE_MAX-1 do
    begin
        CombineTransformationsCube(
            TranslationsCube[n-1,0,0],
            TranslationsCube[1,0,0],
            TranslationsCube[n,0,0]);
        CombineTransformationsCube(
            TranslationsCube[0,n-1,0],
            TranslationsCube[0,1,0],
            TranslationsCube[0,n,0]);
        CombineTransformationsCube(
            TranslationsCube[0,0,n-1],
            TranslationsCube[0,0,1],
            TranslationsCube[0,0,n]);
    end;

    // Zweier-Kombinationen
    for m := 1 to SIZE_MAX-1 do for n := 1 to SIZE_MAX-1 do
    begin
        CombineTransformationsCube(
            TranslationsCube[m,0,0],
            TranslationsCube[0,n,0],
            TranslationsCube[m,n,0]);
        CombineTransformationsCube(
            TranslationsCube[0,m,0],
            TranslationsCube[0,0,n],
            TranslationsCube[0,m,n]);
        CombineTransformationsCube(
            TranslationsCube[0,0,m],
            TranslationsCube[n,0,0],
            TranslationsCube[n,0,m]);
    end;

    // Dreier-Kombinationen
    for x := 1 to SIZE_MAX-1 do
        for y := 1 to SIZE_MAX-1 do
            for z := 1 to SIZE_MAX-1 do
    begin
        CombineTransformationsCube(
            TranslationsCube[x,0,z],
            TranslationsCube[0,y,0],
            TranslationsCube[x,y,z]);
    end;
end;

procedure InitTranslationsPuzzle;
var
    x, y, z: cardinal;
    c: TCell;
    idx: TIdxPuzzle;
    Trans: PTransformationPuzzle;
begin
    // 0. Array (Identitt) initialisieren
    Trans := @(TranslationsPuzzle[0,0,0]);
    for idx := 0 to COUNT_IDX_PUZZLE-1 do Trans[idx] := idx;

    // Fr jede Achse 1. Array ber Trans... initialisieren
    Trans := @(TranslationsPuzzle[1,0,0]);
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
    begin
        c := CellPuzzle(idx);
        TransPuzzleX(c);
        Trans[idx] := IdxPuzzle(c);
    end;

    Trans := @(TranslationsPuzzle[0,1,0]);
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
    begin
        c := CellPuzzle(idx);
        TransPuzzleY(c);
        Trans[idx] := IdxPuzzle(c);
    end;

    Trans := @(TranslationsPuzzle[0,0,1]);
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
    begin
        c := CellPuzzle(idx);
        TransPuzzleZ(c);
        Trans[idx] := IdxPuzzle(c);
    end;

    // Fr jede Achse restliche Arrays ber Combine... initialisieren
    for x := 2 to SIZE_X-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[x-1,0,0],
            TranslationsPuzzle[1,0,0],
            TranslationsPuzzle[x,0,0]);
    end;
    for y := 2 to SIZE_Y-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[0,y-1,0],
            TranslationsPuzzle[0,1,0],
            TranslationsPuzzle[0,y,0]);
    end;
    for z := 2 to SIZE_Z-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[0,0,z-1],
            TranslationsPuzzle[0,0,1],
            TranslationsPuzzle[0,0,z]);
    end;

    // Zweier-Kombinationen
    for x := 1 to SIZE_X-1 do
        for y := 1 to SIZE_Y-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[x,0,0],
            TranslationsPuzzle[0,y,0],
            TranslationsPuzzle[x,y,0]);
    end;
    for y := 1 to SIZE_Y-1 do
        for z := 1 to SIZE_Z-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[0,y,0],
            TranslationsPuzzle[0,0,z],
            TranslationsPuzzle[0,y,z]);
    end;
    for z := 1 to SIZE_Z-1 do
        for x := 1 to SIZE_X-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[0,0,z],
            TranslationsPuzzle[x,0,0],
            TranslationsPuzzle[x,0,z]);
    end;

    // Dreier-Kombinationen
    for x := 1 to SIZE_X-1 do
        for y := 1 to SIZE_Y-1 do
            for z := 1 to SIZE_Z-1 do
    begin
        CombineTransformationsPuzzle(
            TranslationsPuzzle[x,y,0],
            TranslationsPuzzle[0,0,z],
            TranslationsPuzzle[x,y,z]);
    end;
end;

procedure InitStdRotationsCube;
var
    axisSeq: array[0..2] of TAxis;
    axis: TAxis;
    i, iStd: TStdRotationCube;
begin
    // Die Transformationen fr die "Master"-Orientierungen
    // kommen an den Anfang des Arrays

    // 0: Identitt, "Master-Orientierung" fr alle Puzzletypen
    StdRotationsCube[0] := RotationsCube[xAxis,0];

    // Fr den Quader kommt ein Rotationschritt um jede Achse
    // hinzu, fr den quadratischen Quader fllt die Rotation
    // um die Achse weg, die durch die quadratischen Flchen
    // geht. Da das eine beliebige Achse sein darf, muss die
    // Reihenfolge der Transformationen variabel sein.

    if PuzzleShape = squareCuboid then
    begin
        case SquareAxis of
            xAxis: begin
                axisSeq[0] := yAxis;
                axisSeq[1] := zAxis;
                axisSeq[2] := xAxis;
            end;
            yAxis: begin
                axisSeq[0] := zAxis;
                axisSeq[1] := xAxis;
                axisSeq[2] := yAxis;
            end;
            zAxis: begin
                axisSeq[0] := xAxis;
                axisSeq[1] := yAxis;
                axisSeq[2] := zAxis;
            end;
        end;
    end
    else begin
        axisSeq[0] := xAxis;
        axisSeq[1] := yAxis;
        axisSeq[2] := zAxis;
    end;

    // 1..2:  1 Rotationschritt um die "nicht quadratischen" Achsen
    //        ("Master-Orientierung" fr Quader und quadratischen
    //        Quader)
    StdRotationsCube[1] := RotationsCube[axisSeq[0],1];
    StdRotationsCube[2] := RotationsCube[axisSeq[1],1];

    // 3..5: Nur noch fr Quader: 1 Rotationschritt um die letzte Achse...
    StdRotationsCube[3] := RotationsCube[axisSeq[2],1];

    // ... sowie noch 2 Kombinationen: eine der Achsen mit jeder anderen
    CombineTransformationsCube(
        RotationsCube[xAxis,1],
        RotationsCube[yAxis,1],
        StdRotationsCube[4]);
    CombineTransformationsCube(
        RotationsCube[xAxis,1],
        RotationsCube[zAxis,1],
        StdRotationsCube[5]);

    // Ab hier nur noch Orientierungen, die zu vorhandenen symmetrisch sind
    // Wir gewinnen sie aus den vorhandenen Transformationen durch die
    // Symmetrieoperationen (2 Rotationsschritte um jede Achse)

    iStd := 5;
    for i := 0 to 5 do
        for axis := Low(TAxis) to High(TAxis) do
    begin
        inc(iStd);
        CombineTransformationsCube(
            StdRotationsCube[i],
            RotationsCube[axis,2],
            StdRotationsCube[iStd]);
    end;
end;

procedure Init;
begin
    InitCubeToPuzzle;
    InitRotationsCube;
    InitTranslationsCube;
    InitTranslationsPuzzle;
    InitStdRotationsCube;
end;

end.
