{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpPuzzleParts;

/////////////////////////////////////////////////////////////
//  Teile im Puzzle
/////////////////////////////////////////////////////////////


interface

uses SysUtils, classes, ctpDefs, ctpTransform, ctpBaseParts;

type

/////////////////////////////////////////////////////////////
//  TPuzzlePartPos: Ein Teil an einer Position im Puzzle
/////////////////////////////////////////////////////////////
PPuzzlePartPos = ^TPuzzlePartPos;
TPuzzlePartPos = object
public
    bits: int64; // muss der erste und einzige member bleiben

private
    function Transformation(const trans: TTransformationPuzzle):
        TPuzzlePartPos;

    // Zellen aus minimal und maximal belegten Koordinaten
    // (spannen den umschlieenden Quader des Teils auf)
    function GetCellMinMax(var cMin, cMax: TCell): boolean;

public
    // Gleichheit
    function Equals(const p: TPuzzlePartPos): boolean;
    function EqualsBasePart(const p: TBasePart): boolean;
    // Alles auf 0
    procedure Clear;
    function Empty: boolean;
    // Bit an Position
    function At(idx: TIdxPuzzle): boolean; overload;
    // Setzt Bit an Position
    procedure SetAt(idx: TIdxPuzzle; bit: boolean); overload;
    // Bit an Position
    function At(x, y, z: TCoord): boolean; overload;
    // Setzt Bit an Position
    procedure SetAt(x, y, z: TCoord; bit: boolean); overload;

    // Die nchsten 8 Bits ab Index, nicht vorhandene bits
    // sind 0
    function NextBits(idx: TIdxPuzzle): byte;

    // Erstes belegtes bit, -1 wenn leer
    function First: Integer;
    // Erstes freies bit, -1 wenn voll
    function FirstFree: Integer;

    // Kombinierte Translation
    function Translation(x, y, z: integer): TPuzzlePartPos;

    // Translation mit Rand-Prfung, wenn nicht mglich, ist
    // das Ergebnis leer
    function ConstrainedTranslation(x, y, z: integer): TPuzzlePartPos;

    // Umwandlung aus TBasePart, FALSE, wenn BasePart zu gro
    function FromBasePart(const p: TBasePart): boolean;

    // Als "3D-Ascii"
    function AsAscii: string;
end;

TPuzzlePartPosArray = array of TPuzzlePartPos;

PPuzzlePartPositions = ^TPuzzlePartPositions;
TPuzzlePartPositions = object
public
    count: LongWord;
    unused: LongWord; // 8-Byte-Align gewhrleisten
    Positions: TPuzzlePartPosArray;

    // Methoden zum ndern, falls spter dynamische Arrays benutzt werden...
    procedure Clear;
    procedure Add(const Pos: TPuzzlePartPos);
    procedure Reserve(NewCount: cardinal);
    procedure Ensure(MinCount: cardinal);

    function Compare(const PPP: TPuzzlePartPositions): integer;
end;

/////////////////////////////////////////////////////////////
//  TPuzzlePart: Enthlt alle mglichen Positionen eines
//  Teils im Puzzle. Kann diese mindestens nach einem
//  freien Puzzle-Index filtern
/////////////////////////////////////////////////////////////
type

TPuzzlePartClass = class of TPuzzlePart;

TPuzzlePart = class
public

    // TRUE, wenn in GetPositions NextBits bergeben werden
    // muss
    function NeedsNextBits: boolean; virtual;

    procedure Clear; virtual;
    // Alle vorhandenen Positionen
    function GetAllPositions: PPuzzlePartPositions; virtual; abstract;
    // Positionen, die ein bit am bergebenen Index haben.
    // Bei der Suche wird garantiert, dass der bergebene Index der
    // erste freie Index im Puzzle ist.
    // Zustzlich kann auch so gefiltert werden, dass Positionen,
    // die nicht in das bergebene Puzzle passen ausgeschlossen
    // werden.
    function GetPositions(idx: TIdxPuzzle;
        NextBits: longWord): PPuzzlePartPositions;
        virtual; abstract;
    // Erzeugt das Puzzle-Teil aus einer Liste von BasePart-
    // Rotationen. Der Aufrufer ist verantwortlich, dass
    // alle Rotationen demselben BasePart bestehen
    procedure Make(const Rotations: TBasePartRotations);
        virtual; abstract;

    // TRUE, wenn eine bestimmte Position enthalten ist
    // (fr Testzwecke)
    function Contains(const PartPos: TPuzzlePartPos): boolean; virtual; abstract;
end;

TPuzzlePartStd = class(TPuzzlePart)
public
    FAllPositions: TPuzzlePartPositions;
    FPositions: array[TIdxPuzzle] of TPuzzlePartPositions;
protected
    procedure FillPositionsFromAllPositions; virtual;
public
    procedure Clear; override;
    function GetPositions(idx: TIdxPuzzle;
        param: longWord): PPuzzlePartPositions;
        override;
    function GetAllPositions: PPuzzlePartPositions; override;
    procedure Make(const Rotations: TBasePartRotations); override;
    function Contains(const PartPos: TPuzzlePartPos): boolean; override;
end;

TPuzzlePartAdv = class(TPuzzlePartStd)
protected
    procedure FillPositionsFromAllPositions; override;
end;


TPuzzlePartNextBits = class(TPuzzlePartAdv)
public
    FPositionsNextBits: array[0..63,byte] of TPuzzlePartPositions;
protected
    procedure FillPositionsFromAllPositions; override;
public
    function NeedsNextBits: boolean; override;
    procedure Clear; override;
    function GetPositions(idx: TIdxPuzzle;
        param: longWord): PPuzzlePartPositions;
        override;
end;

{$IFDEF PUZZLE_PART_STATS}

var
    glGetPositionsCount:        integer = 0;
    glGetPositionsHits:         integer = 0;
    glPositionsByGetPositions:  integer = 0;

{$ENDIF}

implementation

uses math, windows, ctpBase;

{$IFDEF PUZZLE_PART_STATS}

procedure AddGetPositionsStats(P: PPuzzlePartPositions);
begin
    InterlockedIncrement(glGetPositionsCount);
    if assigned(P) then begin
        InterlockedIncrement(glGetPositionsHits);
        InterlockedExchangeAdd(@glPositionsByGetPositions, P.count);
    end;
end;

{$ENDIF}


/////////////////////////////////////////////////////////////
//  TPuzzlePartPos
/////////////////////////////////////////////////////////////

function TPuzzlePartPos.Transformation(const trans: TTransformationPuzzle):
    TPuzzlePartPos;
var
    idx: TIdxPuzzle;
begin
    result.bits := 0;
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
        result.SetAt(trans[idx], At(idx));
end;

function TPuzzlePartPos.Equals;
begin
    result := bits = p.bits;
end;

function TPuzzlePartPos.EqualsBasePart(const p: TBasePart): boolean;
var
    x, y, z: ctpdefs.TCoord;
begin
    result := false;
    for x := 0 to ctpDefs.SIZE_MAX-1 do
        for y := 0 to SIZE_MAX-1 do
            for z := 0 to SIZE_MAX-1 do
    begin
        if (x >= SIZE_X) or (y >= SIZE_Y) or (z >= SIZE_Z) then
        begin
            if p.At(x,y,z) then exit;
        end
        else if p.At(x,y,z) <> At(x,y,z) then exit;
    end;
    result := true;
end;

procedure TPuzzlePartPos.Clear;
begin
    bits := 0;
end;

function TPuzzlePartPos.Empty: boolean;
begin
    result := bits = 0;
end;

function TPuzzlePartPos.GetCellMinMax(var cMin, cMax: TCell): boolean;
var
    idx: TIdxPuzzle;
    foundOne: boolean;
    c: TCell;
begin
    cMin := Cell(0,0,0);
    cMax := cMin;
    foundOne := false;

    for idx := 0 to COUNT_IDX_PUZZLE-1 do
    begin
        if At(idx) then
        begin
            if not foundOne then cMin := CellPuzzle(idx)
            else begin
                c := CellPuzzle(idx);
                cMin.x := min(cMin.x, c.x);
                cMin.y := min(cMin.y, c.y);
                cMin.z := min(cMin.z, c.z);
            end;
            if not foundOne then cMax := CellPuzzle(idx)
            else begin
                c := CellPuzzle(idx);
                cMax.x := max(cMax.x, c.x);
                cMax.y := max(cMax.y, c.y);
                cMax.z := max(cMax.z, c.z);
            end;
            foundOne := true;
        end;
    end;

    result := foundOne;
end;


function TPuzzlePartPos.At(idx: TIdxPuzzle): boolean; assembler;
asm
    // in Pascal:  result := (bits and (int64(1) shl idx)) <> 0;

    // eax: ->self
    // dl: idx

    and edx,3Fh  // Bereich bis 64 sicherstellen
    bt  [eax],edx
    sbb eax,eax
    and eax,1
end;

procedure TPuzzlePartPos.SetAt(idx: TIdxPuzzle; bit: boolean); assembler;
asm
    { Pascal:
        if bit then
            bits := bits or (int64(1) shl idx)
        else bits := bits and not (int64(1) shl idx);
    }

    // eax: ->self
    // dl: idx
    // cl: bit

    and edx,3Fh  // Bereich bis 64 sicherstellen
    or  bit,bit
    jz @@1
    bts [eax],edx
    ret
@@1:
    btr [eax],edx
end;

const
offs_shift_partpos: array[0..63,0..1] of byte = (
    (1,0), (2,0), (3,0), (4,0), (5,0), (6,0), (7,0), (8,0), 
    (1,1), (2,1), (3,1), (4,1), (5,1), (6,1), (7,1), (8,1),
    (1,2), (2,2), (3,2), (4,2), (5,2), (6,2), (7,2), (8,2),
    (1,3), (2,3), (3,3), (4,3), (5,3), (6,3), (7,3), (8,3),
    (1,4), (2,4), (3,4), (4,4), (5,4), (6,4), (7,4), (8,4),
    (1,5), (2,5), (3,5), (4,5), (5,5), (6,5), (7,5), (8,5),
    (1,6), (2,6), (3,6), (4,6), (5,6), (6,6), (7,6), (8,6),
    (9,6), (10,6), (11,6), (12,6), (13,6), (14,6), (15,6), (16,6));

function TPuzzlePartPos.NextBits(idx: TIdxPuzzle): byte;
asm
    and   edx,7Fh
    movzx ecx,word[edx*2+offs_shift_partpos]
    mov   edx,ecx
    shr   edx,8
    mov   ax,[edx+eax]
    shr   ax,cl
end;

function TPuzzlePartPos.First: Integer;
asm
    // eax: ->self

    mov ecx,eax
    bsf eax,[ecx]
    jz @@1
@@0:
    cmp eax,COUNT_IDX_PUZZLE
    jae @@2
    ret
@@1:
    bsf eax,[ecx+4]
    jz  @@2
    add eax,32
    jmp @@0
@@2:
    mov eax,-1
end;


function TPuzzlePartPos.FirstFree: Integer; assembler;
asm
    // eax: ->self

    mov ecx,eax
    mov edx,[ecx]
    not edx
    bsf eax,edx
    jz @@1
@@0:
    cmp eax,COUNT_IDX_PUZZLE
    jae @@2
    ret
@@1:
    mov edx,[ecx+4]
    not edx
    bsf eax,edx
    jz  @@2
    add eax,32
    jmp @@0
@@2:
    mov eax,-1
end;


function TPuzzlePartPos.At(x, y, z: ctpDefs.TCoord): boolean;
begin
    result := At(IdxPuzzle(x, y, z));
end;

procedure TPuzzlePartPos.SetAt(x, y, z: ctpDefs.TCoord; bit: boolean);
begin
    SetAt(IdxPuzzle(x, y, z), bit);
end;

function TPuzzlePartPos.Translation(x, y, z: integer): TPuzzlePartPos;
var
    theX, theY, theZ: integer;
begin
    theX := IdxTranslationsPuzzleX(x);
    theY := IdxTranslationsPuzzleY(y);
    theZ := IdxTranslationsPuzzleZ(z);
    result := Transformation(TranslationsPuzzle[theX,theY,theZ]);
end;

function TPuzzlePartPos.ConstrainedTranslation(
    x, y, z: integer): TPuzzlePartPos;
var
    cMin, cMax: TCell;
begin
    result.Clear;
    if not GetCellMinMax(cMin, cMax) then exit; // war wohl leer

    if (-x > cMin.x) or (x >= SIZE_X - cMax.x) then exit;
    if (-y > cMin.y) or (y >= SIZE_Y - cMax.y) then exit;
    if (-z > cMin.z) or (z >= SIZE_Z - cMax.z) then exit;

    result := Translation(x, y, z);
end;


function TPuzzlePartPos.FromBasePart(const p: TBasePart): boolean;
var
    idx: TIdxCube;
    myIdx: integer;
    tmp: TPuzzlePartPos;
begin
    result := false;
    tmp.bits := 0;
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        if p.At(idx) then
        begin
            myIdx := TransCubeToPuzzle[idx];
            if (myIdx < 0) or (myIdx > COUNT_IDX_PUZZLE-1) then exit;
            tmp.SetAt(myIdx, true);
        end;
    end;
    bits := tmp.bits;
    result := true;
end;

function TPuzzlePartPos.AsAscii: string;
var
    fmt: string;
    line: string;
    S: TStringList;
    x, y, z: integer;
begin
    fmt := '%2s';
    S := TStringList.Create;
    try
        for z := SIZE_Z-1 downto 0 do
        begin
            for y := SIZE_Y-1 downto 0 do
            begin
                line := '';
                for x := SIZE_Y-1 downto y+1 do
                    line := line
                        + Format(fmt, ['']);
                for x := 0 to SIZE_X - 1 do
                begin
                    if At(x,y,z) then
                        line := line  + Format(fmt, ['X'])
                    else line := line  + Format(fmt, ['-']);
                end;
                S.Add(line);
            end;
        end;
        result := S.Text;
    finally
        S.Free;
    end;
end;

///////////////////////////////////////////////////////////////
// TPuzzlePartPositions
///////////////////////////////////////////////////////////////

procedure TPuzzlePartPositions.Clear;
begin
    Count := 0;
    SetLength(Positions, 0);
end;

procedure TPuzzlePartPositions.Add(const Pos: TPuzzlePartPos);
begin
    Ensure(Count+1);
    Positions[Count] := Pos;
    inc(Count);
end;

procedure TPuzzlePartPositions.Reserve(NewCount: cardinal);
begin
    SetLength(Positions, NewCount);
    if NewCount < Count then Count := NewCount;
end;

procedure TPuzzlePartPositions.Ensure(MinCount: cardinal);
begin
    if cardinal(Length(Positions)) < MinCount then Reserve(MinCount);
end;

function TPuzzlePartPositions.Compare;
var
    arr1, arr2: TPuzzlePartPosArray;
    i: integer;
    len: integer;
    diff: int64;
begin
    arr1 := Positions;
    arr2 := PPP.Positions;
    len := math.min(Count, PPP.Count);

    result := 0;
    i := 0;
    while (result = 0) and (i < len) do
    begin
        diff := arr1[i].bits - arr2[i].bits;
        if diff < 0 then result := -1
        else if diff > 0 then result := 1;
        inc(i);
    end;

    if result = 0 then result := Count - PPP.Count;
end;


///////////////////////////////////////////////////////////////
// TPuzzlePart
///////////////////////////////////////////////////////////////

function TPuzzlePart.NeedsNextBits: boolean;
begin
    result := false;
end;

procedure TPuzzlePart.Clear;
begin
    // leer
end;


///////////////////////////////////////////////////////////////
// TPuzzlePartStd
///////////////////////////////////////////////////////////////

procedure TPuzzlePartStd.Clear;
var
    i: integer;
begin
    FAllPositions.Clear;
    for i := 0 to High(FPositions) do FPositions[i].Clear;
end;

function TPuzzlePartStd.GetAllPositions: PPuzzlePartPositions;
begin
    result := @FAllPositions;
end;

function TPuzzlePartStd.GetPositions(idx: TIdxPuzzle;
        param: longWord): PPuzzlePartPositions;
begin
    result := @(FPositions[idx]);
    if result.count <= 0 then result := nil;
{$IFDEF PUZZLE_PART_STATS}
    AddGetPositionsStats(result);
{$ENDIF}
end;

procedure TPuzzlePartStd.FillPositionsFromAllPositions;
var
    idx: TIdxPuzzle;
    iPos: cardinal;
begin
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
        for iPos := 0 to FAllPositions.count-1 do
    begin
        if FAllPositions.Positions[iPos].At(idx) then
            FPositions[idx].Add(FAllPositions.Positions[iPos]);
    end;
end;

procedure TPuzzlePartStd.Make(const Rotations: TBasePartRotations);
    procedure FillAllPositions;
    var
        i: cardinal;
        dX, dY, dZ: integer;
        cMin, cMax: TCell;
        nRotations: cardinal;
        Part, Transl: TPuzzlePartPos;
    begin
        FAllPositions.Clear;
        // Schon mal raten, wie viele Translationen zusammenkommen,
        // wir nehmen Anzahl Bits / 2 pro Teil
        FAllPositions.Reserve(Rotations.count * (COUNT_IDX_PUZZLE div 2));

        // Die in das Puzzle passenden Rotationen kopieren
        for i := 0 to Rotations.count - 1 do
        begin
            if Part.FromBasePart(Rotations.rotations[i]) then
                FAllPositions.Add(Part);
        end;

        nRotations := FAllPositions.Count;

        for i := 0 to nRotations-1 do
        begin
            // Translationen ermitteln und ablegen
            // Grenzen bestimmen
            if not FAllPositions.positions[i].GetCellMinMax(cMin, cMax) then
            begin
                Clear;
                raise Exception.Create('TPuzzlePartStd: Added invalid BasePart');
            end;

            for dx := 0 to SIZE_X-1 - cMax.x do
                for dy := 0 to SIZE_Y-1 - cMax.y do
                    for dz := 0 to SIZE_Z-1 - cMax.z do
            begin
                // Identitt auslassen
                if (dx = 0) and (dy = 0) and (dz = 0) then
                    continue;

                Transl := FAllPositions.positions[i].Translation(dx, dy, dz);
                if not Transl.Empty then FAllPositions.Add(Transl);
            end;
        end;
    end;
begin
    try
        FillAllPositions;
        FillPositionsFromAllPositions;
    except
        Clear;
        raise;
    end;
end;

function TPuzzlePartStd.Contains(const PartPos: TPuzzlePartPos): boolean;
var
    i: integer;
begin
    result := true;
    for i := 0 to FAllPositions.count-1 do
        if FAllPositions.positions[i].bits = PartPos.bits then exit;
    result := false;
end;

///////////////////////////////////////////////////////////////
// TPuzzlePartAdv
///////////////////////////////////////////////////////////////

procedure TPuzzlePartAdv.FillPositionsFromAllPositions;
var
    idx: TIdxPuzzle;
    iPos: cardinal;
begin
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
        for iPos := 0 to FAllPositions.count-1 do
    begin
        // Der bergebene Index ist immer der erste freie,
        // deshalb machen nur Teile Sinn, bei denen
        // kein Bit vor idx gesetzt ist
        if FAllPositions.Positions[iPos].First = idx then
            FPositions[idx].Add(FAllPositions.Positions[iPos]);
    end;
end;


///////////////////////////////////////////////////////////////
// TPuzzlePartNextBits
///////////////////////////////////////////////////////////////

function TPuzzlePartNextBits.NeedsNextBits: boolean;
begin
    result := true;
end;

procedure TPuzzlePartNextBits.Clear;
var
    idx, nextBits: integer;
begin
    inherited Clear;
    for idx := 0 to COUNT_IDX_PUZZLE-1 do
        for nextBits := 0 to High(byte) do
    begin
        FPositionsNextBits[idx,nextBits].Clear;
    end;
end;


procedure TPuzzlePartNextBits.FillPositionsFromAllPositions;
var
    idx: TIdxPuzzle;
    iPos: integer;
    nextBitsPart, nextBitsCompare: byte;
begin
    inherited FillPositionsFromAllPositions;

    // Jede Liste in Positions nach NextBits(Position) verteilen
    for idx := 0 to COUNT_IDX_PUZZLE-1 do begin
        for iPos := 0 to FPositions[idx].count-1 do begin
            nextBitsPart := FPositions[idx].Positions[iPos].NextBits(idx);
            for nextBitsCompare := 0 to high(byte) do begin
                if nextBitsPart and nextBitsCompare = 0 then begin
                    FPositionsNextBits[idx,nextBitsCompare]
                        .Add(FPositions[idx].Positions[iPos]);
                end;
            end;
        end;
    end;
end;

function TPuzzlePartNextBits.GetPositions(
    idx: TIdxPuzzle;
        param: longWord): PPuzzlePartPositions;
begin
    result := @(FPositionsNextBits[idx,param]);
    if result.count <= 0 then result := nil;
{$IFDEF PUZZLE_PART_STATS}
    AddGetPositionsStats(result);
{$ENDIF}
end;

end.

