{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpPuzzle;

interface

uses classes, Windows, contnrs, ctpDefs, ctpPuzzleParts;

type

/////////////////////////////////////////////////////////////
//  Puzzle, ggf. mit Vorbelegung von Teilen
/////////////////////////////////////////////////////////////

TPuzzleMode = (pmPascal, pmAssembler);

TPuzzle = class;

TPuzzleCallback = procedure(Puzzle: TPuzzle; var cancel: boolean);

TPuzzle = class
private
{$HINTS OFF}
    FAlign8:        LongWord;
{$HINTS ON}
    FBits: TPuzzlePartPos; // belegte Bits
    FPartCount:     integer;
    FSolutions:     integer;
    FLevel:         integer;
    FUsed:          array of boolean;
    FParts:         array of TPuzzlePart;
    FPartLocations: TPuzzlePartPosArray;
    FOnSolution:    TPuzzleCallback;
    FInitialPart:   integer;
    FRunning:       boolean;
    FStopped:       boolean;
    FMode:          TPuzzleMode;
    FCPUTime:       int64;
    FSolveThreadID: LongWord;
    FWithNextBits:  boolean;

    procedure ClearResults;

    procedure SolveAsmWithNextBits;
    procedure SolveAsm;
    procedure SolvePascal;

    procedure EnterSolve;

public
    // Das Array wird kopiert; die bergebenen Parts
    // gehren aber weiterhin dem Aufrufer
    // So brauchen die Daten nur 1x im Speicher zu liegen,
    // auch bei mehreren Threads.
    // ACHTUNG keine Synchronisation beim Zugriff auf die Parts!
    // Nach bergabe keine Schreiboperationen mehr!
    constructor Create(
        Mode: TPuzzleMode;    // Modus
        Parts: TList;         // List of TPuzzlePart
        initialPart: integer; // Teil, das als erstes eingefgt wird (<0: alle)
        OnSolution: TPuzzleCallback);

    destructor Destroy; override;
    procedure Solve;

    function WasStopped: boolean;
    property Solutions: integer read FSolutions;
    property PartLocations: TPuzzlePartPosArray read FPartLocations;
    property InitialPart: integer read FInitialPart;
end;


/////////////////////////////////////////////////////////////
//  Thread-sichere Puzzle-Queue, ist "Eigentmer" der
//  enthaltenen Puzzles (gibt sie im Destruktor frei)
/////////////////////////////////////////////////////////////

TPuzzleQueue = class
private
    FPuzzles: TObjectQueue;
    FLock: TRTLCriticalSection;
public
    constructor Create;
    destructor destroy; override;

    // Thread-sicheres Abholen eines Puzzles,
    // NIL, wenn Queue leer
    function Pop: TPuzzle;
    // Thread-sicheres Hinzufgen eines Puzzles
    procedure Push(p: TPuzzle);
end;


implementation

uses SysUtils, math, ctpUtil;

/////////////////////////////////////////////////////////////
//  TPuzzle
/////////////////////////////////////////////////////////////

constructor TPuzzle.Create(
    Mode: TPuzzleMode;
    Parts: TList;
    initialPart: integer;
    OnSolution: TPuzzleCallback);
var
    i: integer;
begin
    if initialPart >= Parts.Count then
        raise Exception.CreateFmt(
            'TPuzzle: invalid initial part (%d of %d)',
            [initialPart, Parts.Count]);
    
    FInitialPart := math.max(initialPart,-1);
    FMode        := Mode;
    FOnSolution  := OnSolution;
    FPartCount   := Parts.Count;
    
    SetLength(FParts,         FPartCount);
    SetLength(FUsed,          FPartCount);
    SetLength(FPartLocations, FPartCount);

    for i := 0 to FPartCount-1 do begin
        FParts[i] := Parts[i];
        if FParts[i].NeedsNextBits then FWithNextBits := true;
    end;

    ClearResults;
end;

destructor TPuzzle.Destroy;
begin
    // Explizites Lschen, um Speicherfehler leichter zu lokalisieren 
    Finalize(FUsed);
    Finalize(FParts);
    Finalize(FPartLocations);
    inherited destroy;
end;

function TPuzzle.WasStopped: boolean;
begin
    result := (not FRunning) and FStopped;
end;

procedure TPuzzle.ClearResults;
var
    i: integer;
begin
    FBits.Clear;
    FCPUTime := 0;
    
    FSolutions := 0;
    FLevel     := -1;

    for i := 0 to FPartCount-1 do begin
        FUsed[i] := false;
        FPartLocations[i].Bits := 0;
    end;
end;

procedure TPuzzle.Solve;
begin
    if FRunning then exit;
    ClearResults;
    FRunning := true;
    FStopped := false;
    FLevel := FPartCount-1;
    try
        FSolveThreadID := GetCurrentThreadID;
        EnterSolve;
    finally
        FSolveThreadID := 0;
        FRunning := false;
    end;
end;

procedure TPuzzle.EnterSolve;
    procedure doSolve;
    begin
        case FMode of
            pmPascal: SolvePascal;
            pmAssembler: if FWithNextBits
                         then SolveAsmWithNextBits
                         else SolveAsm;
            else raise Exception.Create('TPuzzle: invalid solve mode');
        end;
    end;
var
    iPos, NextIdx: integer;
    NextBits: LongWord;
    Positions: PPuzzlePartPositions;
begin
    if InitialPart < 0 then
    begin
        doSolve;
        exit;
    end;

    Assert(not FUsed[InitialPart]);

    // nur InitialPart abarbeiten (gibt keine Lsung, wenn das
    // Teil kein bit an der erste freien Position (0,0,0) hat)
    // Das darf ruhig in Pascal geschrieben sein...
    NextIdx := FBits.FirstFree;
    Assert(NextIdx >= 0);
    NextBits := FBits.NextBits(NextIdx);

    Positions := FParts[InitialPart].GetPositions(NextIdx, NextBits);
    if assigned(Positions) then
    begin
        for iPos := Positions.Count-1 downto 0 do
            with Positions.Positions[iPos] do
        begin
            if Bits and FBits.Bits = 0 then
            begin
                FUsed[InitialPart] := true;

                FBits.bits := FBits.bits or Bits;
                FPartLocations[InitialPart] := Positions.Positions[iPos];

                if FLevel = 0 then
                begin
                    inc(FSolutions);
                    FStopped := false;
                    if assigned(FOnSolution) then
                        FOnSolution(self, FStopped);
                end
                else begin
                    dec(FLevel);
                    doSolve;
                    inc(FLevel);
                    if FStopped then exit;
                end;

                FUsed[InitialPart] := false;
                FBits.bits := FBits.Bits and not Bits;
            end;
        end;
    end;
end;

procedure TPuzzle.SolveAsmWithNextBits; assembler;
{$DEFINE NEXT_BITS}
{$I ctpPuzzleAsm.inc}

procedure TPuzzle.SolveAsm; assembler;
{$UNDEF NEXT_BITS}
{$I ctpPuzzleAsm.inc}

procedure TPuzzle.SolvePascal;
var
    iPart, iPos, NextIdx: integer;
    NextBits: LongWord;
    Positions: PPuzzlePartPositions;
begin
    NextIdx := FBits.FirstFree;
    if NextIdx < 0 then
        raise Exception.Create('TPuzzle: Called Solve1 without free bits');
    NextBits := FBits.NextBits(NextIdx);


    for iPart := FPartCount-1 downto 0 do
    begin
        if not FUsed[iPart] then
        begin
            Positions := FParts[iPart].GetPositions(NextIdx, NextBits);
            if assigned(Positions) then
            begin
                for iPos := Positions.Count-1 downto 0 do
                    with Positions.Positions[iPos] do
                begin
                    if Bits and FBits.Bits = 0 then
                    begin
                        FUsed[iPart] := true;

                        FBits.bits := FBits.bits or Bits;
                        FPartLocations[iPart] := Positions.Positions[iPos];

                        if FLevel = 0 then
                        begin
                            inc(FSolutions);
                            if assigned(FOnSolution) then
                            begin
                                FStopped := false;
                                FOnSolution(self, FStopped);
                            end;

                        end
                        else begin
                            dec(FLevel);
                            SolvePascal;
                            inc(FLevel);
                        end;
                        if FStopped then exit;

                        FUsed[iPart] := false;
                        FBits.bits := FBits.Bits and not Bits;

                    end;
                end;
            end;
        end;
    end;
end;

/////////////////////////////////////////////////////////////
//  TPuzzleQueue
/////////////////////////////////////////////////////////////

constructor TPuzzleQueue.Create;
begin
    FPuzzles := TObjectQueue.Create;
    InitializeCriticalSection(FLock);
end;

destructor TPuzzleQueue.destroy;
begin
    EnterCriticalSection(FLock);
    try
        while FPuzzles.Count > 0 do FPuzzles.Pop.Free;
        FPuzzles.Free;
        inherited destroy;
    finally
        LeaveCriticalSection(FLock);
        DeleteCriticalSection(FLock);
    end;
end;

function TPuzzleQueue.Pop: TPuzzle;
begin
    EnterCriticalSection(FLock);
    try
        result := nil;
        if FPuzzles.Count > 0 then result := TPuzzle(FPuzzles.Pop);
    finally
        LeaveCriticalSection(FLock);
    end;
end;

procedure TPuzzleQueue.Push(p: TPuzzle);
begin
    EnterCriticalSection(FLock);
    try
        FPuzzles.Push(p);
    finally
        LeaveCriticalSection(FLock);
    end;
end;

end.
