{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpMem;

interface

/////////////////////////////////////////////////////////////////////
// Alternative Memory-Manager.
// Die Delphi-Speicherverwaltung reserviert Speicher an 4-Byte-Grenzen
// Irgendwo habe ich gelesen, dass dies moderne Prozessoren mit
// 8-Byte-Grenzen schneller sind, deshalb hier ein Memory-Manager,
// der dies garantiert.
//
// ACHTUNG: Diese Unit muss als erste in der uses-Anweisung
// in der Projekt-Datei stehen!
/////////////////////////////////////////////////////////////////////

// ffentliche Statistik-Variablen
var
  GetMemCount: Integer;
  FreeMemCount: Integer;
  ReallocMemCount: Integer;
  AlignedCount: Integer;

implementation

uses windows;

/////////////////////////////////////////////////////////////////////
// Typen und Funktionen fr AlignedMemMgr
// AlignedMemMgr benutzt den zuvor gesetzten MemoryManager
// und passt die zurckgegebenen Pointer auf 8-Byte-Grenzen an
// Erzeugt z.T. erheblichen Overhead (besonders bei ReallocMem),
// deshalb whrend der Lsungssuche mglichst wenig dynamischen
// Variablen (Strings!) erzeugen...
/////////////////////////////////////////////////////////////////////

type
    PInfo = ^TInfo;
    TInfo = packed record
        code: array[0..3] of byte;
        offset: LongInt;
    end;

var
  OldMemMgr: TMemoryManager;
  HHeap: THandle = 0;

function AlignedGetMem(Size: Integer): Pointer;
var
  offs: longint;
  origResult: Pointer;
  Info: PInfo;
begin
  InterlockedIncrement(GetMemCount);

  Size := Size + Sizeof(TInfo) + 8;
  result := OldMemMgr.GetMem(Size);
  if result = nil then exit;


  origResult := result;
  result := Pointer((longint(result) + Sizeof(TInfo) + 7) and not 7);
  offs := longint(result) - longint(origResult);
  Info := Pointer(Longint(result)-sizeof(TInfo));
  with Info^ do
  begin
    Code[0] := 41;
    Code[1] := 47;
    Code[2] := 53;
    Code[3] := 57;
    Offset := offs;
  end;

  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);
end;

function AlignedFreeMem(P: Pointer): Integer;
var
  Info: PInfo;
begin
  InterlockedIncrement(FreeMemCount);
  Info := Pointer(longint(P)-sizeof(TInfo));
  with Info^ do
  begin
    if (Code[0] = 41) and (Code[1] = 47) and (Code[2] = 53) and (Code[3] = 57) then
        P := Pointer(longint(P)-Offset);
  end;
  Result := OldMemMgr.FreeMem(P);
end;

function AlignedReallocMem(P: Pointer; Size: Integer): Pointer;
var
  oldOffs, offs: longint;
  origResult, oldResult: Pointer;
  Info: PInfo;
  realSize: Integer;
begin
  InterlockedIncrement(ReallocMemCount);

  Info := Pointer(longint(P)-sizeof(TInfo));
  with Info^ do
  begin
    if (Code[0] = 41) and (Code[1] = 47) and (Code[2] = 53) and (Code[3] = 57) then
    begin
        P := Pointer(longint(P)-Offset);
        oldOffs := Offset;
    end
    else begin
        Result := OldMemMgr.ReallocMem(P, Size);
        exit;
    end;
  end;

  realSize := Size + Sizeof(TInfo) + 8;

  Result := OldMemMgr.ReallocMem(P, realSize);

  origResult := result;
  oldResult := PChar(result) + oldOffs;
  result := Pointer((longint(result) + Sizeof(TInfo) + 7) and not 7);
  Move(oldResult^, result^, Size);
  offs := longint(result) - longint(origResult);

  Info := Pointer(Longint(result)-sizeof(TInfo));
  with Info^ do
  begin
    Code[0] := 41;
    Code[1] := 47;
    Code[2] := 53;
    Code[3] := 57;
    Offset := offs;
  end;

  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);
end;

/////////////////////////////////////////////////////////////////////
// Funktionen fr HeapMemMgr
// HeapMemMgr benutzt die Windows-API-HeapXXX Funktionen
// (liefern unter Windows 98SE leider auch nicht immer 8-Byte-Adressen)
/////////////////////////////////////////////////////////////////////

function HeapGetMem(Size: Integer): Pointer;
begin
  InterlockedIncrement(GetMemCount);

  result := HeapAlloc(HHeap, 0{HEAP_ZERO_MEMORY}, Size);
  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);

  // Da HEAP_ZERO_MEMORY unbekannt ist, zu Fu:
  FillChar(result^, Size, 0);
end;

function HeapFreeMem(P: Pointer): Integer;
begin
  result := 0;
  InterlockedIncrement(FreeMemCount);
  if not HeapFree(HHeap, 0, P) then result := 1;
end;

function HeapReallocMem(P: Pointer; Size: Integer): Pointer;
begin
  InterlockedIncrement(ReallocMemCount);

  result := HeapRealloc(HHeap, 0, P, Size);
  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);
end;

/////////////////////////////////////////////////////////////////////
// Funktionen fr GlobalMemMgr
// HeapMemMgr benutzt die Windows-API-GlobalXXX Funktionen
// (liefern unter Windows 98SE entgegen der Dokumentation
// nicht immer 8-Byte-Adressen)
/////////////////////////////////////////////////////////////////////

function GlobalGetMem(Size: Integer): Pointer;
begin
  InterlockedIncrement(GetMemCount);

  result := GlobalAllocPtr(GHND, Size);
  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);
end;

function GlobalFreeMem(P: Pointer): Integer;
begin
  result := 0;
  InterlockedIncrement(FreeMemCount);
  if GlobalFreePtr(P) <> 0 then result := 1;
end;

function GlobalReallocMem(P: Pointer; Size: Integer): Pointer;
begin
  InterlockedIncrement(ReallocMemCount);

  result := GlobalReallocPtr(P, Size, GMEM_ZEROINIT);
  if longint(result) mod 8 = 0 then InterlockedIncrement(AlignedCount);
end;

const
  AlignedMemMgr: TMemoryManager = (
  GetMem: AlignedGetMem;
  FreeMem: AlignedFreeMem;
  ReallocMem: AlignedReallocMem);

  HeapMemMgr: TMemoryManager = (
  GetMem: HeapGetMem;
  FreeMem: HeapFreeMem;
  ReallocMem: HeapReallocMem);

  GlobalMemMgr: TMemoryManager = (
  GetMem: GlobalGetMem;
  FreeMem: GlobalFreeMem;
  ReallocMem: GlobalReallocMem);


/////////////////////////////////////////////////////////////////////
// Setzen die jeweilgen MemMgr
/////////////////////////////////////////////////////////////////////

procedure SetCtpMemMgr;
begin
  GetMemoryManager(OldMemMgr);
  SetMemoryManager(AlignedMemMgr);
end;

procedure SetHeapMemMgr;
begin
  HHeap := HeapCreate(0, 32768, 0);
  if HHeap = 0 then exit;

  SetMemoryManager(HeapMemMgr);
end;

procedure SetGlobalMemMgr;
begin
  SetMemoryManager(GlobalMemMgr);
end;

initialization
    if not IsMemoryManagerSet then
        SetCtpMemMgr;
end.
