{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpBaseParts;

interface

uses ctpDefs, ctpBase, ctpTransform;

type
///////////////////////////////////////////////////////////////
// TBasePart: Ein Teil im umschlieenden Wrfel
///////////////////////////////////////////////////////////////

TBasePart = object
private
    Bits: array[0..COUNT_IDX_CUBE-1] of boolean;
    function Transformation(const trans: TTransformationCube):
        TBasePart;

    // Zelle aus den minimalen belegten Koordinaten jeder Achse
    function CellMin: TCell;
public
    // Setzt alle bits auf false
    procedure Clear;

    function Empty: boolean;

    // Verschiebt Bits, so dass Teil mglichst nahe an den Ursprung
    // rckt
    procedure Normalize;

    // Gleichheit
    function Equals(const P: TBasePart): boolean;

    // Symmetrien (nur Teile mit 0 Symmetrien knnen "Master" werden
    function Symmetries: cardinal;

    // Bit an Position
    function At(idx: TIdxCube): boolean; overload;
    // Setzt Bit an Position
    procedure SetAt(idx: TIdxCube; bit: boolean); overload;
    // Bit an Position
    function At(x, y, z: TCoord): boolean; overload;
    // Setzt Bit an Position
    procedure SetAt(x, y, z: TCoord; bit: boolean); overload;

    // Rotiert um n*90 Grad um die X-Achse
    // (kippt n* ber die Vorderkante, wenn n > 0)
    function RotationX(n: integer): TBasePart;
    // Rotiert um n*90 Grad um die Y-Achse
    // (kippt n* ber die rechte Kante, wenn n > 0)
    function RotationY(n: integer): TBasePart;
    // Rotiert um 90 Grad um die Z-Achse
    // (im Gegenuhrzeigersinn, wenn n > 0)
    function RotationZ(n: integer): TBasePart;

    // Allgemeine Rotation
    function Rotation(axis: TAxis; n: integer): TBasePart;

    // Standard-Rotation (eine der 24) mit Normalisierung
    function StdRotation(n: TStdRotationCube): TBasePart;

    // Verschiebt count Felder entlang der X-Achse,
    // (rollierend)
    function TranslationX(n: integer): TBasePart;
    // Verschiebt count Felder entlang der Y-Achse
    // (rollierend)
    function TranslationY(n: integer): TBasePart;
    // Verschiebt count Feld entlang der Z-Achse
    // (rollierend)
    function TranslationZ(n: integer): TBasePart;

    // Kombinierte Translation
    function Translation(x, y, z: integer): TBasePart;

    // Als "3D-Ascii"
    function AsAscii: string;

end;

///////////////////////////////////////////////////////////////
// Enthlt Funktionen zur Erzeugung von TBaseParts
///////////////////////////////////////////////////////////////
TBasePartRotations = object
private
    function IndexOf(const P: TBasePart): integer;
public
    count: cardinal;
    rotations: array[0..ROT_VARIATIONS_MAX-1] of TBasePart;
    procedure Make(const P: TBasePart; MasterPart: boolean);
end;

///////////////////////////////////////////////////////////////
// Funktionen zur Erzeugung von TBaseParts
///////////////////////////////////////////////////////////////

function MakeBasePartByIdx(indexes: array of TIdxCube;
    normal: boolean=true): TBasePart;

function MakeBasePart(Cells: array of TCell;
    normal: boolean=true): TBasePart;

implementation

uses SysUtils, Classes, math;

///////////////////////////////////////////////////////////////
// Bit-Positionen in einem 4x4x4-Wrfel:
//
//       60 61 62 63
//          56 57 58 59
//             52 53 54 55
//                48 49 50 51
//       44 45 46 47
//          40 41 42 43
//             36 37 38 39
//                32 33 34 35
//       28 29 30 31
//          24 25 26 27
//             20 21 22 23
// y   z          16 17 18 19
// ^   ^ 12 13 14 15
//  \  |     8  9 10 11
//   \ |        4  5  6  7
//    \|           0  1  2  3
//     o---------------> x



function TBasePart.Empty: boolean;
var
    idx: TIdxCube;
begin
    result := false;
    for idx := 0 to COUNT_IDX_CUBE-1 do
        if At(idx) then exit;
    result := true;
end;

procedure TBasePart.Clear;
var
    idx: TIdxCube;
begin
    for idx := 0 to COUNT_IDX_CUBE-1 do
        SetAt(idx, false);
end;

procedure TBasePart.Normalize;
var
    MinCell: TCell;
begin
    MinCell := CellMin;
    self := Translation(-MinCell.x,-MinCell.y,-MinCell.z);
end;

function TBasePart.At(idx: TIdxCube): boolean;
begin
    result := Bits[idx];
end;

procedure TBasePart.SetAt(idx: TIdxCube; bit: boolean);
begin
    Bits[idx] := bit;
end;

function TBasePart.At(x, y, z: TCoord): boolean;
begin
    result := At(IdxCube(x, y, z));
end;

procedure TBasePart.SetAt(x, y, z: TCoord; bit: boolean);
begin
    SetAt(IdxCube(x, y, z), bit);
end;

function TBasePart.Transformation(const trans: TTransformationCube):
        TBasePart;
var
    idx: TIdxCube;
begin
    for idx := 0 to COUNT_IDX_CUBE-1 do
        result.SetAt(trans[idx], At(idx));
end;

function TBasePart.RotationX;
begin
    result := Transformation(RotationsCube[xaxis,IdxRotationsCube(n)]);
end;

function TBasePart.RotationY;
begin
    result := Transformation(RotationsCube[yaxis,IdxRotationsCube(n)]);
end;

function TBasePart.RotationZ;
begin
    result := Transformation(RotationsCube[zaxis,IdxRotationsCube(n)]);
end;

function TBasePart.Rotation;
begin
    result := Transformation(RotationsCube[axis, IdxRotationsCube(n)]);
end;

function TBasePart.StdRotation(n: TStdRotationCube): TBasePart;
begin
    result := Transformation(StdRotationsCube[n]);
    result.Normalize;
end;

function TBasePart.TranslationX;
begin
    result := Transformation(TranslationsCube[IdxTranslationsCube(n),0,0]);
end;

function TBasePart.TranslationY;
begin
    result := Transformation(TranslationsCube[0,IdxTranslationsCube(n),0]);
end;

function TBasePart.TranslationZ;
begin
    result := Transformation(TranslationsCube[0,0,IdxTranslationsCube(n)]);
end;

function TBasePart.Translation;
var
    theX, theY, theZ: integer;
begin
    theX := IdxTranslationsCube(x);
    theY := IdxTranslationsCube(y);
    theZ := IdxTranslationsCube(z);
    result := Transformation(
        TranslationsCube[
            theX,
            theY,
            theZ]);
end;

function TBasePart.Equals;
var
    idx: TIdxCube;
begin
    result := false;
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        if At(idx) <> P.At(idx) then exit;
    end;
    result := true;
end;

function TBasePart.CellMin: TCell;
var
    idx: TIdxCube;
    foundOne: boolean;
    c: TCell;
begin
    result := Cell(0,0,0);
    foundOne := false;
    for idx := 0 to COUNT_IDX_CUBE-1 do
    begin
        if At(idx) then
        begin
            if not foundOne then result := CellCube(idx)
            else begin
                c := CellCube(idx);
                result.x := min(result.x, c.x);
                result.y := min(result.y, c.y);
                result.z := min(result.z, c.z);
            end;
            foundOne := true;
        end;
    end;
end;

function TBasePart.Symmetries: cardinal;
var
    norm: TBasePart;
    ax: TAxis;

    function IsSymmetric(axis: TAxis): boolean;
    var
        cmp: TBasePart;
    begin
        cmp := norm.Rotation(axis, 2);
        cmp.Normalize;
        result := norm.Equals(cmp);
    end;
begin
    norm := self;
    norm.Normalize;

    result := 0;
    for ax := Low(TAxis) to High(TAxis) do
        if IsSymmetric(ax) then inc(result);
end;

function TBasePart.AsAscii: string;
var
    fmt: string;
    line: string;
    S: TStringList;
    x, y, z: integer;
begin
    fmt := '%2s';
    S := TStringList.Create;
    try
        for z := SIZE_MAX-1 downto 0 do
        begin
            for y := SIZE_MAX-1 downto 0 do
            begin
                line := '';
                for x := SIZE_MAX-1 downto y+1 do
                    line := line
                        + Format(fmt, ['']);
                for x := 0 to SIZE_MAX - 1 do
                begin
                    if At(x,y,z) then
                        line := line  + Format(fmt, ['X'])
                    else line := line  + Format(fmt, ['-']);
                end;
                S.Add(line);
            end;
        end;
        result := S.Text;
    finally
        S.Free;
    end;
end;

///////////////////////////////////////////////////////////////
// TBasePartRotations
///////////////////////////////////////////////////////////////

function TBasePartRotations.IndexOf(const P: TBasePart): integer;
begin
    for result := 0 to count-1 do
        if rotations[result].Equals(P) then exit;
    result := -1;
end;

procedure TBasePartRotations.Make(const P: TBasePart; MasterPart: boolean);
var
    newPart: TBasePart;
    rot, rotMax: TStdRotationCube;
begin
    count := 0;
    if MasterPart and (P.Symmetries > 0) then exit;

    rotMax := High(TStdRotationCube);
    if MasterPart then rotMax := ROT_VARIATIONS_MASTER-1;

    for rot := 0 to rotMax do
    begin
        newPart := P.StdRotation(rot);
        // Eventuelle Symmetrien ausschlieen
        if IndexOf(newPart) < 0 then
        begin
            rotations[count] := newPart;
            inc(count);
        end;
    end;
end;


///////////////////////////////////////////////////////////////
// MakeBasePart
///////////////////////////////////////////////////////////////
function MakeBasePartByIdx(indexes: array of TIdxCube;
    normal: boolean=true): TBasePart;
var
    i: integer;
begin
    result.Clear;
    for i := 0 to High(indexes) do result.SetAt(indexes[i], true);
    if normal then result.Normalize;
end;

function MakeBasePart(Cells: array of TCell;
    normal: boolean=true): TBasePart;
var
    indexes: array of TIdxCube;
    i: integer;
begin
    SetLength(indexes, Length(Cells));
    for i := 0 to High(Cells) do indexes[i] := IdxCube(Cells[i]);
    result := MakeBasePartByIdx(indexes, normal);
end;



initialization

end.
