{-----------------------------------------------------------------------------

    ct-puzzle: Quellcode zum Programmierwettbewerb aus c't Heft 7/2003

    Copyright (c) Andreas Zapf 2003. Alle Rechte vorbehalten.

-----------------------------------------------------------------------------}

unit ctpBase;

/////////////////////////////////////////////////////////////
//  Grundlegende Typen und Funktionen
/////////////////////////////////////////////////////////////

interface

uses ctpDefs;

/////////////////////////////////////////////////////////////
//  Funktionen
/////////////////////////////////////////////////////////////

// Form, die sich aus den Achsenmaen ergibt
function PuzzleShape: TShape;
// Wenn PuzzleShape = squareCuboid:
// Achse, die durch die beiden quadratische Flche geht.
function SquareAxis: TAxis;




//  Umwandlungen Zelle <-> Index
/////////////////////////////////////////////////////////////

// Zelle aus Koordinaten
function Cell(aX, aY, aZ: TCoord): TCell;

// Index einer Zelle im Puzzle-Quader, -1, wenn ungltig
function IdxPuzzle(aX, aY, aZ: TCoord): integer; overload;
function IdxPuzzle(const c: TCell): integer; overload;

// Index einer Zelle im Puzzle-umschlieenden Wrfel,
// -1, wenn ungltig (kann eigentlich nicht sein)
function IdxCube(aX, aY, aZ: TCoord): integer; overload;
function IdxCube(const c: TCell): integer; overload;

// Zelle eines Index ins Puzzle
function CellPuzzle(idx: TIdxPuzzle): TCell;

// Zelle eines Index in den Puzzle-umschlieenden Wrfel
function CellCube(idx: integer): TCell;

//  Zellen - Transformationen
/////////////////////////////////////////////////////////////

// Rotationen (sind nur im Wrfel erforderlich, im Puzzle wird
// nur noch verschoben)

// Rotation einer Zelle im Wrfel um 90 Grad um die X-Achse
// (Kippen ber die Vorderkante)
procedure RotCubeX(var c: TCell);

// Rotation einer Zelle im Wrfel um 90 Grad um die Y-Achse
// (Kippen ber die rechte Kante)
procedure RotCubeY(var c: TCell);

// Rotation einer Zelle im Wrfel um 90 Grad um die Z-Achse
// (Drehen im Gegenuhrzeigersinn)
procedure RotCubeZ(var c: TCell);

// Translationen sind im Wrfel und im Puzzle erforderlich.
// Alle Translationen arbeiten zirkulr (was hinten hinausgeschoben
// wird, kommt vorne wieder rein)
// Translationen im Puzzle geben FALSE zurck, wenn die bergebene
// Zelle auerhalb des Puzzles liegt

procedure TransCubeX(var c: TCell);
procedure TransCubeY(var c: TCell);
procedure TransCubeZ(var c: TCell);

function TransPuzzleX(var c: TCell): boolean;
function TransPuzzleY(var c: TCell): boolean;
function TransPuzzleZ(var c: TCell): boolean;


implementation

uses SysUtils, classes;

/////////////////////////////////////////////////////////////
// Privates
/////////////////////////////////////////////////////////////


function TranslateCoord(var Coord: TCoord; nPositions: Cardinal): boolean;
begin
    result := true;
    if      Coord < nPositions-1 then inc(Coord)
    else if Coord = nPositions-1 then Coord := 0
    else result := false;
end;


/////////////////////////////////////////////////////////////
// ffentliches
/////////////////////////////////////////////////////////////

function PuzzleShape: TShape;
begin
    if (SIZE_X = SIZE_Y) and (SIZE_Y = SIZE_Z) then
        result := cube
    else if (SIZE_X = SIZE_Y) or (SIZE_X = SIZE_Z) or (SIZE_Y = SIZE_Z) then
        result := squareCuboid
    else result := cuboid;
end;

function SquareAxis: TAxis;
begin
    if PuzzleShape <> squareCuboid then
        result := zAxis // sinnloser, aber stabiler Rckgabewert
    else begin
        if (SIZE_X = SIZE_Y) then
            result := zAxis
        else if (SIZE_Y = SIZE_Z) then
            result := xAxis
        else if (SIZE_Z = SIZE_X) then
            result := yAxis
        else raise Exception.Create('SquareCuboid must have Squares...');
    end;
end;

function Cell(aX, aY, aZ: TCoord): TCell;
begin
    with result do
    begin
        x := aX;
        y := aY;
        z := aZ;
    end;
end;

function IdxPuzzle(aX, aY, aZ: TCoord): integer;
begin
    if (aX >= SIZE_X) or (aY >= SIZE_Y) or
        (aZ >= SIZE_Z) then
        result := -1
    else result := aX + SIZE_X*aY + SIZE_X*SIZE_Y*aZ;
end;

function IdxPuzzle(const c: TCell): integer; overload;
begin
    result := IdxPuzzle(c.x, c.y, c.z);
end;

function IdxCube(aX, aY, aZ: TCoord): integer;
begin
    if (aX >= SIZE_MAX) or (aY >= SIZE_MAX) or
        (aZ >= SIZE_MAX) then
        result := -1
    else result := aX + SIZE_MAX*aY + SIZE_MAX*SIZE_MAX*aZ;
end;

function IdxCube(const c: TCell): integer; overload;
begin
    result := IdxCube(c.x, c.y, c.z);
end;

function CellPuzzle(idx: TIdxPuzzle): TCell;
begin
    result.z := idx div (SIZE_X*SIZE_Y);
    idx := idx mod (SIZE_X*SIZE_Y);
    result.y := idx div SIZE_X;
    result.x := idx mod SIZE_X;
end;

function CellCube(idx: integer): TCell;
begin
    result.z := idx div (SIZE_MAX*SIZE_MAX);
    idx := idx mod (SIZE_MAX*SIZE_MAX);
    result.y := idx div SIZE_MAX;
    result.x := idx mod SIZE_MAX;
end;

procedure RotCubeX(var c: TCell);
var
    tmp: TCoord;
begin
    with c do
    begin
        tmp := y;
        y := SIZE_MAX-1 - z;
        z := tmp;
    end;
end;

procedure RotCubeY(var c: TCell);
var
    tmp: TCoord;
begin
    with c do
    begin
        tmp := x;
        x := z;
        z := SIZE_MAX-1 - tmp;
    end;
end;

procedure RotCubeZ(var c: TCell);
var
    tmp: TCoord;
begin
    with c do
    begin
        tmp := x;
        x := SIZE_MAX-1 - y;
        y := tmp;
    end;
end;

procedure TransCubeX(var c: TCell);
begin
    TranslateCoord(c.x, SIZE_MAX);    
end;

procedure TransCubeY(var c: TCell);
begin
    TranslateCoord(c.y, SIZE_MAX);
end;

procedure TransCubeZ(var c: TCell);
begin
    TranslateCoord(c.z, SIZE_MAX);    
end;

function TransPuzzleX(var c: TCell): boolean;
begin
    result := TranslateCoord(c.x, SIZE_X);
end;

function TransPuzzleY(var c: TCell): boolean;
begin
    result := TranslateCoord(c.y, SIZE_Y);    
end;

function TransPuzzleZ(var c: TCell): boolean;
begin
    result := TranslateCoord(c.z, SIZE_Z);
end;

initialization

end.
