unit HookApps;  { 15-MAI-97 as (Arne Schpers) }
{ Allgemein verwendbare Komponente (16/32 Bit) fr
  das berwachen anderer Anwendungen. Meldet sich zurck,
  wenn die berwachte Anwendung gestartet/beendet wird.
  Braucht ShHook.DLL - dort ist der als Wchter verwendete
  Shell-Hook definiert. }
interface
uses
  WinTypes, WinProcs, Messages, SysUtils, Classes, Graphics,
  Controls, Forms, Dialogs;

type { hkWaiting = Warten auf das Erscheinen des Fensters }
  THookedAppState = (hkTerminated, hkWaiting, hkRunning);

  THookedApp = class(TComponent)
  private
    FAppName: String;              { .EXE-Name der Anwendung }
    FAppParams: String;           { Kommandozeilen-Parameter }
    FHookedAppState: THookedAppState;
    FTerminateAppOnDestroy: Boolean; { True: WM_CLOSE senden }
    FOnAppStateChange: TNotifyEvent;           { Rckmeldung }
    FAppWindowHandle: HWnd;     { Hauptfenster der Anwendung }
    FAppWindowName, FAppWindowClass: String;    { guess what }
    FHelpWndHandle: HWnd;          { Listener fr ShHook.DLL }
  protected
    procedure HelpWndProc(var Msg: TMessage);
    procedure CheckRunning;    { interner Test bei Setxxx }
    procedure SetAppName(const Value: String);
    procedure SetAppParams(const Value: String);
    procedure SetAppWindowName(const Value: String);
    procedure SetAppWindowClass(const Value: String);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    { Anwendung starten, auf Fenster warten, existierendes
      Fenster berwachen }
    function Execute(Show: Word): Word;
    procedure WaitFor(AttachIfExists: Boolean);
    procedure AttachTo(Wnd: HWnd);
    procedure Terminate;  { berwachte Anwendung beenden }
  published
    property AppName: String read FAppName write SetAppName;
    property AppParams: String read FAppParams
      write SetAppParams;
    property AppWindowName: String read FAppWindowName
      write SetAppWindowName;
    property AppWindowClass: String read FAppWindowClass
      write SetAppWindowClass;
    property AppWindowHandle: HWnd read FAppWindowHandle;
    property HookedAppState: THookedAppState read FHookedAppState;
    property TerminateAppOnDestroy: Boolean
      read FTerminateAppOnDestroy
      write FTerminateAppOnDestroy default True;
    property OnAppStateChange: TNotifyEvent
      read FOnAppStateChange
      write FOnAppStateChange;
  end;

procedure Register;

implementation

{ Zeiger auf die beiden Routinen in ShHook.DLL. Werden ber
  GetProcAddress gesetzt -> kein implizter Aufruf des Windows-
  Linkers, keine Fehlermeldung, wenn die DLL nicht existiert }
var
  HookDLLRegisterHelpWindow: function(CallBackWnd: HWnd): Boolean;
  {$IFDEF WIN32} stdcall; {$ENDIF}
  HookDLLUnregisterHelpWindow: procedure(CallbackWnd: HWnd);
  {$IFDEF WIN32} stdcall; {$ENDIF}

  WM_NotifyHook: Word; { DLL registriert denselben Wert }
  HookDLLHandle: THandle; { := LoadLibrary('ShHook.dll'); }

constructor THookedApp.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  TerminateAppOnDestroy := True;
  FHookedAppState := hkTerminated;
  if not (csDesigning in ComponentState) { nicht im Designer }
    and (HookDLLHandle <> 0) then
  begin { Hilfsfenster anlegen und bei der DLL registrieren }
    FHelpWndHandle := AllocateHWnd(HelpWndProc);
    if not HookDLLRegisterHelpWindow(FHelpWndHandle)
      then raise Exception.Create('Mehr als MAXHOOKS ShellHooks!');
  end;
end;

destructor THookedApp.Destroy;
begin
  if not (csDesigning in ComponentState) {nicht im Designer }
    and (HookDLLHandle <> 0) then
  begin
    if FHelpWndHandle <> 0 then
    begin { Hilfsfenster abmelden und abbauen }
      HookDLLUnregisterHelpWindow(FHelpWndHandle);
      DeallocateHWnd(FHelpWndHandle);
    end;
  end;
  if TerminateAppOnDestroy then Terminate;  { Anw. beenden }
  inherited Destroy;
end;

procedure THookedApp.Terminate;
begin
  if HookedAppState = hkRunning then
    PostMessage(AppWindowHandle,WM_CLOSE,0,0);
end;

procedure THookedApp.CheckRunning;
var CompName: String; { XRef: alle Setxxx-Methoden }
begin
  if (HookedAppState <> hkTerminated) then
  begin
    if Name <> '' then CompName := Name
      else CompName := 'THookedApp';
    raise Exception.Create(Format('%s: kontrolliert '+
      'laufende App %s',[CompName,FAppName]));
  end;
end;

{ Der etwas langweiligere Teil }
procedure THookedApp.SetAppName(const Value: String);
begin CheckRunning; FAppName := Value; end;

procedure THookedApp.SetAppParams(const Value: String);
begin CheckRunning; FAppParams := Value; end;

procedure THookedApp.SetAppWindowName(const Value: String);
begin CheckRunning; FAppWindowName := Value; end;

procedure THookedApp.SetAppWindowClass(const Value: String);
begin CheckRunning; FAppWindowClass := Value; end;

{ --- Start einer Anwendung ------ }
function THookedApp.Execute(Show: Word): Word;
var CharBuf: Array[0..511] of Char;
begin
  StrPCopy(CharBuf,AppName+' '+AppParams);
  Result := WinExec(CharBuf,Show);
  { Rckmeldung kommt dann beim Erscheinen des Hauptfensters }
  if Result > 31 then FHookedAppState := hkWaiting;
end;

{ ---- berwachen eines existierenden Fensters ------- }
procedure THookedApp.AttachTo(Wnd: HWnd);
var Buf: Array[0..255]of Char;
begin
  CheckRunning;
  if not IsWindow(Wnd) then raise Exception.Create
     ('THookedApp.AttachTo: Fenster-Handle ungltig');
  FAppWindowHandle := Wnd;
  FHookedAppState := hkRunning;
  { Fenstertitel und Klasse einsetzen }
  GetWindowText(Wnd,Buf,SizeOf(Buf));
  FAppWindowName := StrPas(Buf);
  GetClassName(Wnd,Buf,SizeOf(Buf));
  FAppWindowClass := StrPas(Buf);
end;

{ -- Suche nach einem Fenster und berwachung/Warten -- }
type
  TWinInfo = record { Such-Daten (lParam^) in EnumFunc }
    WindowName, WindowClass: String; { Namen in UpperCase }
    AppWindowHandle: HWnd;  { Ergebnis von EnumFunc }
  end;
  PWinInfo = ^TWinInfo;

function EnumFunc(Wnd: HWnd; lParam: LongInt): Boolean; export;
{$IFDEF WIN32} stdcall; {$ENDIF}
var Buf: Array[0..255] of Char;
begin
  Result := True;  { optimistischer Ansatz }
  with PWinInfo(lParam)^ do
  begin
    if WindowName <> '' then
    begin  { Fenster-Titeltext festgelegt? }
      GetWindowText(Wnd,Buf,SizeOf(Buf));
      Result:= Pos(WindowName,AnsiUpperCase(StrPas(Buf))) <> 0;
    end;
    if  {immer noch} Result and (WindowClass <> '') then
    begin
      GetClassName(Wnd,Buf,SizeOf(Buf));
      Result := Result and (WindowClass=AnsiUpperCase(StrPas(Buf)));
    end;
    if {immer noch} Result then AppWindowHandle := Wnd; { OK! }
  end;
  Result := not Result;  { EnumWindows: True = weitersuchen }
end;

procedure THookedApp.WaitFor(AttachIfExists: Boolean);
var WinInfo: PWinInfo;
begin
  CheckRunning;
  if (AppWindowClass = '') and (AppWindowName = '') then
    raise Exception.Create('THookedApp.WaitFor: AppWindowClass '
     + 'und/oder AppWindowName setzen');
  if AttachIfExists then
  begin { Gucken: Gibt es das Fenster schon? }
    New(WinInfo);
    with WinInfo^ do
    begin   { Properties -> Uppercase fr EnumFunc }
      WindowName := AnsiUpperCase(FAppWindowName);
      WindowClass := AnsiUpperCase(FAppWindowClass);
      AppWindowHandle := 0;
    end;
    EnumWindows(@EnumFunc,LongInt(WinInfo)); { abklappern }
    FAppWindowHandle := WinInfo^.AppWindowHandle;{ kann 0 sein }
    Dispose(WinInfo);
  end
    else FAppWindowHandle := 0; { auf neues Fenster warten }
  if FAppWindowHandle = 0 then FHookedAppState := hkWaiting
  else
  begin  { Anwendung luft. Rckmeldung ans Programm }
    FHookedAppState := hkRunning;
    if Assigned(FOnAppStateChange) then OnAppStateChange(Self);
  end;
end;

{ WndProc des (unsichtbaren) Hilfsfensters; bekommt bei
  WH_SHELL-Ereignissen von der DLL ein WM_NotifyHook mit
  Code (wParam) und Fenster-Handle (lParam) }
procedure THookedApp.HelpWndProc(var Msg: TMessage);
var CharBuf: Array[0..255] of Char;
begin
  with Msg do
    if Msg = WM_NotifyHook then
    case wParam of  { kommt die Botschaft von ShHook.dll }
      HSHELL_WINDOWCREATED:
        if HookedAppState = hkWaiting then
        begin
          if AppWindowName <> '' then { Fenster-Titeltext? }
          begin  { -> ist angegeben: vergleichen }
            GetWindowText(lParam,CharBuf,SizeOf(CharBuf));
            if Pos(AnsiUpperCase(AppWindowName),
                   AnsiUpperCase(StrPas(CharBuf))) = 0
             then lParam := 0;
          end;
          if (AppWindowClass <> '') and (lParam <> 0) then
          begin  { Fenster-"Klassenname" }
            GetClassName(lParam,CharBuf,SizeOf(CharBuf));
            if AnsiUpperCase(StrPas(CharBuf)) <>
              AnsiUpperCase(AppWindowClass) then lParam := 0;
          end;
          if lParam <> 0 then  { Prfungen berlebt? }
          begin
            FAppWindowHandle := lParam; { OK. Fenster ist da }
            FHookedAppState := hkRunning; { Anwendung luft }
            if Assigned(FOnAppStateChange)
              then OnAppStateChange(Self); { Rckmeldung }
          end;
        end;
      HSHELL_WINDOWDESTROYED:
        if (HookedAppState = hkRunning) and
          (lParam = AppWindowHandle) then  { "mein Fenster" }
        begin
          FHookedAppState := hkTerminated; { Anwendung beendet }
          if Assigned(FOnAppStateChange)
            then OnAppStateChange(Self);  { Rckmeldung }
          FAppWindowHandle := 0;
        end;
    end else  { alle anderen Botschaften -> DefWindowProc }
     Result := DefWindowProc(FHelpWndHandle,Msg, wParam, lParam);
end;

procedure Register;
begin
  RegisterComponents('Delphi Corner', [THookedApp]);
end;

{$IFNDEF WIN32}  { entfllt fr Win32 komplett, siehe Artikel }
procedure UnloadShellHookDLL; far;
begin
  if HookDLLHandle <> 0 then FreeLibrary(HookDLLHandle);
end;
{$ENDIF}

initialization
  { ShHook.dll holt sich dieselbe Botschafts-Kennziffer }
  WM_NotifyHook := RegisterWindowMessage('ShellHookNotify');
  HookDLLHandle := LoadLibrary('ShHook.DLL');
{$IFDEF WIN32} { #define @ wr natrlich nicht schlecht }
  if HookDLLHandle <> 0 then
  begin  { die Adressen der beiden Routinen abholen }
    HookDLLRegisterHelpWindow :=
      GetProcAddress(HookDLLHandle,'HookDLLRegisterHelpWindow');
    HookDLLUnregisterHelpWindow :=
      GetProcAddress(HookDLLHandle,'HookDLLUnregisterHelpWindow');
    { keine Exit-Prozedur mit FreeLibrary! }
  end;
{$ELSE}  { Win16: "Handles" 1..31 = DOS-Fehlercodes }
  if HookDLLHandle >= 31 then
  begin
    @HookDLLRegisterHelpWindow :=
      GetProcAddress(HookDLLHandle,'HookDLLRegisterHelpWindow');
    @HookDLLUnregisterHelpWindow :=
      GetProcAddress(HookDLLHandle,'HookDLLUnregisterHelpWindow');
    AddExitProc(UnloadShellHookDLL);  { <- FreeLibrary }
  end
    else HookDLLHandle := 0;
{$ENDIF}
end.

