// cdrom.h
// 
// Direkter Zugriff auf SCSI-CD-ROM.
// Datenstrukturen und Funktionsprototypen.
//
// Johannes Karanitsch, 01.12.1996

#include <windows.h>
#include <winioctl.h>
#include <stddef.h>
#include <iostream.h>
#include <fstream.h>
#include <strstrea.h>
#include <iomanip.h>
#include <direct.h>
#include <fcntl.h>

//
// SCSI Definitionen.
//

#define IOCTL_SCSI_BASE                 FILE_DEVICE_CONTROLLER
#define IOCTL_SCSI_GET_CAPABILITIES     CTL_CODE(IOCTL_SCSI_BASE, 0x0404, METHOD_BUFFERED, FILE_ANY_ACCESS)
#define IOCTL_SCSI_PASS_THROUGH_DIRECT  CTL_CODE(IOCTL_SCSI_BASE, 0x0405, METHOD_BUFFERED, FILE_READ_ACCESS | FILE_WRITE_ACCESS)

#define SCSI_IOCTL_DATA_OUT          0
#define SCSI_IOCTL_DATA_IN           1
#define SCSI_IOCTL_DATA_UNSPECIFIED  2

typedef struct _IO_SCSI_CAPABILITIES {
    ULONG Length;
    ULONG MaximumTransferLength;
    ULONG MaximumPhysicalPages;
    ULONG SupportedAsynchronousEvents;
    ULONG AlignmentMask;
    BOOLEAN TaggedQueuing;
    BOOLEAN AdapterScansDown;
    BOOLEAN AdapterUsesPio;
} IO_SCSI_CAPABILITIES, *PIO_SCSI_CAPABILITIES;

typedef struct _SCSI_PASS_THROUGH_DIRECT {
    USHORT Length;
    UCHAR ScsiStatus;
    UCHAR PathId;
    UCHAR TargetId;
    UCHAR Lun;
    UCHAR CdbLength;
    UCHAR SenseInfoLength;
    UCHAR DataIn;
    ULONG DataTransferLength;
    ULONG TimeOutValue;
    PVOID DataBuffer;
    ULONG SenseInfoOffset;
    UCHAR Cdb[16];
} SCSI_PASS_THROUGH_DIRECT, *PSCSI_PASS_THROUGH_DIRECT;

// --------------------------

typedef struct _SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER {
  SCSI_PASS_THROUGH_DIRECT sptd;
  ULONG Filler;      // realign buffer to double word boundary
  UCHAR ucSenseBuf[32];
} SCSI_PASS_THROUGH_DIRECT_WITH_BUFFER, *PSCSI_PASS_THROUGH_DIRECT_WITH_BUFFER;

typedef struct _SCSI_INQUIRY_DEVICE {
  UCHAR Type;
  UCHAR TypeModifier;
  UCHAR Version;
  UCHAR Format;
  UCHAR AddLength; // n-4
  UCHAR Reserved[2];
  UCHAR Flags;
  char  VendorId[8];
  char  ProductId[16];
  char  ProductRevLevel[4];
} SCSI_INQUIRY_DEVICE, *PSCSI_INQUIRY_DEVICE;

//
// SCSI CDB operation codes
//

#define SCSIOP_READ6               0x08
#define SCSIOP_INQUIRY             0x12
#define SCSIOP_MODE_SELECT         0x15
#define SCSIOP_MODE_SENSE          0x1A
//
// SCSI CDB operation codes for NEC CDROM
//

#define SCSIOP_ADD_MODE_SELECT     0xC5
#define SCSIOP_ADD_MODE_SENSE      0xCA
#define SCSIOP_READ_CDDA_10        0xD4

//
// SCSI CDB operation codes for HITACHI & SONY CDROM
//

#define SCSIOP_READ_CDDA_12        0xD8

#define AUDIO_BLOCK_LEN 2352

//
// CDROM Definitionen.
//

enum OPTION
{
  unknown, list, play, record
};



class MSF
{
public:
  MSF(void);
  MSF(unsigned l);
  MSF(unsigned m, unsigned s, unsigned f);
  void SetMinute(unsigned m) { Minute = m; }
  void SetSecond(unsigned s) { Second = s; }
  void SetFrame(unsigned f)  { Frame = f; }
  void SetLbn(unsigned l);
  unsigned GetMinute(void) const { return Minute; }
  unsigned GetSecond(void) const { return Second; }
  unsigned GetFrame(void) const  { return Frame; }
  unsigned GetLbn(void) const { return (Minute * 60 + Second) * 75 + Frame; }
  MSF operator+ (const MSF& msf) const;
  int operator< (const MSF& msf) const { return GetLbn() < msf.GetLbn(); }
  int operator> (const MSF& msf) const { return GetLbn() > msf.GetLbn(); }
  friend ostream& operator<< (ostream& os, MSF& msf);

private:
  unsigned Minute;
  unsigned Second;
  unsigned Frame;
};



class TRACK
{
public:
  TRACK(void);
  void SetType(int t) { Type = t; }
  void SetStart(unsigned m, unsigned s, unsigned f)
    { Start.SetMinute(m); Start.SetSecond(s); Start.SetFrame(f); }
  void SetLength(unsigned m, unsigned s, unsigned f)
    { Length.SetMinute(m); Length.SetSecond(s); Length.SetFrame(f); }
  int GetType(void) { return Type; }
  MSF& GetStart(void) { return Start; }
  MSF& GetLength(void) { return Length; }
  friend ostream& operator<< (ostream& os, TRACK& track);

private:
  int Type;
  MSF Start;
  MSF Length;
};



class TOC
{
public:
  TOC(void);
  ~TOC(void);
  int Read(char* d);
  TRACK* GetTrack(int t);
  MSF& GetCDLength(void) { return CDLength; }
  friend ostream& operator<< (ostream& os, TOC& toc);

private:
  char* MediaIdentity;
  char* Upc;
  MSF CDLength;
  int NumTracks;
  TRACK* Tracks;
};



class CDROM
{
public:
  CDROM(char* d);
  virtual ~CDROM(void);
  TRACK* GetTrack(int t) { return toc.GetTrack(t); }
  MSF& GetCDLength(void) { return toc.GetCDLength(); }
  int readtoc(void) { return toc.Read(&Device[4]); }
  DWORD openmci(void);
  DWORD playmci(MSF& Start, MSF& Length);
  DWORD closemci(void);
  virtual DWORD open(void);
  virtual DWORD read(DWORD lb, DWORD bc, void* db, DWORD dl);
  virtual DWORD close(void);
  friend ostream& operator<< (ostream& os, CDROM& cdrom);
  friend class CDROM_TYPE_1;
  friend class CDROM_TYPE_2;
  friend class CDROM_TYPE_3;

private:
  char* Device;
  HANDLE hFile;
  MCIDEVICEID DeviceID;
  DWORD mb;
  TOC toc;
};



class CDROM_TYPE_1 : public CDROM
{
public:
  CDROM_TYPE_1(char* d);
  ~CDROM_TYPE_1(void);
  DWORD open(void);
  DWORD read(DWORD lb, DWORD bc, void* db, DWORD dl);
  DWORD close(void);
  DWORD ModeSense(void *db, DWORD dl);
  DWORD ModeSelect(void *db, DWORD dl);

private:
  UCHAR SaveMode;
};



class CDROM_TYPE_2 : public CDROM
{
public:
  CDROM_TYPE_2(char* d);
  DWORD read(DWORD lb, DWORD bc, void* db, DWORD dl);
};



class CDROM_TYPE_3 : public CDROM
{
public:
  CDROM_TYPE_3(char* d);
  ~CDROM_TYPE_3(void);
  DWORD open(void);
  DWORD close(void);
  DWORD ModeSense(void *db, DWORD dl);
  DWORD ModeSelect(void *db, DWORD dl);

private:
  UCHAR SaveMode[12];
};



//
// Wave Datei Definitionen.
//

class WAVFILE
{
public:
  WAVFILE(void);
  ~WAVFILE(void);
  DWORD open(char* Filename, MSF& Length);
  DWORD write(char* Buffer, long Length);
  DWORD close(void);

private:
  HMMIO hFile;
  MMCKINFO ckRiff, ckData;
};

//
// Function prototypes
//

// cdinfo.cpp

char GetFirstCDRom(void);
DWORD GetCDRomType(char* device, int* type);

// scsi.c

DWORD ScsiMaxBlocks(HANDLE fh, DWORD* mb);
DWORD ScsiInquire(HANDLE fh, PSCSI_INQUIRY_DEVICE id);
// DWORD ScsiModeSense(HANDLE fh, BYTE PageCode, void* db, DWORD dl);
