unit Hotspot; { as (Arne Schpers) / c't 6/95 }
{ THotspot: Transparentes Fenster mit OnClick-Ereignis
  THotspotPoly: dito, aber Klick-Bereich und Cursorvernderung nur
  innerhalb eines Polygons. Setzen der Eckpunkte siehe SetHelp
  - TPoint hat in Delphi 2.0 zwei LongInt-Felder, weshalb sich die
    Zeiger von FPoints nicht direkt fr Polygone und Regionen verwenden
    lassen; eine direkte Typumwandlung von Pointer nach TPoint ist
    ebenfalls unmglich. Die Routine GetPoint bernimmt die notwendige
    Konvertierung.
  - LineProc fr LineDDA mu in Delphi 1.0 als EXPORT, in Delphi 2.0
    als STDCALL deklariert sein. Der letzte Parameter von LineDDA ist
    in Win16 ein Pointer, in Win32 ein LongInt ) }
interface
uses Classes, Controls, WinTypes, WinProcs, Messages, Graphics, Forms;

const crHotspot = $7000;  { Kennziffer fr selbstdefinierten Cursor }

type

  THotspot = class(TCustomControl)
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure WMEraseBkgnd(var Message: TWmEraseBkgnd);
              message WM_ERASEBKGND;
    procedure SetBounds(ALeft,ATop,AWidth,AHeight: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property OnClick;
    { weitere Eigenschaften: OnDblBlick, OnMouseDown... }
  end;

  THotspotPoly = class(THotspot)  { erweiterte Version }
  private
    FCursor: TCursor;
    FPoints: TList;  { keine Pointer, sondern TPoint-Elemente }
    FEditPoint: Integer; { <> -1 -> Points verschieben }
    FRegion: HRgn;  { Region fr das Polygon }
    FPointsChanged: Boolean;  { Region neu aufbauen }
    FShowFrame: Boolean;  {True: Polygon auch zur Laufzeit (Debugging)}
  protected
    procedure DefineProperties(Filer: TFiler); override;
    procedure WriteFPointData(Writer: TWriter);
    procedure ReadFPointData(Reader: TReader);
    procedure PointsAltered;  { Notify an Designer und Invalidate }
    procedure Paint; override;
    procedure Invalidate; override;
    procedure SetShowFrame(Value: Boolean);
    procedure WMLButtonDown(var Message: TWMMouse);
              message WM_LBUTTONDOWN;
    procedure WMMouseMove(var Message: TWMMouse);
              message WM_MOUSEMOVE;
    procedure WMSetCursor(var Message: TWMSetCursor);
    message WM_SETCURSOR;
    function GetHitPolyPoint(MouseX,MouseY: Integer): Integer;
    function AddPoint(X,Y: Integer): Boolean;
    function GetPoint(Index: Integer): TPoint;
    procedure WndProc(var Message: TMessage); override;
    procedure SetCursor(Value: TCursor); function GetCursor: TCursor;
    function GetHelp: Boolean;  { liefert immer False }
    procedure SetHelp(Value: Boolean); { Kurzanleitung }
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function MouseInPoly(MouseX,MouseY: Integer): Boolean;
  published
    property OnClick;
    property Cursor read GetCursor write SetCursor;
    property ShowFrame: Boolean read FShowFrame write SetShowFrame;
    property AHelp: Boolean read GetHelp write SetHelp;
  end;

  procedure Register;

implementation
{$IFDEF WIN32} {$R HOTSPT32.RES} {$ELSE} {$R HOTSPOT.RES} {$ENDIF}

constructor THotspot.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Width := 100; Height := 100; { Grenvorgabe fr Designer }
end;

procedure THotspot.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.ExStyle := Params.ExStyle or WS_EX_TRANSPARENT;
end;

procedure THotspot.WMEraseBkgnd(var Message: TWmEraseBkgnd);
begin  { Hintergrund nicht lschen } Message.Result := 1; end;

procedure THotspot.SetBounds(ALeft,ATop,AWidth,AHeight: Integer);
var VisibleState: Boolean;
begin
  VisibleState := (Parent <> nil) and IsWindowVisible(Handle);
  { Sonst kopiert Windows den alten Fensterinhalt mit }
  if VisibleState then ShowWindow(Handle,SW_HIDE);
  inherited SetBounds(ALeft,ATop,AWidth,AHeight);
  if VisibleState then ShowWindow(Handle,SW_SHOW);
end;


{ -------------------- THotspotPoly ----------------------------- }
var
  LastMouseX, LastMouseY: Integer; {global: jeweils letztes MouseMove}

{ ReadFPoints/WriteFPoints beim Lesen/Speichern der Komponente
  aufrufen. Ohne die Definition von FPoints als Property wrde der
  Designer dieses Feld ignorieren. }
procedure THotspotPoly.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);
  Filer.DefineProperty('FPoints', ReadFPointData,
     WriteFPointData, FPoints.Count > 0);
end;

procedure THotspotPoly.WriteFPointData(Writer: TWriter);
var ElemCount: Word;
begin
  ElemCount := FPoints.Count;
  Writer.Write(ElemCount,SizeOf(ElemCount));
  Writer.Write(FPoints.List^,SizeOf(Pointer)*ElemCount);
end;

procedure THotspotPoly.ReadFPointData(Reader: TReader);
var ElemCount: Word;
begin
  FPoints.Clear;
  Reader.Read(ElemCount, SizeOf(ElemCount));
  FPoints.Count := ElemCount;
  Reader.Read(FPoints.List^,SizeOf(Pointer)*ElemCount);
end;

{ ---------------------------------- }
constructor THotspotPoly.Create(AOwner: TComponent);
begin
  inherited Create(AOwner); FPoints := TList.Create;
  FEditPoint := -1; { im Moment keine Verschiebung von Eckpunkten }
  FRegion := 0; FPointsChanged := True;  { Region neu aufbauen }
  Cursor := crHotspot;
end;

destructor THotspotPoly.Destroy;
begin
  FPoints.Destroy;  { Points anstelle von Pointern in der Liste }
  if FRegion <> 0 then DeleteObject(FRegion);
  inherited Destroy;
end;

{ Strg+Mausklick im Designer: Wenn weniger als drei Eckpunkte:
  neuen Eckpunkt einfach anhngen; ansonsten die Linien via LineDDA
  abklappern und auf diese Weise den Einfgepunkt bestimmen. }
procedure LineDDAProc(X,Y: Integer; lpFoundLine: PBool);
{$IFDEF WIN32} stdcall; {$ELSE} export; {$ENDIF}
begin
  if (Abs(X-LastMouseX) < 5) and (Abs(Y-LastMouseY) < 5)
     then lpFoundLine^ := True;
end;

{ TPoint hat in Win32 zwei LongInt-Felder, keine direkter Cast
  Typumwandlung mglich }
function THotspotPoly.GetPoint(Index: Integer): TPoint;
begin
{$IFDEF WIN32}  { obere 16 Bit 0, lt sich bei MM_TEXT verschmerzen }
   Result.X := LoWord(LongInt(FPoints.List^[Index]));
   Result.Y := HiWord(LongInt(FPoints.List^[Index]));
{$ELSE}  { Win16: direkte Typumwandlung }
   Result := TPoint(FPoints.List^[Index]);
{$ENDIF}
end;

function THotspotPoly.AddPoint(X,Y: Integer): Boolean;
var InsIndex: Integer;
{ mu im DSeg liegen, Compilerfehler in Delphi 1.0 }
const FoundLine: Boolean = False;
  procedure CallLineDDA(Start,Stop: Integer);
  var StartP,StopP: TPoint;
  begin
    StartP := GetPoint(Start); StopP := GetPoint(Stop);
{$IFDEF WIN32}  { letzter Parameter: LongInt }
   LineDDA( StartP.X,StartP.Y,StopP.X,StopP.Y,@LineDDAProc,
            LongInt(@FoundLine));
{$ELSE}         { Win16: letzter Parameter: Pointer }
   LineDDA(StartP.X,StartP.Y,StopP.X,StopP.Y,@LineDDAProc,@FoundLine);
{$ENDIF}
  end;
begin
  Result := True;
  if FPoints.Count < 3 then FPoints.Add(Pointer(MakeLong(X,Y)))
  else begin
    LastMouseX := X; LastMouseY := Y; FoundLine := False;
    for InsIndex := 0 to FPoints.Count-2 do begin
      CallLineDDA(InsIndex,InsIndex+1);
      if FoundLine then Break;
    end;
    if FoundLine then FPoints.Insert(InsIndex+1,Pointer(MakeLong(X,Y)))
    else begin { Verbindungslinie vom letzten Punkt zum Anfang }
      CallLineDDA(FPoints.Count-1,0);
      if FoundLine then FPoints.Add(Pointer(MakeLong(X,Y)))
      else Result := False;
    end;
  end;
end;

procedure THotspotPoly.SetShowFrame(Value: Boolean);
begin
  if Value = ShowFrame then Exit;
  FShowFrame := Value; Invalidate;
end;

{ Meldung an den Designer: Komponentendaten verndert, d.h. Speichern
   bzw. Rckfrage beim Schlieen des Projekts notwendig }
procedure THotspotPoly.PointsAltered;
begin
  with GetParentForm(Self) do if Designer<>nil then Designer.Modified;
  Invalidate;
end;

procedure THotspotPoly.Invalidate;
var R: TRect;
begin
  {Neuzeichnen des dahinterliegenden (= bergeordneten) Fensters}
  R := ClientRect; MapWindowPoints(Handle,Parent.Handle,R,2);
  InvalidateRect(Parent.Handle,@R,False);
  { Neuzeichnen des Spots }
  InvalidateRect(Handle,nil,False);
end;

procedure THotspotPoly.Paint;
type PtArray = Array[0..999] of TPoint;
var x: Integer; Pt: TPoint; PTemp: ^PtArray;
begin
  if FPointsChanged then begin { .Create, Add/Delete/Move von Points }
    if FRegion <> 0 then DeleteObject(FRegion);
    FRegion := 0;
    if FPoints.Count > 2 then begin
{$IFDEF WIN32}  { Pointer-Array nicht direkt fr Region verwendbar }
      GetMem(PTemp,FPoints.Count*SizeOf(TPoint));
      for x := 0 to FPoints.Count-1 do PTemp^[x] := GetPoint(x);
      FRegion := CreatePolygonRgn(PTemp^,FPoints.Count,WINDING);
      FreeMem(PTemp,FPoints.Count*SizeOf(TPoint));
{$ELSE}
      FRegion:= CreatePolygonRgn(FPoints.List^,FPoints.Count,WINDING);
{$ENDIF}
      FPointsChanged := False;
    end;
  end;
  if (FPoints.Count > 0) and (csDesigning in ComponentState)
  or ShowFrame then with Canvas do begin
    { Polygon }
    Pen.Style := psSolid; Pen.Width   := 1; Pen.Color := clBlack;
    Pen.Mode := pmNotXor; Brush.Style := bsClear;
    if FPoints.Count>0 then begin
      Pt := GetPoint(0); MoveTo(Pt.X,Pt.Y);
      for x := 0 to FPoints.Count-1 do begin
        Pt := GetPoint(x); LineTo(Pt.X,Pt.Y);
        if csDesigning in ComponentState then { Anfasser }
          with Pt do Rectangle(X-3,Y-3,X+3,Y+3);
      end;
      Pt := GetPoint(0); LineTo(Pt.X,Pt.Y);  { letzte Linie }
    end;
    { In der 16-Bit-Version lieen sich die Linien direkt zeichnen }
    { WinProcs.Polygon(Canvas.Handle,FPoints.List^,FPoints.Count); }
    { Fensterrahmen }
    Rectangle(0,0,Width,Height);
  end;
end;

{Trefferprfung in den Kstchen um die PolyPoints herum. Nur Designer}
function THotspotPoly.GetHitPolyPoint(MouseX,MouseY: Integer):Integer;
var x: Integer; R: TRect; P: TPoint;
begin
  Result := -1; P.X := MouseX; P.Y := MouseY;
  for x := 0 to FPoints.Count-1 do begin
    with GetPoint(x) do begin
      R.Left := X-3; R.Top := Y-3;
      R.Right := R.Left +6; R.Bottom := R.Top+6;
    end;
    if PtInRect(R,P) then begin Result := x; Break; end;
  end;
end;

{ Trefferprfung: Maus innerhalb des Polygons? XRef: MouseDowns }
function THotspotPoly.MouseInPoly(MouseX,MouseY: Integer): Boolean;
begin
  if FRegion = 0 then Result := False
  else Result := PtInRegion(FRegion,MouseX,MouseY);
end;

procedure THotspotPoly.SetCursor(Value: TCursor);
begin FCursor := Value; end;

function THotspotPoly.GetCursor: TCursor;
var szCursor : ARRAY [0..10] of char;
begin
  if (csDesigning in ComponentState)
  or MouseInPoly(LastMouseX,LastMouseY) then Result := FCursor
    else Result := crDefault;
end;

procedure THotspotPoly.WMSetCursor(var Message: TWMSetCursor);
begin
  if MouseInPoly(LastMouseX,LastMouseY) then inherited Cursor:=FCursor
  else inherited Cursor := crDefault;
  inherited;
end;

procedure THotspotPoly.WMLButtonDown(var Message: TWMMouse);
var EditPoint: Integer;
begin
  if csDesigning in ComponentState then begin
    EditPoint := GetHitPolyPoint(Message.XPos,Message.YPos);
    Message.Result := 1;
    if Message.Keys and MK_CONTROL <> 0 then begin { neuer Punkt }
      if EditPoint <> -1
      then MessageBeep(0) { IN einem Kstchen - nix da }
      else if AddPoint(Message.XPos, Message.YPos) then PointsAltered;
    end
    else if Message.Keys and MK_SHIFT <> 0 then
    begin { Punkt lschen }
      if EditPoint <> -1 then begin
        FPoints.Delete(EditPoint);
        PointsAltered;  { Notify an den Designer und Invalidate }
      end
    end
    else if EditPoint<>-1 then FEditPoint := EditPoint { verschieben }
    else Message.Result := 0;  { -> WndProc }
  end    { Laufzeit: ButtonDown weiterleiten, wenn innerhalb Polygon }
  else if MouseInPoly(Message.XPos,Message.YPos) then inherited;
end;

procedure THotspotPoly.WMMouseMove(var Message: TWMMouse);
begin
  LastMouseX := Message.XPos; LastMouseY := Message.YPos;
  if (csDesigning in ComponentState) and (FEditPoint<>-1) then begin
    FPoints.Items[FEditPoint]
      := Pointer(MakeLong(Message.XPos,Message.YPos));
    PointsAltered;  { Notify an den Designer + Invalidate }
    Message.Result := 1;  { erledigt }
  end
  else inherited;
end;

procedure THotspotPoly.WndProc(var Message: TMessage);
begin
  Message.Result := 0;
  if (csDesigning in ComponentState) then case Message.Msg of
      WM_LBUTTONDOWN: WMLButtonDown(TWMMouse(Message));
      WM_MOUSEMOVE:
        if FEditPoint <> -1 then WMMouseMove(TWMMouse(Message));
      WM_LBUTTONUP: FEditPoint := -1;  { Ende Punktverschiebung }
    end; { case }
  if Message.Result <> 1 then inherited WndProc(Message);
end;

function THotspotPoly.GetHelp: Boolean;
begin Result := False; end;

procedure THotspotPoly.SetHelp(Value: Boolean);
begin
  if Value then MessageBox(Handle,
    'STRG+Klick bzw. STRG+Klick auf Linie: Eckpunkt setzen.'#10+
    'SHIFT+Klick auf Kstchen: Eckpunkt lschen.'#10+
    'Kstchen ziehen: Eckpunkt verschieben.'#10,
    'Kurzanleitung fr THotspotPoly',MB_OK);
end;

procedure Register;
begin
  RegisterComponents('Delphi Corner', [THotspot, THotspotPoly]);
end;

initialization
  Screen.Cursors[crHotspot] := LoadCursor(HInstance,'HSCursor');
end.
