/************************ PLUCK.C ******************************
C-Simulation fortschreitender Wellen (traveling waves) zur
Erzeugung von Wellenformen,(nach Helmholtz 1857, Cremer 1979
und J.O. Smith 1992), speicherbar als ASCII- oder .WAV-Datei.
Demo-Programm fuer eine Waveguide-Implementation (physical
modeling): Anzupfen einer Gitarren-Saite.
            Borland 3.1, Speichermodell Large
                    Dr. Justus Noll
***************************************************************/

#include <stdlib.h> 
#include <stdio.h>
#include <string.h>
#include <conio.h>

#define SAMPLINGRATE 44100U                   /* Sampling-Rate */

/* Skalierungmakro: */
#define DOUBLE_ZU_SHORT(x) ((int) ((x)*32768.0))     /* 16 Bit */

/* Datentypen: */
typedef unsigned long DWORD;
typedef unsigned WORD;

typedef struct VerzoegerungsLeitung         /* fuer Wellenform */
 {
  unsigned short* data;                     /* Anfangs-Pointer */
  int Laenge;                                        /* Laenge */
  unsigned short* ptr;                         /* Lese-Pointer */
  unsigned short* sptr;                       /* Start-Pointer */
  unsigned short* end;                         /* Ende-Pointer */
 } VerzLeitung;

typedef struct WaveDateiHeader            /* .wav-Datei-Header */
 {
  const  char main_chunk[4];                        /*  "RIFF" */
  DWORD  dlength;                 /* Dateilaenge ohne Padbytes */
  const  char chunk_type[4];                        /*  "WAVE" */
  const  char sub_chunk[4];                          /* "fmt " */
  DWORD  slength;                /* Sub-Chunk-Laenge (16 Byte) */
  WORD   format;                     /* zur Zeit immer 1 (PCM) */
  WORD   modus;                          /* Mono: 1, Stereo: 2 */
  DWORD  sample_fq;                         /* Sample-Frequenz */
  DWORD  byte_p_sec;                      /* Daten pro Sekunde */
  WORD   byte_p_spl;          /* Bytes/Sample: 1=8bit, 2=16bit */
  WORD   bit_p_spl;              /* Bits/Sample: 8, 12 oder 16 */
  const  char data_chunk[4];          /*  Datenkennung: "data" */
  DWORD  data_length;               /*  Laenge des Datenblocks */
 } WAVEHD;

/* Globale Variablen: */
VerzLeitung *ULEITUNG;             /* Delay fuer rechte Welle  */
VerzLeitung *OLEITUNG;               /* Delay fuer linke Welle */
unsigned short MAXSAMPLE;
int ASCII=0;                          /* ASCII-Datei schreiben */
unsigned short FILTER_STATUS=0;    /* Speicher fuer Filterwert */

/*** Speicherpruefung, Exit-Nr. ********************************/
void Fehler(int nummer)
 {
  switch(nummer)
   {
    case 1:
     fprintf(stderr,
      "\nSpeicherfehler vl-Struktur.");
     break;
    case 2:
     fprintf(stderr,
      "\nSpeicherfehler vl->data.");
     break;
    case 3:
     fprintf(stderr,
      "\nSpeicherfehler AnfangsForm.");
     break;
    case 4:
     fprintf(stderr,
      "\nSpeicherfehler ASCII-Datei.");
     break;
    case 5:
     fprintf(stderr,
      "\nSpeicherfehler beim Oeffnen Wave-Datei.");
     break;
    case 6:
     fprintf(stderr,
      "\nSpeicherfehler beim wav-Header.");
     break;
    case 7:
     fprintf(stderr,
      "\nSpeicherfehler in der Schnappschuss-Routine.");
     break;
    case 8:
     fprintf(stderr,
      "Syntax:\n"
      "-------\n"
      "Amplitude (<1.0)                [Space]\n"
      "Tonhoehe (Hz)                   [Space]\n"
      "Zupfposition (<1.0)             [Space]\n"
      "Zupfhoehe (<1.0)                [Space]\n"
      "Dauer (sec)                     [Space]\n"
      "Schnappschuss-Sample-Nr.        [Space]\n"
      "Dateiname (ohne Ext.)           [Space]\n"
      "ASCII-Datei schreiben? (0/1)    [Return]\n"
      "---------\n"
      "Beispiel:\n"
      "---------\n"
      "pluck 0.5 100 0.1 0.2 0.7 22 test 0 [Return]\n");
     break;
    case 9:
     fprintf(stderr,
      "\nSpeicherfehler in der Main-Routine.");
     break;
   }
  exit(nummer);
 }
/*** Einrichtung einer Verzgerungsleitung *********************/
VerzLeitung *VerzLeitungAn(VerzLeitung *vl,
                           double* daten,
                           double Skalieren,
                           int len)
 {
  int i;
/*--- Speicher reservieren ------------------------------------*/
  vl=(VerzLeitung*) calloc(len,sizeof(VerzLeitung));
  if(vl==NULL) Fehler(1);
  vl->Laenge=len;
  vl->data=(unsigned short*) calloc(len,sizeof(unsigned short));
  if(vl->data==NULL) Fehler(2);
  vl->ptr=vl->data;
  vl->sptr=vl->data+MAXSAMPLE;
  vl->end=vl->data+len -1;
/*--- mit Daten fllen ----------------------------------------*/
  for(i=0;i<vl->Laenge;i++)
   vl->data[i]=DOUBLE_ZU_SHORT(daten[i]*Skalieren);
  return vl;
 }
/*** Speicherfreigabe ******************************************/
 void VerzLeitungAus(VerzLeitung *vl)
 {
  if(vl && vl->data) free(vl->data);
  vl->data=0;
  free(vl);
 }
/*** freeString ************************************************/
 void freeString(void)
 {
  VerzLeitungAus(ULEITUNG);
  VerzLeitungAus(OLEITUNG);
 }
/*** Pointer erhhen *******************************************/
unsigned short* inkrementieren(VerzLeitung *vl,
                               unsigned short* ptr)
 {
  ptr++;
  if(ptr>vl->end) ptr=vl->data;
  return ptr;
 }
/*** Pointer erniedrigen ***************************************/
unsigned short* dekrementieren(VerzLeitung* vl,
                               unsigned short* ptr)
 {
  ptr--;
  if(ptr<vl->data) ptr=vl->end;
  return ptr;
 }
/*** Filter fr den Steg: Einfaches 1-poliges Filter **********/
 unsigned short StegFilter(int InputSample)
 {
  unsigned short OutputSample=0.5*(FILTER_STATUS+InputSample);
  FILTER_STATUS = OutputSample;
  return OutputSample;
 }
/*** Sample + Reflexion berechnen ******************************/
unsigned short naechstesSample(void)
 {
  unsigned short StegSample,OutputSample;

/*--- neuen Wellenwert einlesen (sptr) ------------------------*/
  OutputSample=*ULEITUNG->sptr;              /* untere Leitung */
  ULEITUNG->sptr=dekrementieren(ULEITUNG,ULEITUNG->sptr);
  OutputSample+=*OLEITUNG->sptr;/* obere Leitung hinzuaddieren */
  OLEITUNG->sptr=inkrementieren(OLEITUNG,OLEITUNG->sptr);
/*--- Reflektion an den Saitenenden (ptr) ---------------------*/
  OLEITUNG->ptr=inkrementieren(OLEITUNG,OLEITUNG->ptr);
  StegSample=-StegFilter(*OLEITUNG->ptr);   /* Filtern am Steg */
  ULEITUNG->ptr=dekrementieren(ULEITUNG,ULEITUNG->ptr);/* 2x...*/
  ULEITUNG->ptr=dekrementieren(ULEITUNG,ULEITUNG->ptr);/*weiter*/
  *OLEITUNG->ptr=-(*ULEITUNG->ptr); /* negatives Sattel-Sample */
  ULEITUNG->ptr=inkrementieren(ULEITUNG,ULEITUNG->ptr);
  *ULEITUNG->ptr=StegSample;          /* Steg-Sample speichern */
/*--- Rckgabe-Sample -----------------------------------------*/
  return OutputSample;
 }
/*** InitWelle *************************************************
 Die Dreiecksform aus ansteigender und absteigender Flanke einer
 angezupften Saite wird ausgerechnet. Die Anzahl der Samples ist
 von der Frequenz und der Sampling-Rate abhngig, z.B.
 44200/100/2 macht fr 100 Hz 221 Samples notwendig, um mit der
 Nyqvist-Frequenz zu sampeln.
 ***************************************************************/
int initWelle(double Amplitude,                    /* z.B. 0.5 */
               double Frequenz,               /* z.B. 100 [Hz] */
               double ZupfStelle,   /* z.B. 0.1 (Saitenlaenge) */
               double SampleOffset)                 /* z.B. 22 */
 {
  int i;
  int LeitungsLaenge,StartSample;
  double *AnfangsForm,anFlanke,abFlanke;

  LeitungsLaenge=SAMPLINGRATE/Frequenz/2+1;
  StartSample=MAXSAMPLE=max(LeitungsLaenge*ZupfStelle,1);
  AnfangsForm=(double*) calloc(LeitungsLaenge,sizeof(double));
  if(AnfangsForm==NULL) Fehler(3);
  anFlanke=Amplitude/StartSample;
  abFlanke=Amplitude/(LeitungsLaenge-StartSample-1);
  for(i=0;i<StartSample;i++) AnfangsForm[i]=anFlanke*i;
  for(i=StartSample;i<LeitungsLaenge;i++)
   AnfangsForm[i]=abFlanke*(LeitungsLaenge-1-i);
  ULEITUNG
   =VerzLeitungAn(ULEITUNG,AnfangsForm,0.5,LeitungsLaenge);
  OLEITUNG
   =VerzLeitungAn(OLEITUNG,AnfangsForm,0.5,LeitungsLaenge);
  return SampleOffset*LeitungsLaenge;
 }
/*** schreibeASCII *********************************************/
 int schreibeASCII(char* name,
                      unsigned short* soundData,
                      int sampleCount)
 {
  int i;
  FILE* fil;

  if((fil=fopen(name,"w+b"))==NULL) Fehler(4);
  printf("\nSchreibe ASCII-Datei %s.",name);
  for(i=0;i<sampleCount;i++) fprintf(fil,"%i,",soundData[i]);
  fclose(fil);
  return (0);
 }
/*** schreibeWave **********************************************
 Schreiben einer .wav-Datei mit Header und folgendem Datenblock.
 Falls die Chunk-Lngen nicht gerade sind, mssen "Padbytes"
 eingefgt werden, die bei den Lngenangaben nicht
 mitgerechnet werden.
 ***************************************************************/
 int schreibeWave(char* name,
                  unsigned short* soundData,
                  int sampleCount)
 {
  int i;
  FILE* fil;
  WAVEHD* wav;

  if((fil=fopen(name,"w+b"))==NULL) Fehler(5);
  printf("\nSchreibe Wave-Datei %s.",name);
  wav=(WAVEHD*) calloc(1,sizeof(WAVEHD));
  if(wav==NULL) Fehler(6);
/*--- .wav-Datei schreiben ------------------------------------*/
  strcpy(wav->main_chunk,"RIFF");           /*  Kennung "RIFF" */
  strcpy(wav->chunk_type,"WAVE");            /* Kennung "WAVE" */
  strcpy(wav->sub_chunk,"fmt ");             /* Kennung "fmt " */
  strcpy(wav->data_chunk,"data");
  wav->slength=16;               /* Sub-Chunk-Laenge (16 Byte) */
  wav->format=1;                    /*  zur Zeit immer 1 (PCM) */
  wav->modus=1;                          /* Mono: 1, Stereo: 2 */
  wav->sample_fq=SAMPLINGRATE;            /* Sampling-Frequenz */
  wav->dlength=sizeof(int)*sampleCount+36;
  wav->byte_p_sec=2*SAMPLINGRATE;         /* Bytes pro Sekunde */
  wav->byte_p_spl=2;               /* Bytes/Sample: 1 = 16 Bit */
  wav->bit_p_spl=16;                        /* Bits pro Sample */
  wav->data_length=sizeof(int)*sampleCount;   /* Laenge "data" */
  fwrite(wav,1,sizeof(WAVEHD),fil);
  fwrite(soundData,2,sampleCount,fil);
  fclose(fil);
/*--- ASCII-Datei schreiben -----------------------------------*/
  if(ASCII>0)
   {
    for(i=0;i<strlen(name)-1;i++) if(name[i]=='.') name[i]=0;
    strcat(name,".snd");
    schreibeASCII(name,soundData,sampleCount);
   }
  return (0);
 }
/*** schreibeSchnappschuss **************************************/
void schreibeSchnappschuss(double SampleNr)
 {
  int i, sampleCount;
  unsigned short *data;
  char str[13];

  sampleCount=ULEITUNG->Laenge;
  data
   =(unsigned short*) calloc(sampleCount,sizeof(unsigned short));
  if(data==NULL) Fehler(7);
  if(ASCII>0)
   {
    printf("\nSchreibe ASCII-Sample %i.",(unsigned int)SampleNr);
    sprintf(str,"%io.snd",(unsigned int) SampleNr);
    for(i=0;i<sampleCount;i++)
   data[i]=*(OLEITUNG->data+i);
    schreibeASCII(str, data, sampleCount);
    for(i=0;i<sampleCount;i++)
     data[i]=*(ULEITUNG->data+i);
    sprintf(str,"%iu.snd",(unsigned int) SampleNr);
    schreibeASCII(str, data, sampleCount);
   }
  for(i=0;i<sampleCount;i++)
   data[i]=*(OLEITUNG->data+i);
  if(ASCII>0)
   {
    sprintf(str,"%i.snd",(unsigned int) SampleNr);
    schreibeASCII(str, data, sampleCount);
   }
  sprintf(str,"%i.wav",(unsigned int) SampleNr);
  schreibeWave(str,data,sampleCount);
 }
/*** Main ******************************************************/
void main (int argc, char* argv[])
 {
  unsigned short i,sampleCount;
  unsigned short* data;
  char dateiName[9];
  char name[13];
  double Amplitude,Tondauer,Tonhoehe,ZupfStelle;
  double SampleOffset;
  int einSample;

  clrscr();
  printf("================================\n"
         " PLUCK.C   (C) Dr.Justus Noll   \n"
         " Zur Zeit ist die Sample-Laenge \n"
         " auf 0.7 Sekunden begrenzt.     \n"
         "================================\n\n");
  if(argc!=9) Fehler(8);
  sscanf(argv[1],"%lf",&Amplitude);
  sscanf(argv[2],"%lf",&Tonhoehe);
  sscanf(argv[3],"%lf",&ZupfStelle);
  sscanf(argv[4],"%lf",&SampleOffset);
  sscanf(argv[5],"%lf",&Tondauer);
  if(Tondauer>0.7)
   {
    printf("Tondauer auf 0.7 Sekunden begrenzt!\n");
    Tondauer=0.7;
   }
  sscanf(argv[6],"%d",&einSample);
  strncpy(dateiName,argv[7],8);
  sampleCount=Tondauer*SAMPLINGRATE;
  sscanf(argv[8],"%i",&ASCII);
  if((data=(unsigned short*)
   malloc(sampleCount*sizeof(unsigned short)))==NULL) Fehler(9);
  MAXSAMPLE
   =initWelle(Amplitude,Tonhoehe,ZupfStelle,SampleOffset);
  for(i=0;i<sampleCount;i++)
   {    /* Schnappschuss eines Samples (0=kein Schnappschuss): */
    if(einSample>0)
     if(i-1==einSample) schreibeSchnappschuss(einSample);
    data[i]=naechstesSample();
   }
  strncpy(name,dateiName,8);
  strcat(name,".wav");
  schreibeWave(name,data,sampleCount);
  freeString();
  exit(0);
 }
