unit IForceEffects_Intl;  // Version 1.00; 21-MAR-99 Arne Schpers
{ Delphi classes for force feedback FX created with Immersion's
  I-FORCE Studio (IFR files) & support for compound effects.
  These classes allow the same amount of manipulation at runtime as Immersion's
  IFRHelper, i.e. duration, gain, direction, and type specific params.
  No provisions exist for runtime modification of envelopes & axis count --
  after all, that's what I-FORCE Studio is intended for.

  Usage:
  var
   MyEffectsFile: TDIFFEffectFile;
   MyJoystick: IDirectInputDevice2;
   MyEffect: TDIFFSingleEffect;
  ...

  MyEffectsFile := TEffectFile.Create(MyJoystick);
  with MyEffectsFile do
  begin
    // Creates objects (not effects), names in EffectNames[]
    LoadFromFile('effects.ifr');
    ...
    // Create effect named 'Jump', set direction to 90 degrees, play effect
    with LoadEffect('Jump') as TDIFFSingleEffect do
    begin
      Direction := 90;
      Play(0);  // <- implicit modification and download
    end;
    ...
    UnloadEffect('Jump');  // free device memory

    // Play compound effect named 'DoubleDrill', switch off all other effects
    LoadEffect('DoubleDrill').Play(DIES_SOLO)
    ...
    UnloadEffect('DoubleDrill');

    // Assign Button 1 to single effect 'Fire Bow' (w/o playing) & download
    MyEffect := LoadEffect('Fire Bow') as TDIFFSingleEffect;
    with MyEffect do
    begin
      FireButton := DIJOFS_BUTTON1;
    end;


  Property ranges
  ---------------
  Direction: 0..360, Gain: 0..100; Duration: usecs

  Enhancements over Immersion's IFRHelper class & ReadIFR demo
  ------------------------------------------------------------
  - doesn't reload the IFR file for every single effect
  - plays compound effects correctly
  - SOLO effects (also for compounds)
  - rebuilds effects only after real changes
  - assignment of trigger button (alas, not for compound effects)

  Runtime requirements
  --------------------
  a) IForce2.dll - download latest version from www.immersion.com or www.force-feedback.com
  b) DirectX 5

  Compile time requirements
  -------------------------
  a) DirectX headers converted by Erik Unger - download from www.bigfoot.com/~ungerik/
  b) IForce2.pas - Header file with declarations for IForce2.dll
  c) Delphi 3 or 4. VB just won't make it :-)

  Limits
  ------
  Due to the logic of Immersion's IForce2.DLL, TSingleEffect
  and TCompoundEffect objects can't exist w/o an effect file.
  Adding single effects to compound effects and/or removing
  single effects from compound effects on the fly isn't
  possible, either.

  Happy punching!
}

interface
uses Windows, Classes, SysUtils, DInput, // Erik Unger's header file
     Dialogs, IForce2;  // = IForce2.h from Immersion Corp.

const MAX_AXES = 32;

type
  TDIFFEffectFile = class;  // forward

  TAxisArray = Array[0..MAX_AXES-1] of Integer;

  // abstract base for single and compound effects. Don't use directly
  TDIFFEffect = class(TObject)
    private
      FFXName: String;
    public
      constructor Create(FXName: String);
      function Start(dwFlags: Cardinal): HResult; virtual; abstract;
      function Stop: HResult; virtual; abstract;
      property FXName: String read FFXName;
    end;

  // single force feedback effect
  TDIFFSingleEffect = class(TDIFFEffect)
    private
      ChangeFlags: Cardinal;  // DIEP_DIRECTION or DIEP_...
      FDIEffect: TDIEffect;
      FDirections: TAxisArray; // currently just 2 elements
      FAxes: TAxisArray;       // ditto
      FEnvelope: TDIEnvelope;
    protected
      FEffectObject: PIDirectInputEffect;
      FOwner: TDIFFEffectFile;
    protected
      procedure ResetDIEffectStruct;
      procedure LoadEffect(EffObject: PIDirectInputEffect);
      procedure UnloadEffect;
      function ApplyChanges: HResult;
      function GetLoaded: Boolean;
      function GetEffectObject: IDirectInputEffect;
    protected
      function GetDuration: Cardinal;
      procedure SetDuration(Value: Cardinal);
      function GetDirection: Cardinal;
      procedure SetDirection(Value: Cardinal);
      function GetGain: Integer;
      procedure SetGain(Value: Integer);
      function GetTriggerButton: Cardinal;
      procedure SetTriggerButton(Value: Cardinal);
      function GetTypeSpcParamSize: Cardinal;
      function GetTypeSpcParams: Pointer;
    public  // don't use directly!
      destructor Destroy; override;
    public
      property TriggerButton: Cardinal read GetTriggerButton write SetTriggerButton;
      property Duration: Cardinal read GetDuration write SetDuration;
      property Gain: Integer read GetGain write SetGain;
      property Direction: Cardinal read GetDirection write SetDirection;
      property TypeSpcParamSize: Cardinal read GetTypeSpcParamSize;
      property TypeSpcParams: Pointer read GetTypeSpcParams;
      procedure SetTypeSpcParams(SpcSize: Cardinal; Data: Pointer);
      property EffectObject: IDirectInputEffect read GetEffectObject;
      property Loaded: Boolean read GetLoaded;
    public
      function Start(dwFlags: Cardinal): HResult; override;
      function Stop: HResult; override;
  end;

  // compound force feedback effect
  TDIFFCompoundEffect = class(TDIFFEffect)
    private
      FCompounds: TList;
    protected
      function GetCompoundCount: Integer;
      function GetCompounds(Index: Integer): TDIFFSingleEffect;
      procedure AddCompound(Effect: TDIFFSingleEffect);
    public
      constructor Create(FXName: String);
      destructor Destroy; override;
      // individual effects
      property CompoundCount: Integer read GetCompoundCount;
      property Compounds[Index: Integer]: TDIFFSingleEffect read GetCompounds;
      // start & stop a compound effect (i.e. several effects at once)
      function Start(dwFlags: Cardinal): HResult; override;
      function Stop: HResult; override;
  end;

  // loads an IFR file, creates single & compound effects
  TDIFFEffectFile = class(TObject)
    private
      FHandle: HIForceProject;
      Device: IDirectInputDevice2;
      FEffects: TStringList;  // Objects: TDIFFEffect
    protected
      procedure Cleanup;  // XRef: LoadFromXXX, Destroy
    public
      constructor Create(IFDevice: IDirectInputDevice2);
      destructor Destroy; override;
      function LoadFromFile(FName: String): Boolean;
      function LoadFromStream(S: TStream): Boolean;
    public
      property EffectNames: TStringList read FEffects;
      function LoadEffect(FXName: String): TDIFFEffect;
      procedure UnloadEffect(FXName: String);
    end;

implementation

constructor TDIFFEffect.Create(FXName: String);
begin
  inherited Create; FFXName := FXName;
end;

function TDIFFSingleEffect.GetLoaded: Boolean;
begin
  Result := FEffectObject <> nil;
end;


function TDIFFSingleEffect.GetEffectObject: IDirectInputEffect;
begin
  Result := FEffectObject^;
end;

// XRef: TDIFFEffectFile.LoadFromStream after creation of
// IDirectInputEffect interface by IFORCE2.DLL. Gets all data for this effect.
procedure TDIFFSingleEffect.LoadEffect(EffObject: PIDirectInputEffect);
var NewSpcSize: Cardinal; Res: HResult;
begin
  FEffectObject := EffObject;
  ResetDIEffectStruct;
  Res := EffectObject.GetParameters(FDIEffect,DIEP_ALLPARAMS);
  if Res = DIERR_MOREDATA then
  begin  // (comment hijacked from IFRHelper.cpp)
  // if cbTypeSpecificParams is incorrect or lpvTypeSpecificParams is NULL,
  // DirectInput returns DIERR_MOREDATA and fills cbTypeSpecificParams with
  // the correct value.  Allocate memory for lpvTypeSpecificParams and call
  // GetParameters again.
  // Note that GetParameters will overwrite some DIEFFECT struct member data,
  // such as dwFlags and cAxes.  See DI documentation for more details.
    NewSpcSize := TypeSpcParamSize;
    ResetDIEffectStruct;
    SetTypeSpcParams(NewSpcSize,nil);  // just allocate, don't copy
    Res := EffectObject.GetParameters(FDIEffect,DIEP_ALLPARAMS);
  end;
  if FAILED(Res) then raise Exception.Create('SingleEffect.LoadEffect: '+ErrorString(Res));
  ChangeFlags := 0;  // not changes to this data (yet)
end;

procedure TDIFFSingleEffect.UnloadEffect;
begin
  SetTypeSpcParams(0,nil);  // = FreeMem(lpvSpcTypeSpecificParams)
  FEffectObject := nil;
end;

// (Re-)initialize DIEffect structure. XRef: LoadEffect only (2 times)
procedure TDIFFSingleEffect.ResetDIEffectStruct;
begin
  SetTypeSpcParams(0,nil);  // free;
  // DI requires certain struct members to have non-zero data
  // before calling GetParameters. See DI documentation for more details
  with FDIEffect do
  begin
    dwSize := SizeOf(FDIEffect);
    dwFlags := DIEFF_SPHERICAL or DIEFF_POLAR or DIEFF_CARTESIAN or DIEFF_OBJECTOFFSETS;
    cAxes := MAX_AXES;
    rglDirection := @FDirections; rgdwAxes := @FAxes; lpEnvelope := @FEnvelope;
  end;
  FEnvelope.dwSize := SizeOf(FEnvelope);
end;

destructor TDIFFSingleEffect.Destroy;
begin
  UnloadEffect;
  inherited;
end;

function TDIFFSingleEffect.Stop: HResult;
begin
  Result := EffectObject.Stop;
end;

// ------ Get & set effect properties (quite boring) ---------------
function TDIFFSingleEffect.GetDuration: Cardinal;
begin
  Result := FDIEffect.dwDuration;
end;

procedure TDIFFSingleEffect.SetDuration(Value: Cardinal);
begin
  FDIEffect.dwDuration := Value;
  ChangeFlags := ChangeFlags or DIEP_DURATION;
end;

function TDIFFSingleEffect.GetDirection: Cardinal;
begin
  Result := FDirections[0] div DI_DEGREES;
end;

procedure TDIFFSingleEffect.SetDirection(Value: Cardinal);
begin
  FDirections[0] := Value * DI_DEGREES;
  ChangeFlags := ChangeFlags or DIEP_DIRECTION;
end;

function TDIFFSingleEffect.GetGain: Integer;
begin
  Result := FDIEffect.dwGain div 100;
end;

procedure TDIFFSingleEffect.SetGain(Value: Integer);
begin
  FDIEffect.dwGain := Value * 100;
  ChangeFlags := ChangeFlags or DIEP_GAIN; 
end;

function TDIFFSingleEffect.GetTriggerButton: Cardinal;
begin
  Result := FDIEffect.dwTriggerButton;
end;

procedure TDIFFSingleEffect.SetTriggerButton(Value: Cardinal);
begin
  FDIEffect.dwTriggerButton := Value;
  ChangeFlags := ChangeFlags or DIEP_TRIGGERBUTTON;
end;

function TDIFFSingleEffect.GetTypeSpcParamSize: Cardinal;
begin
  Result := FDIEffect.cbTypeSpecificParams;
end;

function TDIFFSingleEffect.GetTypeSpcParams: Pointer;
begin
  Result := FDIEffect.lpvTypeSpecificParams;
end;

// reallocate buffer and copy data (if data <> nil).
// XRefs: LoadEffect, ResetDIEffectStruct, Destroy (with Data = nil)
procedure TDIFFSingleEffect.SetTypeSpcParams(SpcSize: Cardinal; Data: Pointer);
begin
  if TypeSpcParamSize <> SpcSize then
  with FDIEffect do
  begin
    if cbTypeSpecificParams <> 0 then FreeMem(lpvTypeSpecificParams);
    cbTypeSpecificParams := SpcSize;
    if SpcSize <> 0 then GetMem(lpvTypeSpecificParams,cbTypeSpecificParams)
      else lpvTypeSpecificParams := nil;
  end;
  if Data <> nil then Move(Data^,TypeSpcParams^,TypeSpcParamSize);
  ChangeFlags := ChangeFlags or DIEP_TYPESPECIFICPARAMS;
end;

function TDIFFSingleEffect.ApplyChanges: HResult;
begin
  if ChangeFlags = 0 then Result := DI_OK
  else
  begin
    Result := EffectObject.SetParameters(FDIEffect,ChangeFlags);
    if SUCCEEDED(Result) then ChangeFlags := 0;
  end;
end;

// Start the effect (after possible modification & download)
function TDIFFSingleEffect.Start(dwFlags: Cardinal): HResult;
begin
  Result := ApplyChanges;
  if SUCCEEDED(Result) then
    Result := EffectObject.Start(1,dwFlags);
end;

// -------- Compound effects -------------
constructor TDIFFCompoundEffect.Create(FXName: String);
begin   // EffObject is actually a pointer to array element 0 here
  inherited Create(FXName);
  FCompounds := TList.Create; // lists effects to be started/stopped simultaneously
end;

procedure TDIFFCompoundEffect.AddCompound(Effect: TDIFFSingleEffect);
begin
  FCompounds.Add(Effect);
end;

destructor TDIFFCompoundEffect.Destroy;
begin
  FCompounds.Free;  // leave IForce2.DLL's data structures unaltered
  inherited;
end;

function TDIFFCompoundEffect.GetCompoundCount: Integer;
begin
  Result := FCompounds.Count;
end;

function TDIFFCompoundEffect.GetCompounds(Index: Integer): TDIFFSingleEffect;
begin
  Result := FCompounds[Index];
end;

// Start several effects simultaneously
function TDIFFCompoundEffect.Start(dwFlags: Cardinal): HResult;
var x: Integer;
begin
  Result := 0;
  for x := 0 to CompoundCount-1 do
  begin
    Result := Compounds[x].Start(dwFlags);
    if FAILED(Result) then Break;
    // if dwFlags contain DIES_SOLO, take it out now. Otherwise,
    // starting the 2nd effect would stop the first, and so on
    dwFlags := dwFlags and not DIES_SOLO;
  end;
end;

function TDIFFCompoundEffect.Stop: HResult;  // stop several effects simultaneously
var x: Integer; TempRes: HResult;
begin
  Result := DI_OK;
  for x := 0 to CompoundCount-1 do
  begin
    TempRes := Compounds[x].Stop;
    if Result = DI_OK then Result := TempRes;  // return first error
  end;
end;


// ------ TDIEffectFile -------------------------------------------
constructor TDIFFEffectFile.Create(IFDevice: IDirectInputDevice2);
begin
  inherited Create;
  Device := IFDevice;  // IFORCE2.DLL needs the device for creation of FX
  FEffects := TStringList.Create;
end;

procedure TDIFFEffectFile.Cleanup;
var x: Integer;
begin
  Device.Unacquire;  // SideWinder & WingMan will lose a couple of effects otherwise
  for x := 0 to FEffects.Count-1 do
  begin
    UnloadEffect(FEffects[x]);
    FEffects.Objects[x].Free;
  end;
  FEffects.Clear;
  if FHandle <> nil then IFReleaseProject(FHandle);  // in IFORCE2.DLL
end;

destructor TDIFFEffectFile.Destroy;
begin
  Cleanup;
  inherited;
end;

function TDIFFEffectFile.LoadFromFile(FName: String): Boolean;
var S: TFileStream;
begin
  if Pos('.',ExtractFileName(FName)) = 0 then FName := FName+'.ifr';
  S := TFileStream.Create(FName,fmOpenRead or fmShareDenyWrite);
  Result := LoadFromStream(S);
  S.Free;
end;

function EqualGUID(const G1, G2: TGUID): Boolean;
begin
  Result := CompareMem(@G1,@G2,SizeOf(TGUID));
end;

// See I-FORCE documentation for file structure (or use a hex dumper)
// Basically, a IFR file is a list of named effects and named effect
// properties where each structure and each substructure is preceeded
// by a length count. For single effects, IFCreateEffects (from IFORCE2.DLL)
// returns a single IDirectInputEffect interface; for compound effects, it would
// return an array of pointers to IDirectInputEffect interfaces duplicating
// the single effects defined in the IFR file.
// After a successful load FEffects will contain the names of all effects
// (compound and single), FEffects.Objects holds pointers to TDIEFFSingleEffect
// for single effects, and pointers to TDIEFFCompoundEffect for compound effects.
// TDIEFFCompoundEffect objects maintain an own list with pointers to
// the TIEFFSingleEffect objects they are composed of.
function TDIFFEffectFile.LoadFromStream(S: TStream): Boolean;
var M: TMemoryStream; ObjSize: Cardinal;
    PC: PChar; x,y: Integer;
    NewCompound: TDIFFCompoundEffect;
    GUIDs, SubGUIDs: TStringList; SubGUID: String;
begin
  Cleanup;  // guess...
  Result := Device <> nil;
  if not Result then Exit;
  M := TMemoryStream.Create; M.CopyFrom(S,0); M.Seek(0,0);
  GUIDs := TStringList.Create; SubGUIDs := TStringList.Create;
  try
    FHandle := IFLoadProjectPointer(M.Memory,Device);
    Result := FHandle <> nil; if not Result then Exit;
    M.Seek(8,soFromBeginning);  // signature check performed by IFLoadProject...
    while M.Position < M.Size do
    begin  // collect effect names, and skip the effect data
      M.Read(ObjSize,4);
      PC := M.Memory; Inc(PC,M.Position); FEffects.Add(StrPas(PC));
      // differentiate between compounds and single effects
      SubGUID := '';
      Inc(PC,Strlen(PC)+1);  // skip name
      if PC = 'Compound' then
      begin
        Inc(PC,Strlen(PC)+6);
        if PC = 'ContainedObjects' then
        begin  // GUIDs of single effects: {...};{...};#0
          Inc(PC,StrLen(PC)+1); SubGUID := PC; Inc(PC,StrLen(PC)+1);
        end;
      end else
      begin  // single effect
        Inc(PC,StrLen(PC)+6);
      end;
      SubGUIDs.Add(SubGUID);  // '' or {GUID};{GUID}...
      if PC <> 'ID' then raise Exception.Create('No Effect ID');
      Inc(PC,StrLen(PC)+1); GUIDs.Add(PC); // GUID of this effect or compound
      M.Seek(ObjSize-4,soFromCurrent);
    end;
    // Create objects for single effects w/o creating the effects themselves
    for x := 0 to FEffects.Count-1 do
      if SubGUIDs[x] = '' then
       FEffects.Objects[x] := TDIFFSingleEffect.Create(FEffects[x]);

    // Create compound objects. These objects simply maintain a list of subcomponents
    for x := 0 to FEffects.Count-1 do
      if SubGUIDs[x] <> '' then
      begin
        NewCompound := TDIFFCompoundEffect.Create(FEffects[x]);
        FEffects.Objects[x] := NewCompound;
        for y := 0 to GUIDs.Count-1 do
          if Pos(GUIDs[y], SubGUIDs[x]) <> 0
            then NewCompound.AddCompound(FEffects.Objects[y] as TDIFFSingleEffect);
      end;
  finally
    M.Free; GUIDs.Free; SubGUIDs.Free;
  end;
end;

function TDIFFEffectFile.LoadEffect(FXName: String): TDIFFEffect;
var x, FXIndex: Integer;
  procedure LoadSingleEffect(FX: TDIFFSingleEffect);
  var PIEffect: PIDirectInputEffect; NumEffects: Cardinal;
  begin
    with FX do
      if not Loaded then
      begin
        PIEffect := IFCreateEffects(FHandle,PChar(FXName),NumEffects);
        if (PIEffect = nil) or (NumEffects = 0) then
          raise Exception.Create('Unable to load FFEffect '+FXName);
        LoadEffect(PIEffect);
      end;
  end;

begin
  FXIndex := EffectNames.IndexOf(FXName);
  if FXIndex = -1 then
    raise Exception.Create('EffectFile: no effect named '+FXName);
  Result := FEffects.Objects[FXIndex] as TDIFFEffect;
  if Result is TDIFFSingleEffect
    then LoadSingleEffect(Result as TDIFFSingleEffect)
    else with Result as TDIFFCompoundEffect do
      for x := 0 to CompoundCount-1 do
        LoadSingleEffect(Compounds[x]);
end;

procedure TDIFFEffectFile.UnloadEffect(FXName: String);
var x, FXIndex: Integer; CurFX: TDIFFEffect;
  procedure UnloadSingleEffect(FX: TDIFFSingleEffect);
  begin
    with FX do if Loaded then
    begin
      IFReleaseEffects(FHandle, FEffectObject);
      UnloadEffect;
    end;
  end;
begin
  FXIndex := EffectNames.IndexOf(FXName);
  if FXIndex = -1 then
    raise Exception.Create('EffectFile: no effect named '+FXName);
  CurFX := FEffects.Objects[FXIndex] as TDIFFEffect;
  if CurFX is TDIFFSingleEffect
    then UnloadSingleEffect(CurFX as TDIFFSingleEffect)
  else with CurFX as TDIFFCompoundEffect do
    for x := 0 to CompoundCount-1 do
      UnloadSingleEffect(Compounds[x]);
end;

end.

