unit IForceEffects;  // Version 1.00; 21-MAR-99 Arne Schpers
{ Delphi-Klassen fr Kraftrckmeldungseffekte, die mit Immersions
  I-FORCE Studio definiert wurden (IFR-Dateien) & Verbundeffekte.

  Diese Klassen erlauben dieselben Manipulationen zur Laufzeit
  wie Immersions IFRHelper: Dauer, Strke, Richtung und typ-
  spezifische Parameter. Methoden zur Vernderunge von Hll-
  kurven und der Achsenzahl gibt es nicht - dafr ist schlielich
  das Programm IForce Studio gedacht.

  Einsatz:

  var
   MyEffectsFile: TDIFFEffectFile;
   MyJoystick: IDirectInputDevice2;
   MyEffect: TDIFFSingleEffect;
  ...

  MyEffectsFile := TEffectFile.Create(MyJoystick);
  with MyEffectsFile do
  begin
    // Objekte (nicht: Effekte) anlegen, Namen in EffectNames[]
    LoadFromFile('effects.ifr');
    ...
    // Effekt 'Jump' anlegen, Richtung auf 90 Grad, abspielen
    with LoadEffect('Jump') as TDIFFSingleEffect do
    begin
      Direction := 90;
      Play(0);  // <- implizite Modifikation und Download
    end;
    ...
    UnloadEffect('Jump');  // Gertespeicher freigeben

    // Verbundeffekt 'DoubleDrill' anlegen und spielen,
    // alle anderen Effekte aus
    LoadEffect('DoubleDrill').Play(DIES_SOLO)
    ...
    UnloadEffect('DoubleDrill');

    // Effekt 'Fire Bow' laden, auf Button1 legen
    MyEffect := LoadEffect('Fire Bow') as TDIFFSingleEffect;
    with MyEffect do
    begin
      FireButton := DIJOFS_BUTTON1;
    end;


  Wertebereiche
  -------------
  Direction: 0..360, Gain: 0..100; Duration: usecs

  Erweiterungen gegenber Immersions IFRHelper & Demo ReadIFR
  -----------------------------------------------------------
  - IFR-Daten werden nur einmal (nicht einmal pro Effekt) geladen
  - spielt Verbundeffekte richtig ab
  - SOLO-Effekte (auch fr Verbundeffekte)
  - Effekt-Download nur nur nach echten Vernderungen
  - Trigger-Button (leider nicht fr Verbundeffekte)

  Voraussetzungen zur Laufzeit
  ----------------------------
  a) IForce2.dll - von www.immersion.com oder www.force-feedback.com
  b) DirectX 5

  Voraussetzungen fr die Compilierung
  ------------------------------------
  a) DirectX-Header von Erik Unger - www.bigfoot.com/~ungerik/
  b) IForce2.pas - Header mit Deklarationen von IForce2.dll
  c) Delphi 3 oder 4. VB bringt's einfach nicht :-)

  Grenzen
  -------
  Die Speicherung von Parametern bernimmt IForce2.DLL. Einzel-
  und Verbundeffekte knnen deshalb nicht ohne eine Effektdatei
  existieren, beim Unload eines Effekts gehen Parameternderungen
  verloren. Effektdateien lassen sich nicht zur Laufzeit um
  neue Einzel- oder Verbundeffekte erweitern.

  Happy punching!
}

interface
uses Windows, Classes, SysUtils, DInput, // Erik Ungers Header
     Dialogs, IForce2;  // = IForce2.h von Immersion

const MAX_AXES = 32;

type
  TDIFFEffectFile = class;  // forward

  TAxisArray = Array[0..MAX_AXES-1] of Integer;

  // Abstrakte Basisklasse fr Einzel- und Verbundeffekte.
  TDIFFEffect = class(TObject)
    private
      FFXName: String;
    public
      constructor Create(FXName: String);
      function Start(dwFlags: Cardinal): HResult; virtual; abstract;
      function Stop: HResult; virtual; abstract;
      property FXName: String read FFXName;
    end;

  // Einzelner Kraftrckmeldungseffekt
  TDIFFSingleEffect = class(TDIFFEffect)
    private
      ChangeFlags: Cardinal;  // DIEP_DIRECTION or DIEP_...
      FDIEffect: TDIEffect;
      FDirections: TAxisArray; // im Moment immer 2 Elemente
      FAxes: TAxisArray;       // ditto
      FEnvelope: TDIEnvelope;
    protected
      FEffectObject: PIDirectInputEffect;
      FOwner: TDIFFEffectFile;
    protected
      procedure ResetDIEffectStruct;
      procedure LoadEffect(EffObject: PIDirectInputEffect);
      procedure UnloadEffect;
      function ApplyChanges: HResult;
      function GetLoaded: Boolean;
      function GetEffectObject: IDirectInputEffect;
    protected
      function GetDuration: Cardinal;
      procedure SetDuration(Value: Cardinal);
      function GetDirection: Cardinal;
      procedure SetDirection(Value: Cardinal);
      function GetGain: Integer;
      procedure SetGain(Value: Integer);
      function GetTriggerButton: Cardinal;
      procedure SetTriggerButton(Value: Cardinal);
      function GetTypeSpcParamSize: Cardinal;
      function GetTypeSpcParams: Pointer;
    public  // nicht direkt aufrufen, auch nicht ber Free!
      destructor Destroy; override;
    public
      property TriggerButton: Cardinal read GetTriggerButton write SetTriggerButton;
      property Duration: Cardinal read GetDuration write SetDuration;
      property Gain: Integer read GetGain write SetGain;
      property Direction: Cardinal read GetDirection write SetDirection;
      property TypeSpcParamSize: Cardinal read GetTypeSpcParamSize;
      property TypeSpcParams: Pointer read GetTypeSpcParams;
      procedure SetTypeSpcParams(SpcSize: Cardinal; Data: Pointer);
      property EffectObject: IDirectInputEffect read GetEffectObject;
      property Loaded: Boolean read GetLoaded;
    public
      function Start(dwFlags: Cardinal): HResult; override;
      function Stop: HResult; override;
  end;

  // Verbundeffekt
  TDIFFCompoundEffect = class(TDIFFEffect)
    private
      FCompounds: TList;
    protected
      function GetCompoundCount: Integer;
      function GetCompounds(Index: Integer): TDIFFSingleEffect;
      procedure AddCompound(Effect: TDIFFSingleEffect);
    public
      constructor Create(FXName: String);
      destructor Destroy; override;
      // Einzeleffekte, aus denen der Verbund besteht
      property CompoundCount: Integer read GetCompoundCount;
      property Compounds[Index: Integer]: TDIFFSingleEffect read GetCompounds;
      // Start und Stop eines Verbundeffekts (d.h. simultaner
      // Start/Stop der Einzeleffekte)
      function Start(dwFlags: Cardinal): HResult; override;
      function Stop: HResult; override;
  end;

  // Ldt eine IFR-Datei, legt Einzel- und Verbundeffekte an
  TDIFFEffectFile = class(TObject)
    private
      FHandle: HIForceProject;
      Device: IDirectInputDevice2;
      FEffects: TStringList;  // Objects: TDIFFEffect
    protected
      procedure Cleanup;  // XRef: LoadFromXXX, Destroy
    public
      constructor Create(IFDevice: IDirectInputDevice2);
      destructor Destroy; override;
      function LoadFromFile(FName: String): Boolean;
      function LoadFromStream(S: TStream): Boolean;
    public
      property EffectNames: TStringList read FEffects;
      function LoadEffect(FXName: String): TDIFFEffect;
      procedure UnloadEffect(FXName: String);
    end;

implementation

constructor TDIFFEffect.Create(FXName: String);
begin
  inherited Create; FFXName := FXName;
end;

function TDIFFSingleEffect.GetLoaded: Boolean;
begin
  Result := FEffectObject <> nil;
end;


function TDIFFSingleEffect.GetEffectObject: IDirectInputEffect;
begin
  Result := FEffectObject^;
end;

// XRef: TDIFFEffectFile.LoadFromStream nach Anlegen einer
// IDirectInputEffect-Schnittstelle durch IFORCE2.DLL.
procedure TDIFFSingleEffect.LoadEffect(EffObject: PIDirectInputEffect);
var NewSpcSize: Cardinal; Res: HResult;
begin
  FEffectObject := EffObject;
  ResetDIEffectStruct;
  Res := EffectObject.GetParameters(FDIEffect,DIEP_ALLPARAMS);
  if Res = DIERR_MOREDATA then
  begin  // (Kommentar aus IFRHelper.cpp)
  // if cbTypeSpecificParams is incorrect or lpvTypeSpecificParams is NULL,
  // DirectInput returns DIERR_MOREDATA and fills cbTypeSpecificParams with
  // the correct value.  Allocate memory for lpvTypeSpecificParams and call
  // GetParameters again.
  // Note that GetParameters will overwrite some DIEFFECT struct member data,
  // such as dwFlags and cAxes.  See DI documentation for more details.
    NewSpcSize := TypeSpcParamSize;
    ResetDIEffectStruct;
    SetTypeSpcParams(NewSpcSize,nil);  // just allocate, don't copy
    Res := EffectObject.GetParameters(FDIEffect,DIEP_ALLPARAMS);
  end;
  if FAILED(Res) then raise Exception.Create('SingleEffect.LoadEffect: '+ErrorString(Res));
  ChangeFlags := 0;  // not changes to this data (yet)
end;

procedure TDIFFSingleEffect.UnloadEffect;
begin
  SetTypeSpcParams(0,nil);  // = FreeMem(lpvSpcTypeSpecificParams)
  FEffectObject := nil;
end;

// (Re-)init der DIEffect-Struktur. XRef: LoadEffect (2 x)
procedure TDIFFSingleEffect.ResetDIEffectStruct;
begin
  SetTypeSpcParams(0,nil);  // = Free
  // (Kommentar aus IFRHelper.cpp)
  // DI requires certain struct members to have non-zero data
  // before calling GetParameters. See DI documentation for more details
  with FDIEffect do
  begin
    dwSize := SizeOf(FDIEffect);
    dwFlags := DIEFF_SPHERICAL or DIEFF_POLAR or DIEFF_CARTESIAN or DIEFF_OBJECTOFFSETS;
    cAxes := MAX_AXES;
    rglDirection := @FDirections; rgdwAxes := @FAxes; lpEnvelope := @FEnvelope;
  end;
  FEnvelope.dwSize := SizeOf(FEnvelope);
end;

destructor TDIFFSingleEffect.Destroy;
begin
  UnloadEffect;
  inherited;
end;

function TDIFFSingleEffect.Stop: HResult;
begin
  Result := EffectObject.Stop;
end;

// ------ Get- un Set-Methoden, reichlich langweilig  ----------
function TDIFFSingleEffect.GetDuration: Cardinal;
begin
  Result := FDIEffect.dwDuration;
end;

procedure TDIFFSingleEffect.SetDuration(Value: Cardinal);
begin
  FDIEffect.dwDuration := Value;
  ChangeFlags := ChangeFlags or DIEP_DURATION;
end;

function TDIFFSingleEffect.GetDirection: Cardinal;
begin
  Result := FDirections[0] div DI_DEGREES;
end;

procedure TDIFFSingleEffect.SetDirection(Value: Cardinal);
begin
  FDirections[0] := Value * DI_DEGREES;
  ChangeFlags := ChangeFlags or DIEP_DIRECTION;
end;

function TDIFFSingleEffect.GetGain: Integer;
begin
  Result := FDIEffect.dwGain div 100;
end;

procedure TDIFFSingleEffect.SetGain(Value: Integer);
begin
  FDIEffect.dwGain := Value * 100;
  ChangeFlags := ChangeFlags or DIEP_GAIN;
end;

function TDIFFSingleEffect.GetTriggerButton: Cardinal;
begin
  Result := FDIEffect.dwTriggerButton;
end;

procedure TDIFFSingleEffect.SetTriggerButton(Value: Cardinal);
begin
  FDIEffect.dwTriggerButton := Value;
  ChangeFlags := ChangeFlags or DIEP_TRIGGERBUTTON;
end;

function TDIFFSingleEffect.GetTypeSpcParamSize: Cardinal;
begin
  Result := FDIEffect.cbTypeSpecificParams;
end;

function TDIFFSingleEffect.GetTypeSpcParams: Pointer;
begin
  Result := FDIEffect.lpvTypeSpecificParams;
end;

// Puffergre neu setzen und Daten kopieren (falls Data <> nil)
// XRefs: LoadEffect, ResetDIEffectStruct, Destroy (mit Data = nil)
procedure TDIFFSingleEffect.SetTypeSpcParams(SpcSize: Cardinal; Data: Pointer);
begin
  if TypeSpcParamSize <> SpcSize then
  with FDIEffect do
  begin
    if cbTypeSpecificParams <> 0 then FreeMem(lpvTypeSpecificParams);
    cbTypeSpecificParams := SpcSize;
    if SpcSize <> 0 then GetMem(lpvTypeSpecificParams,cbTypeSpecificParams)
      else lpvTypeSpecificParams := nil;
  end;
  if Data <> nil then Move(Data^,TypeSpcParams^,TypeSpcParamSize);
  ChangeFlags := ChangeFlags or DIEP_TYPESPECIFICPARAMS;
end;

function TDIFFSingleEffect.ApplyChanges: HResult;
begin
  if ChangeFlags = 0 then Result := DI_OK
  else
  begin
    Result := EffectObject.SetParameters(FDIEffect,ChangeFlags);
    if SUCCEEDED(Result) then ChangeFlags := 0;
  end;
end;

// Effekt (nach eventueller Modifikation & Download) starten
function TDIFFSingleEffect.Start(dwFlags: Cardinal): HResult;
begin
  Result := ApplyChanges;
  if SUCCEEDED(Result) then
    Result := EffectObject.Start(1,dwFlags);
end;

// -------- Verbundeffekte -------------
constructor TDIFFCompoundEffect.Create(FXName: String);
begin
  inherited Create(FXName);
  // Zeiger auf die Einzeleffekte (Delphi-Objekte)
  FCompounds := TList.Create;
end;

procedure TDIFFCompoundEffect.AddCompound(Effect: TDIFFSingleEffect);
begin
  FCompounds.Add(Effect);
end;

destructor TDIFFCompoundEffect.Destroy;
begin
  FCompounds.Free;  // Datenstrukturen von IForce2.DLL
  inherited;        // bleiben unverndert
end;

function TDIFFCompoundEffect.GetCompoundCount: Integer;
begin
  Result := FCompounds.Count;
end;

function TDIFFCompoundEffect.GetCompounds(Index: Integer): TDIFFSingleEffect;
begin
  Result := FCompounds[Index];
end;

// Mehrere Einzeleffekte zusammen starten
function TDIFFCompoundEffect.Start(dwFlags: Cardinal): HResult;
var x: Integer;
begin
  Result := 0;
  for x := 0 to CompoundCount-1 do
  begin
    Result := Compounds[x].Start(dwFlags);
    if FAILED(Result) then Break;
    // Wenn dwFlags DIES_SOLO enthlt, raus damit. Ansonsten wrde
    // der Start des 2. Effekts den 1. wieder anhalten usw.
    dwFlags := dwFlags and not DIES_SOLO;
  end;
end;

// Mehrere Einzeleffekte zusammen anhalten
function TDIFFCompoundEffect.Stop: HResult;
var x: Integer; TempRes: HResult;
begin
  Result := DI_OK;
  for x := 0 to CompoundCount-1 do
  begin
    TempRes := Compounds[x].Stop;
    if Result = DI_OK            // bei Problemen 1. Fehlercode
      then Result := TempRes;    // als Ergebnis liefern
  end;
end;


// ------ TDIEffectFile -------------------------------------------
constructor TDIFFEffectFile.Create(IFDevice: IDirectInputDevice2);
begin
  inherited Create;
  // IFORCE2.DLL braucht das Gert zum Anlegen der Effekte
  Device := IFDevice;
  FEffects := TStringList.Create;
end;

procedure TDIFFEffectFile.Cleanup;
var x: Integer;
begin
  // Bugfix: SideWinder & WingMan verlieren sonst Speicher!
  Device.Unacquire;
  for x := 0 to FEffects.Count-1 do
  begin
    UnloadEffect(FEffects[x]);
    FEffects.Objects[x].Free;
  end;
  FEffects.Clear;
  if FHandle <> nil then IFReleaseProject(FHandle);  // IFORCE2.DLL
end;

destructor TDIFFEffectFile.Destroy;
begin
  Cleanup;
  inherited;
end;

function TDIFFEffectFile.LoadFromFile(FName: String): Boolean;
var S: TFileStream;
begin
  if Pos('.',ExtractFileName(FName)) = 0 then FName := FName+'.ifr';
  S := TFileStream.Create(FName,fmOpenRead or fmShareDenyWrite);
  Result := LoadFromStream(S);
  S.Free;
end;

function EqualGUID(const G1, G2: TGUID): Boolean;
begin
  Result := CompareMem(@G1,@G2,SizeOf(TGUID));
end;

{ IFR-Dateien enthalten im Wesentlichen eine Liste benannter
  Effekte und benannter Eigenschaften; jedem Parametersatz
  geht eine Lngenangabe voraus. Fr Einzeleffekte liefert
  IFCreateEffects (in IFORCE2.DLL) einen einzelnen IDirectInput-
  Effect-Zeiger, bei Verbundeffekten kme ein Zeigerarray heraus.
  Nach einer fehlerfreien Ladeaktion enthlt FEffects die Namen
  aller Effekte (einzeln und Verbund), FEffects.Objects enthlt
  Zeiger auf TDIEFFSingleEffect-Objekte fr Einzeleffekte
  und TDIEFFCompoundEffect-Objekte fr Verbundeffekte. }
function TDIFFEffectFile.LoadFromStream(S: TStream): Boolean;
var M: TMemoryStream; ObjSize: Cardinal;
    PC: PChar; x,y: Integer;
    NewCompound: TDIFFCompoundEffect;
    GUIDs, SubGUIDs: TStringList; SubGUID: String;
begin
  Cleanup;  // guess...
  Result := Device <> nil;
  if not Result then Exit;
  M := TMemoryStream.Create; M.CopyFrom(S,0); M.Seek(0,0);
  GUIDs := TStringList.Create; SubGUIDs := TStringList.Create;
  try
    FHandle := IFLoadProjectPointer(M.Memory,Device);
    Result := FHandle <> nil; if not Result then Exit;
    M.Seek(8,soFromBeginning);  // Signatur-Check: IFLoadProject
    while M.Position < M.Size do
    begin  // Effektnamen einsammeln, Daten berspringen
      M.Read(ObjSize,4);
      PC := M.Memory; Inc(PC,M.Position); FEffects.Add(StrPas(PC));
      // Unterscheidung zwischen Einzel- und Verbundeffekten
      SubGUID := '';
      Inc(PC,Strlen(PC)+1);  // ans Namensende
      if PC = 'Compound' then
      begin
        Inc(PC,Strlen(PC)+6);
        if PC = 'ContainedObjects' then
        begin  // GUIDs der Einzeleffekte: {...};{...};#0
          Inc(PC,StrLen(PC)+1); SubGUID := PC; Inc(PC,StrLen(PC)+1);
        end;
      end else
      begin  // Einzeleffekt
        Inc(PC,StrLen(PC)+6);
      end;
      SubGUIDs.Add(SubGUID);  // '' oder {GUID};{GUID}...
      if PC <> 'ID' then raise Exception.Create('No Effect ID');
      Inc(PC,StrLen(PC)+1);
      GUIDs.Add(PC); // GUID des Effekts oder des Verbunds
      M.Seek(ObjSize-4,soFromCurrent);
    end;

    // Einzeleffekt-Objekte (keine IDirectInputEffect-Zeiger)
    for x := 0 to FEffects.Count-1 do
      if SubGUIDs[x] = '' then
       FEffects.Objects[x] := TDIFFSingleEffect.Create(FEffects[x]);

    // Verbundobjekte enhalten nur eine Liste mit Zeigern
    // auf die Einzeleffekte
    for x := 0 to FEffects.Count-1 do
      if SubGUIDs[x] <> '' then
      begin
        NewCompound := TDIFFCompoundEffect.Create(FEffects[x]);
        FEffects.Objects[x] := NewCompound;
        for y := 0 to GUIDs.Count-1 do
          if Pos(GUIDs[y], SubGUIDs[x]) <> 0 then
            NewCompound.AddCompound(FEffects.Objects[y]
               as TDIFFSingleEffect);
      end;
  finally
    M.Free; GUIDs.Free; SubGUIDs.Free;
  end;
end;

function TDIFFEffectFile.LoadEffect(FXName: String): TDIFFEffect;
var x, FXIndex: Integer;
  procedure LoadSingleEffect(FX: TDIFFSingleEffect);
  var PIEffect: PIDirectInputEffect; NumEffects: Cardinal;
  begin
    with FX do
      if not Loaded then
      begin
        PIEffect := IFCreateEffects(FHandle,PChar(FXName),NumEffects);
        if (PIEffect = nil) or (NumEffects = 0) then
          raise Exception.Create('Unable to load FFEffect '+FXName);
        LoadEffect(PIEffect);
      end;
  end;

begin
  FXIndex := EffectNames.IndexOf(FXName);
  if FXIndex = -1 then
    raise Exception.Create('EffectFile: no effect named '+FXName);
  Result := FEffects.Objects[FXIndex] as TDIFFEffect;
  if Result is TDIFFSingleEffect
    then LoadSingleEffect(Result as TDIFFSingleEffect)
    else with Result as TDIFFCompoundEffect do
      for x := 0 to CompoundCount-1 do
        LoadSingleEffect(Compounds[x]);
end;

procedure TDIFFEffectFile.UnloadEffect(FXName: String);
var x, FXIndex: Integer; CurFX: TDIFFEffect;
  procedure UnloadSingleEffect(FX: TDIFFSingleEffect);
  begin
    with FX do if Loaded then
    begin
      IFReleaseEffects(FHandle, FEffectObject);
      UnloadEffect;
    end;
  end;
begin
  FXIndex := EffectNames.IndexOf(FXName);
  if FXIndex = -1 then
    raise Exception.Create('EffectFile: no effect named '+FXName);
  CurFX := FEffects.Objects[FXIndex] as TDIFFEffect;
  if CurFX is TDIFFSingleEffect
    then UnloadSingleEffect(CurFX as TDIFFSingleEffect)
  else with CurFX as TDIFFCompoundEffect do
    for x := 0 to CompoundCount-1 do
      UnloadSingleEffect(Compounds[x]);
end;

end.

