unit DXDrawAS; //  Original 18-OCT-98 as (Arne Schpers)
{ 26-FEB-99: Umstellung auf DirectX 6. nderungen:

  Globale Variable DirectDraw: IDirectDraw2 -> IDirectDraw4
   (+ Anpassung der Routine DirectDrawInitialize)
  IDirectDrawSurface3 -> Surface4 (global)
  TDDSurfaceDesc -> TDDSurfaceDesc2 (global)
  Methode SurfaceFromSurfDesc: QueryInterface entfllt

  Erweiterungen: Methode DrawClipped

  03-APR-99: Cardinal/Integerzeugs wg. D4, sonst nichts  
}

{ Delphi-Komponenten fr DirectDraw; baut auf den von Erik Unger
  umgesetzten Interface-Deklarationen fr DirectX auf.

  Pro Anwendung: *ein* IDirectDraw-Objekt (globale Variable
  DirectDraw, ber DirectDrawInitialize besetzt), beliebige
  Zahl von TDDSurface-Komponenten.

Der dritte Parameter von DirectDrawInitialize legt fest, ob
die Komponenten mit (natrlich langsamen) Kopien arbeiten
oder die Oberflchen direkt sperren, der vierte aktiviert die Sperr-
prfung ber Application.OnIdle. In der Entwicklungsphase
DirectDrawInitialize grundstzlich mit (...., True, True) aufrufen!

Besonderheiten im Handling von TDDSurface:
- Zweistufiges Create: erst die Delphi-Komponente anlegen, dann
  entweder SurfDesc ausfllen und SurfaceFromSurfDesc aufrufen
  oder SurfaceFromGDIBitmap mit einem Bitmap benutzen.
  Destroy ist einstufig.
- LockBits, UnlockBits und Canvas/Canvas.ReleaseDC anstelle von
  Lock/Unlock und GetDC/ReleaseDC des Oberflchen-Objekts
  verwenden. Im Modus "UseSafeCopy" (= DirectDrawInitialize
  mit TRUE als letztem Parameter) erzeugen diese Routinen lokale
  Kopien bzw. DCs.
- Eigenschaft SurfaceObject (= IDirectDrawSurface4), ber die
  man an die restlichen zig Funktionen herankommt.

Grenzen (nur im SafeCopy-Modus):
- SafeCopy-Modus ist bei 256 Farben sagenhaft langsam (Faktor 40)
- Elsa Erazor I braucht bei Oberflchen im Bildspeicher
  eine Zwischenkopie (VIDEOMEMORY -> SYSTEMMEMORY -> Bitmap);
  ggf. ber VMemBltIndirect = FALSE abschalten.
}

interface
{$UNDEF DIRECTX5}  // sonst werden DX6-Strukturtypen verwendet!
uses
  Windows, SysUtils, Classes, ActiveX, Dialogs, Forms,
  Graphics,
  DDraw;       // Erik Ungers Umsetzung fr DX5, DX6

// SafeCopy-Modus: Bildspeicher-Oberflchen nicht direkt
// in ein Bitmap kopieren, sondern erstmal in den Hauptspeicher
// (umgeht ein Problem mit der Elsa Erazor I)

const VMemBltIndirect: Boolean = True;

type
  TDDCanvas = class;

  TDDSurface = class(TComponent)
  private
    SafeBitBufferSize: Cardinal; // Bits der Oberflche im
    SafeBitBuffer: Pointer;  // Modus "UseSafeCopy"
    FCanvas: TDDCanvas;  // wie fr normale GDI-Bitmaps
    FClipWnd: HWnd;  // Clipper-Zielfenster (nur PrimarySurf)
    FLocked: Boolean;  // True: Canvas oder LockBits
  protected
    procedure SetClipWnd(Value: HWnd);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  public
    SurfaceObject: IDirectDrawSurface4;  // DirectDraw-Objekt
    SurfDesc: TDDSurfaceDesc2;  // und seine Beschreibung
    procedure SurfaceFromSurfDesc;  // vorher SurfDesc setzen
    procedure SurfaceFromGDIBitmap(BMP: TBitmap;
        MemoryLocation: DWord);  // BMP->SurfDesc->Oberflche
  public  // nur der Schreibfaulheit halber
    property Width: DWORD read SurfDesc.dwWidth;
    property Height: DWORD read SurfDesc.dwHeight;
    property Pitch: Integer read SurfDesc.lPitch;
  public  // Zugriffs- und Zeichenfunktionen
    BltFlags: DWORD;  // Flags fr BltFX
    BltFX: TDDBltFX;  // Effekte bei Blt
    // fr primre Oberflchen: Wenn 0, Koordinaten-Umsetzung
    property ClipperWnd: HWnd read FClipWnd write SetClipWnd;
    function LockBits: Pointer;
    procedure UnlockBits(WriteBack: Boolean);
    procedure Draw(XPos,YPos: Integer; Source: TDDSurface);
    procedure DrawClipped(XPos, YPos: Integer; Source: TDDSurface);
    property Canvas: TDDCanvas read FCanvas;
    // True nach Canvas.CreateHandle und LockBits
    property Locked: Boolean read FLocked;
  end;

  // Unterschiede zum Canvas eines GDI-Bitmaps:
  // 1. Im Modus "UseSafeCopy": Bitmap-Kopie der Oberflche,
  //    der DC ist dann SafeBitmap.Canvas.Handle
  // 2. Nach Verwendung *unbedingt* ReleaseDC aufrufen!
  TDDCanvas = class(TCanvas)
   private
     FOwner: TDDSurface;
     FHandle: HDC;  // NOT UseSafeCopy: SurfaceObject.GetDC
     SafeBitmap: TBitmap;  // NOT UseSafeCopy: unbenutzt
   protected  // nur innerhalb dieser Unit erzeugbar....
     constructor Create(Owner: TDDSurface);
{$HINTS OFF} // "Destroy has lower visibility than TCanvas.Destroy"
     destructor Destroy; override;
     function GetHandle: HDC;
     procedure CreateHandle; override;
   public
     property Handle: HDC read GetHandle;
     procedure ReleaseDC;
   end;
{$HINTS ON}

// Initialisierung, Deinitialisierung (optional)
procedure DirectDrawInitialize(lpGUID: PGUID;  // nil=primary
  CoopLevel: Integer; Wnd: HWnd;  // Hauptfenster Programm
  _UseSafeCopy, _IdleCheck: Boolean);  // 2*TRUE fr Experimente!
procedure DirectDrawUninitialize;

// Besetzt DirectDrawPalette, verpat jeder existierenden
// und zuknftigen Oberflche diese Palette
procedure DirectDrawCreateGlobalPalette(GDIPal: HPalette);
// das macht DirectDrawUninitialize automatisch
procedure DirectDrawDeleteGlobalPalette;

// Sozusagen ein Makro. Wenn ClipperWindow <> 0 -> CreateClipper
function CreatePrimarySurface(ClipperWindow: HWnd): TDDSurface;

// ----------- Fehlerprfungen, Exceptions ------------------
// Direktaufruf der OnIdle-Prfung, fr DXTimer-Aktionen
procedure DDIdleCheck;

// Prft ResVal und sorgt vor der Ausgabe von Fehlermeldungen
// fr klar Schiff (= Oberflchen und GDI entsperren)
procedure DDCheck(Msg: String; ResVal: Integer);

// Dito fr selbst erzeugte Fehlermeldungen
procedure DDError(Msg: String);

type // von DDCheck und DDError erzeugt
  DDrawException = class(Exception);

// -------- Globale Variablen -------------------
// ber DirectDrawInitialize gesetzt: Das DirectDrawObjekt
var DirectDraw: IDirectDraw4;
// ber CreatePrimarySurface (mit ClipperWindow <> 0) gesetzt
var DirectDrawClipper: IDirectDrawClipper;
// XRef: Create/DeleteGlobalPalette. Wenn <> nil, dann kriegt
// jede Oberflche diese Palette zugeordnet
var DirectDrawPalette: IDirectDrawPalette;

implementation
// Bei DirectDrawInitialize gesetzt. Bei UseSafeCopy = True arbeiten
// TDDSurface.LockBits und UnlockBits sowie TDDSurface.Canvas
// mit kopierten Daten, die Oberflchen (und die GDI) werden nur
// kurzfristig gesperrt. Bei False wird mit den Originaldaten
// gearbeitet - Breakpoints im Debugger, MessageBoxes, Exceptions
// und so weiter bringen Win9x (nicht aber NT 5.0) zum vlligen
// Stillstand.
// Bei IdleCheck = True wird vor dem Rcksprung zu Windows (auch ohne
// SafeCopy) geprft, ob alle Sperren beseitigt sind.
// Bei Experimenten *jeder* Art DirectDrawInitialize
// mit (..., True,True) aufrufen!
var UseSafeCopy: Boolean;  // True: Sicherungskopien
var IdleCheck: Boolean;  // True: Check ber Application.OnIdle

type
  // Verwaltungsklasse um das DirectDraw-Objekt herum
  TDDraw = class(TComponent)
    public
     Surfaces: TList;    // alle TDDSurface-Objekte
     PaletteBMP: TBitmap;
     constructor Create(AOwner: TComponent); override;
     destructor Destroy; override;
     procedure CreatePalette(GDIPal: HPalette);
     procedure DeletePalette;
     procedure DoIdleCheck(Sender: TObject; var Done: Boolean);
  end;

// --- Das (einzige) Objekt der Klasse TDDraw --------
var DelphiDrawObject: TDDraw;

constructor TDDraw.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Surfaces := TList.Create;
end;

// rumt alle Oberflchen, den Clipper und DirectDraw ab
destructor TDDraw.Destroy;
var x: Integer;
begin
  DeletePalette;
  for x := Surfaces.Count-1 downto 0 do
    TDDSurface(Surfaces[x]).Free;
  Surfaces.Free;

  DirectDrawClipper := nil;
  DirectDraw := nil;  // IDirectDraw-Schnittstelle raus
  inherited Destroy;
end;

procedure TDDraw.CreatePalette(GDIPal: HPalette);
var LogPal: TMaxLogPalette; x: Integer;
begin
  with LogPal do
  begin
    palVersion := $0300; palNumEntries := 256;
    GetPaletteEntries(GDIPal,0,256,palPalEntry[0]);
  end;
  DDCheck('DirectDrawCreateGlobalPalette',
    DirectDraw.CreatePalette( DDPCAPS_8BIT,
      @LogPal.palPalEntry[0],DirectDrawPalette, nil));
  // allen Oberflchen diese Palette verpassen
  for x := 0 to Surfaces.Count-1 do
    with TDDSurface(Surfaces[x]) do
      SurfaceObject.SetPalette(DirectDrawPalette);
  if UseSafeCopy then
  begin
    PaletteBMP := TBitmap.Create; PaletteBMP.HandleType := bmDDB;
    PaletteBMP.Width := 1; PaletteBMP.Height := 1;
    PaletteBMP.Palette := GDIPal;
  end;
end;

procedure TDDraw.DeletePalette;
var x: Integer;
begin
  for x := 0 to Surfaces.Count-1 do
    with TDDSurface(Surfaces[x]) do
      if SurfaceObject <> nil then SurfaceObject.SetPalette(nil);
  DirectDrawPalette := nil;
  PaletteBMP.Free; PaletteBMP := nil;
end;

// = Application.OnIdle: Prfung vor dem Rcksprung zu Windows
procedure TDDraw.DoIdleCheck(Sender: TObject; var Done: Boolean);
var x: Integer;
begin
  Done := True;
  for x := 0 to Surfaces.Count-1 do
    if TDDSurface(Surfaces[x]).Locked
      then DDError('IdleCheck: Locked Surface(s)!');
end;

procedure DDIdleCheck;  // Direktaufruf
var Dummy: Boolean;
begin
  if DelphiDrawObject <> nil then
    DelphiDrawObject.DoIdleCheck(nil,Dummy);
end;

procedure DDError(Msg: String);
begin  // fr selbsterzeugte Fehlermeldungen
  DDCheck(Msg,-1);
end;

procedure DDCheck(Msg: String; ResVal: Integer);
var x: Integer;
const
  CallCount: Integer = 0;  // Rekursionsbremse
  Uninitializing: Boolean = False;  // Last resort, sozusagen
begin
  if SUCCEEDED(ResVal) then Exit;
  if CallCount > 1 then
  begin // Fehler whrend der Aufrumarbeiten: Notausstieg
    if not Uninitializing then
    begin
      Uninitializing := True;
      DirectDrawUninitialize;
    end
      else Exit;  // (Folge-)Fehler bei der Deinitialisierung
  end else
  begin  // alle GDI-Blockierer (DCs und Locks) raus
    Inc(CallCount);
    for x := 0 to DelphiDrawObject.Surfaces.Count-1 do
      with TDDSurface(DelphiDrawObject.Surfaces[x]) do
      try
        if SurfaceObject <> nil then
          if SurfDesc.lpSurface <> nil then UnlockBits(False)
           else if Canvas.FHandle <> 0 then Canvas.ReleaseDC;
        if CallCount > 1 then Break;
      except
      end;
    Dec(CallCount);
  end;
  // jetzt sollte die GDI in (fast) jedem Fall wieder reagieren
  if Uninitializing then Msg := Msg + '(DDraw uninitialized)';
  Uninitializing := False;
  if ResVal = -1 then raise DDrawException.Create(Msg)
   else
     raise DDrawException.Create(Msg+': '+ErrorString(ResVal));
end;

procedure DirectDrawInitialize(lpGUID: PGUID;  // nil=primary
  CoopLevel: Integer; Wnd: HWnd; _UseSafeCopy, _IdleCheck: Boolean);
var DDraw1: IDirectDraw;
begin
  UseSafeCopy := _UseSafeCopy;  // fr alle DDraw-Objekte
  IdleCheck := _IdleCheck; // nur frs Destroy
  DelphiDrawObject := TDDraw.Create(nil);
  if IdleCheck then Application.OnIdle := DelphiDrawObject.DoIdleCheck;

  // Anlegen des DirectDraw-Objekts fr das Gert (nil = primr)
  // Originalversion der DirectDraw-Schnittstelle
  DDCheck('DirectDrawCreate',
    DirectDrawCreate(lpGUID, DDraw1, nil));
{$IFDEF DIRECTX5}
  // Abfrage dieses Objekts nach IDirectDraw2
  DDCheck('DirectDraw.QueryInterface for DirectDraw2',
     DDraw1.QueryInterface(IID_IDirectDraw2,DirectDraw));
{$ELSE}
  // Abfrage dieses Objekts nach IDirectDraw4
  DDCheck('DirectDraw.QueryInterface for DirectDraw4',
     DDraw1.QueryInterface(IID_IDirectDraw4,DirectDraw));
{$ENDIF}

  // Festlegen der Kooperationsebene mit dem Handle der Form
  DDCheck('SetCooperativeLevel',
    DirectDraw.SetCooperativeLevel(Wnd, CoopLevel));
end;

procedure DirectDrawUninitialize;  // XRef: finalization
begin
  if IdleCheck then Application.OnIdle := nil;
  if DelphiDrawObject <> nil then DelphiDrawObject.Free;
  DelphiDrawObject := nil;
end;

procedure DirectDrawCreateGlobalPalette(GDIPal: HPalette);
begin
  if DelphiDrawObject = nil then
    DDError('CreateGlobalPalette: need DirectDrawInitialize');
  DelphiDrawObject.CreatePalette(GDIPal);
end;

procedure DirectDrawDeleteGlobalPalette;
begin
  if DelphiDrawObject <> nil then
    DelphiDrawObject.DeletePalette;
  DirectDrawPalette := nil;
end;

// Makro, sozusagen - knnte man auch zu Fu erledigen
function CreatePrimarySurface(ClipperWindow: HWnd): TDDSurface;
begin
  Result := TDDSurface.Create(DelphiDrawObject);
  with Result.SurfDesc do
  begin
    dwFlags := DDSD_CAPS;  // heit: Feld dwCaps ist besetzt
    ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE;
  end;
  Result.SurfaceFromSurfDesc;  // Oberflche anlegen
  Result.ClipperWnd := ClipperWindow;
end;

// ----------- Methoden von TDDSurface ------------------------
constructor TDDSurface.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  if DelphiDrawObject = nil then
    DDError('TDDSurface.Create: No DirectDraw object');
  FCanvas := TDDCanvas.Create(Self);
  DelphiDrawObject.Surfaces.Add(Self);
  FillChar(SurfDesc,SizeOf(SurfDesc),0);
end;

destructor TDDSurface.Destroy;
begin
  if DelphiDrawObject = nil then
    DDError('TDDSurface.Destroy: No DirectDraw object');
  DelphiDrawObject.Surfaces.Remove(Self);
  if SurfaceObject <> nil then SurfaceObject.SetClipper(nil);
  SurfaceObject := nil;
  FCanvas.Free;
  inherited Destroy;
end;

procedure TDDSurface.SurfaceFromSurfDesc;
{$IFDEF DIRECTX5} var Surf1: IDirectDrawSurface; {$ENDIF}
begin
  SurfDesc.dwSize := SizeOf(SurfDesc);
{$IFDEF DIRECTX5}
  // Oberflche anlegen -> Urversion der Schnittstelle
  DDCheck('CreateSurface',
    DirectDraw.CreateSurface(SurfDesc, Surf1, nil));
  // Abfrage nach IDirectDrawSurface3
  DDCheck('Surface.QueryInterface for IDirectDrawSurface4',
    Surf1.QueryInterface(IID_IDirectDrawSurface3,SurfaceObject));
  Surf1 := nil;
{$ELSE}
  DDCheck('CreateSurface (IDirectDrawSurface4)',
    DirectDraw.CreateSurface(SurfDesc, SurfaceObject, nil));
{$ENDIF}
  // komplette Beschreibung einsetzen
  SurfaceObject.GetSurfaceDesc(SurfDesc);
  // Palette einsetzen, falls vorhanden
  if DirectDrawPalette <> nil
    then SurfaceObject.SetPalette(DirectDrawPalette);
end;

procedure TDDSurface.SurfaceFromGDIBitmap(BMP: TBitmap;
     MemoryLocation: DWord);
var SurfDC: HDC;
begin  // BMP -> SurfDesc -> CreateFromSurfDesc; BMP -> Surface
  with SurfDesc do
  begin
    // soll heien: diese Felder sind besetzt
    dwFlags := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
    // Haupt- oder Bildspeicher, AGP?
    ddsCaps.dwCaps := MemoryLocation;
    // Breite und Hhe in Pixeln
    dwWidth := BMP.Width; dwHeight := BMP.Height;
  end;
  SurfaceFromSurfDesc;
  // Gertekontext abholen; Bitmap -> Oberflche
  DDCheck('SurfaceFromGDIBitmap: Surface.GetDC',
     SurfaceObject.GetDC(SurfDC));
  with BMP do
  begin
    BitBlt(SurfDC,0,0,Width,Height, Canvas.Handle,0,0,SRCCOPY);
    DDCheck('Surface.ReleaseDC',SurfaceObject.ReleaseDC(SurfDC));
  end;
end;

procedure TDDSurface.SetClipWnd(Value: HWnd);
begin
  if Value = FClipWnd then Exit;
  if SurfaceObject = nil then DDError('SetClipWnd: no surface!');
  if SurfDesc.ddsCaps.dwCaps and DDSCAPS_PRIMARYSURFACE = 0 then
    DDError('SetClipWnd: only for primary surfaces');
  if Value = 0 then SurfaceObject.SetClipper(nil)
  else
  begin
    if DirectDrawClipper = nil then
      // Clipper ohne spezifischen Besitzer anlegen
      DDCheck('DirectDrawCreateClipper',
        DirectDrawCreateClipper(0, DirectDrawClipper, nil));
    if FClipWnd = 0 then
    // Clipper mit der primren Oberflche verbinden
    DDCheck('SetClipper',
      SurfaceObject.SetClipper(DirectDrawClipper));
    // Clipper auf das angegebene Fenster setzen
    DDCheck('DirectDrawClipper.SetHWnd',
      DirectDrawClipper.SetHWnd(0, Value));
  end;
  FClipWnd := Value;
end;

// Liefert einen direkten Zeiger auf die Bits der Oberflche
// oder eine Kopie der Bits (UseSafeCopy = True). Im ersten
// Fall bleibt die Oberflche gesperrt - und damit die GDI(!)
function TDDSurface.LockBits: Pointer;
var CalledTwice: Boolean;
begin
  if Locked then DDError('TDDSurface.LockBits: Surface locked');
  DDCheck('LockBits: Surface.Lock',
           SurfaceObject.Lock(nil, SurfDesc, DDLOCK_WAIT,0));
  FLocked := True;
  if UseSafeCopy then  // Kopie anlegen, danach entsperren
  with SurfDesc do
  begin
    CalledTwice := SafeBitBuffer <> nil;
    if CalledTwice then FreeMem(SafeBitBuffer,SafeBitBufferSize);
    SafeBitBufferSize := dwHeight*Cardinal(lPitch);  // oder Height*Pitch
    GetMem(Result,SafeBitBufferSize);
    Move(lpSurface^,Result^,SafeBitBufferSize);
    // Wenn dieses Unlock pltzlich nicht mehr funktioniert,
    // dann ist wohl endgltig Schlu
    DDCheck('LockBits: Surface.Unlock',
       SurfaceObject.Unlock(nil));
    SafeBitBuffer := Result; lpSurface := Result;
    if CalledTwice
      then DDError('LockBits: Two calls in a row');
  end
    else Result := SurfDesc.lpSurface;  // Originaldaten
end;

// Entsperren der Oberflche. UseSafeCopy = True: Kopie
// wieder freigeben, bei WriteBack = True vorher zurckschreiben
procedure TDDSurface.UnlockBits(WriteBack: Boolean);
begin
  if UseSafeCopy then
  begin
    if SafeBitBuffer = nil then
     DDError('UnlockBits: Two calls in a row');
    if WriteBack then
    begin  // Daten zurckschreiben
      DDCheck('UnlockBits (WriteBack): Surface.Lock',
         SurfaceObject.Lock(nil, SurfDesc, DDLOCK_WAIT,0));
      with SurfDesc do
      begin
        Move(SafeBitBuffer^, lpSurface^,SafeBitBufferSize);
        // wie Unlock bei LockBits (lies: hoffnungslos)
        DDCheck('UnlockBits (WriteBack): Surface.Unlock',
          SurfaceObject.Unlock(nil));
      end;
    end;
    FreeMem(SafeBitBuffer);  // Kopie freigeben
    SafeBitBuffer := nil; // gleichzeitig Flag fr "Unlocked"
  end
    else SurfaceObject.Unlock(nil);  // Originaldaten entsperren
  SurfDesc.lpSurface := nil;  // in jedem Fall
  FLocked := False;  // entsperrt
end;

procedure TDDSurface.Draw(XPos,YPos: Integer; Source: TDDSurface);
var DestRect: TRect;
begin
  if SurfaceObject = nil then
    DDError('TDDSurface.Draw: No surface object');
  if (Source = nil) or (Source.SurfaceObject = nil) then
    DDError('TDDSurface.Draw: No source/source surface object');
  if Locked or Source.Locked then
    DDError('TDDSurface.Draw: Surface locked');
  with Source do
    DestRect := Rect(XPos,YPos,XPos+Integer(Width),YPos+Integer(Height));
  // Ist das Ziel die primre Oberflche?
  if (SurfDesc.ddsCaps.dwCaps and DDSCAPS_PRIMARYSURFACE <> 0)
      and (ClipperWnd <> 0)
    then MapWindowPoints(ClipperWnd,0,DestRect,2);
  // Spezialeffekte beim Blitting: Wie vom Aufrufer gesetzt
  BltFX.dwSize := SizeOf(BltFX);
  SurfaceObject.Blt(@DestRect, Source.SurfaceObject,
    nil, DDBLT_WAIT or DDBLT_DDFX or BltFlags, @BltFX);
end;

procedure TDDSurface.DrawClipped(XPos,YPos: Integer; Source: TDDSurface);
var DestRect, SrcRect, DummyRect: TRect; PSrc: PRect;
begin
  if SurfaceObject = nil then
    DDError('TDDSurface.Draw: No surface object');
  if (Source = nil) or (Source.SurfaceObject = nil) then
    DDError('TDDSurface.Draw: No source/source surface object');
  if Locked or Source.Locked then
    DDError('TDDSurface.Draw: Surface locked');
  with Source do
    DestRect := Rect(XPos,YPos,XPos+Integer(Width),YPos+Integer(Height));
  PSrc := nil;
  // Begrenzung auf die Zieloberflche
  with Source do SrcRect := Rect(0,0,Width,Height);
  if XPos < 0 then
  begin
    Inc(SrcRect.Left,-XPos); DestRect.Left := 0; PSrc := @SrcRect;
  end;
  if YPos < 0 then
  begin
    Inc(SrcRect.Top,-YPos); DestRect.Top := 0; PSrc := @SrcRect;
  end;
  if XPos+Integer(Source.Width) > Integer(Width) then
  begin
    Dec(SrcRect.Right,XPos+Integer(Source.Width-Width)); PSrc := @SrcRect;
    Dec(DestRect.Right,XPos+Integer(Source.Width-Width));
  end;
  if YPos+Integer(Source.Height) > Integer(Height) then
  begin
    Dec(SrcRect.Bottom,YPos+Integer(Source.Height-Height)); PSrc := @SrcRect;
    Dec(DestRect.Bottom,YPos+Integer(Source.Height-Height));
  end;
  // Negative Breiten und/oder Hhen?
  if (PSrc <> nil) and not IntersectRect(DummyRect,SrcRect,DestRect) then Exit;

  // Ist das Ziel die primre Oberflche?
  if (SurfDesc.ddsCaps.dwCaps and DDSCAPS_PRIMARYSURFACE <> 0)
      and (ClipperWnd <> 0)
    then MapWindowPoints(ClipperWnd,0,DestRect,2);
  // Spezialeffekte beim Blitting: Wie vom Aufrufer gesetzt
  BltFX.dwSize := SizeOf(BltFX);
  SurfaceObject.Blt(@DestRect, Source.SurfaceObject,
    PSrc, DDBLT_WAIT or DDBLT_DDFX or BltFlags, @BltFX);
end;

constructor TDDCanvas.Create(Owner: TDDSurface);
begin
  inherited Create; FOwner := Owner;
end;

destructor TDDCanvas.Destroy;
begin
  // mit aktivem Handle gibt das sonst einen GPF in der VCL...
  if FHandle <> 0 then
    if UseSafeCopy then inherited Handle := 0
     else with FOwner do if SurfaceObject <> nil
       then SurfaceObject.ReleaseDC(FHandle);
  SafeBitmap.Free;
  if FHandle <> 0 then DDError('TDDCanvas.Destroy: DC in use!'); 
  inherited Destroy;
end;

procedure TDDCanvas.CreateHandle;
var DDrawDC: HDC; SrcSurf, TempSurf: TDDSurface;
begin
  if FOwner.SurfaceObject = nil then
    DDError('TDDCanvas.CreateHandle: No owner surface object');
  SrcSurf := FOwner; TempSurf := nil;
  if SrcSurf.Locked then
    DDError('TDDCanvas.CreateHandle: Surface locked');
  if not UseSafeCopy then
  begin
    DDCheck('TDDCanvas.CreateHandle: SurfaceObject.GetDC',
      SrcSurf.SurfaceObject.GetDC(FHandle));
      inherited Handle := FHandle;
  end else
  begin
    if SafeBitmap = nil then
    begin
      SafeBitmap := TBitmap.Create;
      SafeBitmap.HandleType := bmDDB;
     {$IFNDEF INIDIVDUALPALETTE}
      with DelphiDrawObject do
        if PaletteBMP <> nil then SafeBitmap.Assign(PaletteBMP);
     {$ELSE}
      // das mu vor dem Festhalten von Canvas.Handle sein:
      // Zuweisung an Palette -> Recreate Bitmap
      Res := SurfaceObject.GetPalette(DDPalette);
      if SUCCEEDED(Res) then
      begin
        DDPalette.GetEntries(0,0,256,@LogPal.palPalEntry[0]);
        LogPal.palVersion := $0300; LogPal.palNumEntries := 256;
        // Zuweisung an Palette braucht mehr als DOPPELT so lange
        // wie die gesamte restliche Aktion inklusive BitBlt
        SafeBitmap.Palette := CreatePalette(PLogPalette(@LogPal)^);
      end;  // else kommt dabei DDERR_NOPALETTEATTACHED raus
      {$ENDIF}
    end;
    with SrcSurf do
    begin
      SafeBitmap.Width := Width; SafeBitmap.Height := Height;
      FHandle := SafeBitmap.Canvas.Handle;
      if VMemBltIndirect and (SurfDesc.ddsCaps.dwCaps and
        DDSCAPS_VIDEOMEMORY <> 0) then
      begin
        TempSurf := TDDSurface.Create(nil);
        with TempSurf, TempSurf.SurfDesc do
        begin
          dwFlags := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
          dwWidth := SafeBitmap.Width;
          dwHeight := SafeBitmap.Height;
          // ohne OFFSCREENPLAIN geht's bei 256 Farben nicht...
          ddsCaps.dwCaps := DDSCAPS_SYSTEMMEMORY
             or DDSCAPS_OFFSCREENPLAIN;
          SurfaceFromSurfDesc;
          Draw(0,0,FOwner);
        end;
        SrcSurf := TempSurf;
      end;
    end;
    with SrcSurf do  // FOwner oder TempSurf
    begin
      DDCheck('TDDCanvas.CreateHandle: SurfaceObject.GetDC',
        SurfaceObject.GetDC(DDrawDC));
      BitBlt(FHandle,0,0,Width,Height,DDrawDC,0,0,SRCCOPY);
      DDCheck('TDDCanvas.GetDC: SurfaceObject.ReleaseDC',
        SurfaceObject.ReleaseDC(DDrawDC));
    end;
    TempSurf.Free;
  end;
  inherited Handle := FHandle;
  SrcSurf.FLocked := True;  // Gertekontext abgeholt
end;

function TDDCanvas.GetHandle: HDC;
begin // ruft ggf. CreateHandle auf. Nur neu definiert, damit
      // man Canvas.Handle nicht direkt auf 0 setzen kann
  Result := inherited Handle;
end;

// NOT UseSafeCopy: SurfaceObject.ReleaseDC
// UseSafeCopy: SafeBitmap -> Oberflche, SafeBitmap.Free
procedure TDDCanvas.ReleaseDC;
var DDrawDC: HDC;
begin
  if FOwner.SurfaceObject = nil then
    DDError('TDDCanvas.ReleaseDC: No owning surface');
  if FHandle = 0 then Exit;  // schadet nichts
  with FOwner do
    if not UseSafeCopy then
    begin
      DDCheck('TDDCanvas.ReleaseDC: SurfaceObject.ReleaseDC',
        SurfaceObject.ReleaseDC(FHandle));
    end else
    begin
      DDCheck('TDDCanvas.ReleaseDC, GetDC for writeback',
        SurfaceObject.GetDC(DDrawDC));
      // das geht bei 256 Farben leider per GetNearestColor...
      BitBlt(DDrawDC,0,0,Width,Height,FHandle,0,0,SRCCOPY);
      DDCheck('TDDCanvas.ReleaseDC, SurfaceObject.ReleaseDC',
        SurfaceObject.ReleaseDC(DDrawDC));
    end;
  FHandle := 0; inherited Handle := 0;
  FOwner.FLocked := False;
end;


initialization
finalization
  IdleCheck := False;  // kein Zugriff auf Application, bitte...
  DirectDrawUninitialize;
end.

