/**************************************************************************/
/**************************************************************************/
/**                                                                      **/
/**               TU Muenchen - Institut fuer Informatik                 **/
/**                                                                      **/
/** Program for the game of Solitair  V1.1                               **/
/**                                                                      **/
/**            Joern Eichler                                             **/
/**            Jochen Jaeger                                             **/
/**            Thomas Ludwig                                             **/
/**                                                                      **/
/** File:      symsol.c                                                  **/
/**                                                                      **/
/**************************************************************************/
/**************************************************************************/

/*
  Solitaire game tools
  
  the board is represented in 5 bytes
  byte 0,1,2,3 build the corners
  and byte 4 bit 1 is the center of the board
  
  so one corner consists of 8 bit and looks like this:
  
  0 4 2
  1 5 3
  7 6
  
  the whole board looks like this: xy means: byte x bit y
  
        00 04 02
        01 05 03
  12 13 07 06 37 31 30
  14 15 16 50 36 35 34
  10 11 17 26 27 33 32
        23 25 21
        22 24 20
  
 */

typedef unsigned char UC;
typedef unsigned int UI;

// rotate the whole board i times by 90 to the right
void rotate_board(int i, UC * b)
{
    UC tmp;
    tmp = b[0];
    switch (i) {
      case 1:
	  b[0] = b[1];
	  b[1] = b[2];
	  b[2] = b[3];
	  b[3] = tmp;
	  break;
      case 2:
	  b[0] = b[2];
	  b[2] = tmp;
	  tmp = b[1];
	  b[1] = b[3];
	  b[3] = tmp;
	  break;
      case 3:
	  b[0] = b[3];
	  b[3] = b[2];
	  b[2] = b[1];
	  b[1] = tmp;
	  break;
    }
}

// mirror the whole board on the y-axis 
//
//  0 4 2       2 4 0
//  1 5 3  ==>  3 5 1
//  7 6           6 7
//
// so bit 4,5,6 remain untouched (b[0] & 0x70)
// bit 0,1 change places with  2,3 ((b[0] << 2) & 12) | ((b[0] >> 2) & 3)
// bit 7 is the former highest bit of the next corner
void mirror_board(UC * b)
{
    UC tmp;
    tmp = b[1];
    b[1] = b[3];
    b[3] = tmp;

    tmp = b[0];
    b[0] = (b[0] & 0x70) | ((b[0] << 2) & 12) | ((b[0] >> 2) & 3) | (b[1] & 0x80);
    b[1] = (b[1] & 0x70) | ((b[1] << 2) & 12) | ((b[1] >> 2) & 3) | (b[2] & 0x80);
    b[2] = (b[2] & 0x70) | ((b[2] << 2) & 12) | ((b[2] >> 2) & 3) | (b[3] & 0x80);
    b[3] = (b[3] & 0x70) | ((b[3] << 2) & 12) | ((b[3] >> 2) & 3) | (tmp & 0x80);
}

// sort 4 Bytes by writing a sorted index
// the first value of the index_array is the index of the biggest bytes
// the second value the index of the second biggest, ...
// e.g.: 23, 12, 4, 78
// would get the indexes 3,0,1,2 because the last value with index 3 is the biggest

// return the sorted array, in this example 78, 23, 12, 4
UI sort_board(UC * board, UC * index)
{
    if (board[0] > board[1] && board[0] > board[2] && board[0] > board[3]) {
	index[0] = 0;
	if (board[1] > board[2] && board[1] > board[3]) {
	    index[1] = 1;
	    if (board[2] > board[3]) {
		index[2] = 2;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 2;
	    }			// 0123, 0132

	} else if (board[2] > board[3]) {
	    index[1] = 2;
	    if (board[1] > board[3]) {
		index[2] = 1;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 1;
	    }			// 0213, 0231

	} else {
	    index[1] = 3;
	    if (board[1] > board[2]) {
		index[2] = 1;
		index[3] = 2;
	    } else {
		index[2] = 2;
		index[3] = 1;
	    }			// 0312, 0321

	}
    } else if (board[1] > board[2] && board[1] > board[3]) {
	index[0] = 1;
	if (board[0] > board[2] && board[0] > board[3]) {
	    index[1] = 0;
	    if (board[2] > board[3]) {
		index[2] = 2;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 2;
	    }			// 1023, 1032

	} else if (board[2] > board[3]) {
	    index[1] = 2;
	    if (board[0] > board[3]) {
		index[2] = 0;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 0;
	    }			// 1203, 1230

	} else {
	    index[1] = 3;
	    if (board[0] > board[2]) {
		index[2] = 0;
		index[3] = 2;
	    } else {
		index[2] = 2;
		index[3] = 0;
	    }			// 1302, 1320

	}
    } else if (board[2] > board[3]) {
	index[0] = 2;
	if (board[0] > board[1] && board[0] > board[3]) {
	    index[1] = 0;
	    if (board[1] > board[3]) {
		index[2] = 1;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 1;
	    }			// 2013, 2031

	} else if (board[1] > board[3]) {
	    index[1] = 1;
	    if (board[0] > board[3]) {
		index[2] = 0;
		index[3] = 3;
	    } else {
		index[2] = 3;
		index[3] = 0;
	    }			// 2103, 2130

	} else {
	    index[1] = 3;
	    if (board[0] > board[1]) {
		index[2] = 0;
		index[3] = 1;
	    } else {
		index[2] = 1;
		index[3] = 0;
	    }			// 2301, 2310

	}
    } else {
	index[0] = 3;
	if (board[0] > board[1] && board[0] > board[2]) {
	    index[1] = 0;
	    if (board[1] > board[2]) {
		index[2] = 1;
		index[3] = 2;
	    } else {
		index[2] = 2;
		index[3] = 1;
	    }			// 3012, 3021

	} else if (board[1] > board[2]) {
	    index[1] = 1;
	    if (board[0] > board[2]) {
		index[2] = 0;
		index[3] = 2;
	    } else {
		index[2] = 2;
		index[3] = 0;
	    }			// 3102, 3120

	} else {
	    index[1] = 2;
	    if (board[0] > board[1]) {
		index[2] = 0;
		index[3] = 1;
	    } else {
		index[2] = 1;
		index[3] = 0;
	    }			// 3201, 3210

	}
    }
    return (((((board[index[0]] << 8) | board[index[1]]) << 8) | board[index[2]]) << 8) | board[index[3]];
}

// norm the board so that the biggest corner is on the top of the board
// i.e. rotate and mirror it so that the biggest number is upside
// if the 3 biggest numbers are equal, put the smallest number downwards
// if the 2 biggest numbers are equal rotate this position to the top which
// has a bigger left neighbor
void normalize_board(UC * board)
{
    UC mirrored_board_sorted_index[4];
    UC normal_board_sorted_index[4];
    UC *best_index;
    UI max_normal_board, max_mirrored_board;
    UC normal_board[4];
    memcpy(normal_board, board, 4);

// first sort the corners of the board, biggest first
    max_normal_board = sort_board(board, normal_board_sorted_index);

    mirror_board(board);
// then sort the corners of the mirrored board
    max_mirrored_board = sort_board(board, mirrored_board_sorted_index);

// then decide whether to take the mirrored board or the normal one
    // which board has greater values? the mirrored or the normal one
    if (max_mirrored_board > max_normal_board)
	best_index = mirrored_board_sorted_index;
    else {
	best_index = normal_board_sorted_index;
	memcpy(board, normal_board, 4);
    }

// now we have the bigger board in the array board, we only need to rotate the board to the maximum corners up
    // do we have 3 maximum corners?
    if (board[best_index[0]] == board[best_index[1]] && board[best_index[1]] == board[best_index[2]])
	rotate_board((best_index[3] + 2) % 4, board);	// the least value down

    else
// do we have two corners with maximal value?
    if (board[best_index[0]] == board[best_index[1]]) {
	// if the left neighbor of best_index[0] is bigger
	if (board[(best_index[0] + 1) % 4] > board[(best_index[1] + 1) % 4])
	    // rotate best_index[0] to the top otherwise best_index[1] 
	    rotate_board(best_index[0], board);
	else
	    rotate_board(best_index[1], board);
    } else
// ok, only one corner with maximal value
	rotate_board(best_index[0], board);
}
